/********************************************************************************
*                                                                               *
*                           R e g i s t r y   C l a s s                         *
*                                                                               *
*********************************************************************************
* Copyright (C) 1998 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXRegistry.h,v 1.8 2000/02/21 20:47:31 jeroen Exp $                      *
********************************************************************************/
#ifndef FXREGISTRY_H
#define FXREGISTRY_H


/**
* The registry maintains a database of persistent settings for an application,
* or suite of applications.
*/
class FXAPI FXRegistry {
  friend class FXApp;
protected:
  FXSectionDict  sections;        // Dictionary of sections
  const FXchar  *applicationkey;  // Application key
  const FXchar  *vendorkey;       // Vendor key
  FXbool         modified;        // Settings changed
  FXbool	 ascii;		  // ASCII file-based registry
protected:
  FXbool readFromDir(const FXchar* dirname,FXbool mark);
#ifdef WIN32
  FXbool readFromRegistry(void* hRootKey,FXbool mark);
  FXbool readRegistrySection(void* hAppKey,FXStringDict*& group,const char *groupname,FXbool mark);
  FXbool writeToRegistry(void* hRootKey);
#endif
  FXbool parseFile(const FXchar* filename,FXbool mark);
  FXbool parseBuffer(FXStringDict*& group,const FXchar* buffer,const FXchar *filename,FXint lineno,FXbool mark);
  FXbool parseValue(FXchar* value,const FXchar* buffer,const FXchar *filename,FXint lineno);
  FXbool unparseValue(FXchar* buffer,const FXchar* value);
  FXbool unparseFile(const FXchar* filename);
protected:
  FXRegistry(){}
private:
  FXRegistry(const FXRegistry&);
  FXRegistry &operator=(const FXRegistry&);
public:
  
  /**
  * Construct registry object; akey and vkey must be string constants.
  * Regular applications SHOULD set a vendor key!
  */
  FXRegistry(const FXchar* akey,const FXchar* vkey=NULL);

  /// Read registry
  FXbool read();
  
  /// Write registry
  FXbool write();
  
  /// Read a string registry entry; if no value is found, the default value def is returned
  const FXchar *readStringEntry(const FXchar *section,const FXchar *key,const FXchar *def=NULL);
  
  /// Read a integer registry entry; if no value is found, the default value def is returned
  FXint readIntEntry(const FXchar *section,const FXchar *key,FXint def=0);
  
  /// Read a unsigned integer registry entry; if no value is found, the default value def is returned
  FXuint readUnsignedEntry(const FXchar *section,const FXchar *key,FXuint def=0);
  
  /// Read a double-precision floating point registry entry; if no value is found, the default value def is returned
  FXdouble readRealEntry(const FXchar *section,const FXchar *key,FXdouble def=0.0);
  
  /// Read a color value registry entry; if no value is found, the default value def is returned
  FXColor readColorEntry(const FXchar *section,const FXchar *key,FXColor def=0);
 
  
  /// Write a string registry entry
  FXbool writeStringEntry(const FXchar *section,const FXchar *key,const FXchar *val);
  
  /// Write a integer registry entry
  FXbool writeIntEntry(const FXchar *section,const FXchar *key,FXint val);
  
  /// Write a unsigned integer registry entry
  FXbool writeUnsignedEntry(const FXchar *section,const FXchar *key,FXuint val);
  
  /// Write a double-precision floating point registry entry
  FXbool writeRealEntry(const FXchar *section,const FXchar *key,FXdouble val);
  
  /// Write a color value entry
  FXbool writeColorEntry(const FXchar *section,const FXchar *key,FXColor val);
  
  /// Delete a registry entry
  FXbool deleteEntry(const FXchar *section,const FXchar *key);
  
  /// Delete section
  FXbool deleteSection(const FXchar *section);
  
  /// Mark as changed
  void setModified(FXbool mdfy=TRUE){ modified=mdfy; }

  /// Is it modified
  FXbool isModified() const { return modified; }

  /**
  * Set ASCII mode; under MS-Windows, this will switch the system to a
  * file-based registry system, instead of using the System Registry API.
  */
  void setAsciiMode(FXbool asciiMode){ ascii=asciiMode; }

  /// Get ASCII mode
  FXbool getAsciiMode() const { return ascii; }

  /// Destructor
  virtual ~FXRegistry();
  };
  
  
#endif
