/********************************************************************************
*                                                                               *
*                  S t r i n g   D i c t i o n a r y    C l a s s               *
*                                                                               *
*********************************************************************************
* Copyright (C) 1998 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXDict.h,v 1.6 1999/12/18 05:35:31 jeroen Exp $                          *
********************************************************************************/
#ifndef FXDICT_H
#define FXDICT_H



/**
* The dictionary class maintains a fast-access hash table of entities
* indexed by a character string.  
* It is typically used to map strings to pointers; however, overloading
* the createData() and deleteData() members allows any type of data to
* be indexed by strings.
*/
class FXAPI FXDict : public FXObject {
  FXDECLARE(FXDict)
protected:
  struct FXDictEntry {
    FXchar *key;              // Key string
    void   *data;             // Data
    FXint   hash;             // Hash value of key
    FXbool  mark;             // Entry is marked
    };
protected:
  FXDictEntry  *dict;         // Dictionary
  FXint        total;         // Dictionary size
  FXint        number;        // Number of entries
protected:
  
  /**
  * Overload this function in a derived class to return the
  * data pointer given an input pointer; the default implementation
  * just returns the input pointer.
  */
  virtual void *createData(const void*);

  /**
  * Overload this function in a derived class to delete the pointer
  * previously returned by createData(); the default implementation
  * does nothing.
  */
  virtual void deleteData(void*);
private:
  FXDict(const FXDict&);
  FXDict &operator=(const FXDict&);
public:
  
  /**
  * Construct an empty dictionary.
  */
  FXDict();

  /**
  * Return the size of the table, including the empty slots.
  */
  FXint size() const { return total; }

  /** 
  * Resize the table to the given size.
  */
  void size(FXint m);
  
  /**
  * Return the total number of entries in the table.
  */
  FXint no() const { return number; }

  /**
  * Insert a new entry into the table given key and mark.
  * If there is already an entry with that key, leave it unchanged,
  * otherwise insert the new entry.
  */
  void* insert(const FXchar* ky,const void* ptr,FXbool mrk=FALSE);
  
  /**
  * Replace data at key, if the entry's mark is less than
  * or equal to the given mark.  If there was no existing entry,
  * a new entry is inserted with the given mark.
  */
  void* replace(const FXchar* ky,const void* ptr,FXbool mrk=FALSE);
  
  /**
  * Remove data given key.
  */
  void* remove(const FXchar* ky);
  
  /**
  * Find data pointer given key.
  */
  void* find(const FXchar* ky);
  
  /**
  * Return key at position pos.
  */
  const FXchar* key(FXuint pos) const { return dict[pos].key; }

  /**
  * return data pointer at position pos.
  */
  void* data(FXuint pos) const { return dict[pos].data; }

  /**
  * Return mark flag of entry at position pos.
  */
  FXbool mark(FXuint pos) const { return dict[pos].mark; }

  /**
  * Return position of first filled slot, or >= total
  */
  FXint first() const;

  /**
  * Return position of last filled slot or -1
  */
  FXint last() const;
  
  
  /**
  * Return position of next filled slot in hash table
  * or a value greater than or equal to total if no filled 
  * slot was found
  */
  FXint next(FXint pos) const;  
  
  /**
  * Return position of previous filled slot in hash table
  * or a -1 if no filled slot was found
  */
  FXint prev(FXint pos) const;  
  
  /// Clear all entries
  void clear();
  
  /// Save to stream
  virtual void save(FXStream& store) const;
  
  /// Load from stream
  virtual void load(FXStream& store);

  /// Destructor
  virtual ~FXDict();
  };


/**
* String dictionary maps a character string to a character string.
* The inserted strings are copied when they're inserted.
*/
class FXAPI FXStringDict : public FXDict {
  FXDECLARE(FXStringDict)
protected:
  virtual void *createData(const void*);
  virtual void deleteData(void*);
private:
  FXStringDict(const FXStringDict&);
  FXStringDict &operator=(const FXStringDict&);
public:
  /// Construct a string dictionary
  FXStringDict();

  /// Insert a new string indexed by key, with given mark flag
  FXchar* insert(const FXchar* ky,const FXchar* str,FXbool mrk=FALSE){ return (FXchar*)FXDict::insert(ky,str,mrk); }

  /// Replace or insert a new string indexed by key, unless given mark is lower that the existing mark
  FXchar* replace(const FXchar* ky,const FXchar* str,FXbool mrk=FALSE){ return (FXchar*)FXDict::replace(ky,str,mrk); }

  /// Remove entry indexed by key
  FXchar* remove(const FXchar* ky){ return (FXchar*)FXDict::remove(ky); }

  /// Return the entry indexed by key, or return NULL if the key does not exist
  FXchar* find(const FXchar* ky){ return (FXchar*)FXDict::find(ky); }

  /// Return the string at position pos
  FXchar* data(FXuint pos) const { return (FXchar*)dict[pos].data; }

  /// Save to stream
  virtual void save(FXStream& store) const;
  
  /// Load from stream
  virtual void load(FXStream& store);

  /// Destructor
  virtual ~FXStringDict();
  };


/**
* Section dictionary is a dictionary of string dictionaries.
*/
class FXAPI FXSectionDict : public FXDict {
  FXDECLARE(FXSectionDict)
protected:
  virtual void *createData(const void*);
  virtual void deleteData(void*);
private:
  FXSectionDict(const FXSectionDict&);
  FXSectionDict &operator=(const FXSectionDict&);
public:

  /// Construct new empty section dictionary
  FXSectionDict();

  /** 
  * Insert new section indexed by key, or return already existing
  * section if the entry with the given key already was inserted previously.
  * Return the pointer to the string dictionary.
  */
  FXStringDict* insert(const FXchar* ky){ return (FXStringDict*)FXDict::insert(ky,NULL); }

  /**
  * Remove a section dictionary.
  * This deletes all entries in the string dictionary.
  * Returns NULL.
  */
  FXStringDict* remove(const FXchar* ky){ return (FXStringDict*)FXDict::remove(ky); }

  /**
  * Return the string dictionary indexed by the given key.
  * If there is no entry with that key, return NULL.
  */
  FXStringDict* find(const FXchar* ky){ return (FXStringDict*)FXDict::find(ky); }

  /**
  * Return the string dictionary at the given position.
  */
  FXStringDict* data(FXuint pos) const { return (FXStringDict*)dict[pos].data; }

  /// Save to stream
  virtual void save(FXStream& store) const;
  
  /// Load from stream
  virtual void load(FXStream& store);

  /// Destructor
  virtual ~FXSectionDict();
  };

#endif
