/********************************************************************************
*                                                                               *
*                            L i s t   O b j e c t                              *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXOldList.cpp,v 1.6 2000/02/28 17:44:53 jeroen Exp $                     *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "fxkeys.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXObjectList.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXFont.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXLabel.h"
#include "FXButton.h"
#include "FXComposite.h"
#include "FXScrollbar.h"
#include "FXScrollArea.h"
#include "FXOldList.h"


////////////////////  THIS WIDGET HAS BEEN DEPRECATED  //////////////////////////


#define ICON_SPACING             4    // Spacing between icon and label
#define SIDE_SPACING             6    // Left or right spacing between items
#define LINE_SPACING             4    // Line spacing between items

#define SELECT_MASK (OLDLIST_SINGLESELECT|OLDLIST_BROWSESELECT)
#define OLDLIST_MASK   (OLDLIST_SINGLESELECT|OLDLIST_BROWSESELECT|OLDLIST_AUTOSELECT|OLDLIST_WANTSELECTION|OLDLIST_RIGHTJUSTIFIED)



/*******************************************************************************/

// Map
FXDEFMAP(FXOldList) FXOldListMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXOldList::onPaint),
  FXMAPFUNC(SEL_MOTION,0,FXOldList::onMotion),
  FXMAPFUNC(SEL_TIMEOUT,FXWindow::ID_AUTOSCROLL,FXOldList::onAutoScroll),
  FXMAPFUNC(SEL_UNGRABBED,0,FXOldList::onUngrabbed),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXOldList::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXOldList::onLeftBtnRelease),
  FXMAPFUNC(SEL_KEYPRESS,0,FXOldList::onKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,0,FXOldList::onKeyRelease),
  FXMAPFUNC(SEL_ACTIVATE,0,FXOldList::onActivate),
  FXMAPFUNC(SEL_DEACTIVATE,0,FXOldList::onDeactivate),
  FXMAPFUNC(SEL_FOCUSIN,0,FXOldList::onFocusIn),
  FXMAPFUNC(SEL_FOCUSOUT,0,FXOldList::onFocusOut),
  FXMAPFUNC(SEL_SELECTION_LOST,0,FXOldList::onSelectionLost),
  FXMAPFUNC(SEL_SELECTION_GAINED,0,FXOldList::onSelectionGained),
  FXMAPFUNC(SEL_CHANGED,0,FXOldList::onChanged),
  FXMAPFUNC(SEL_CLICKED,0,FXOldList::onClicked),
  FXMAPFUNC(SEL_DOUBLECLICKED,0,FXOldList::onDoubleClicked),
  FXMAPFUNC(SEL_TRIPLECLICKED,0,FXOldList::onTripleClicked),
  FXMAPFUNC(SEL_SELECTED,0,FXOldList::onSelected),
  FXMAPFUNC(SEL_DESELECTED,0,FXOldList::onDeselected),
  };


// Object implementation
FXIMPLEMENT(FXOldList,FXScrollArea,FXOldListMap,ARRAYNUMBER(FXOldListMap))

  
// List
FXOldList::FXOldList(){
  flags|=FLAG_ENABLED;
  firstitem=NULL;
  lastitem=NULL;
  anchoritem=NULL;
  currentitem=NULL;
  font=(FXFont*)-1;
  sortfunc=NULL;
  textColor=0;
  selbackColor=0;
  seltextColor=0;
  totalWidth=0;
  totalHeight=0;
  }

  
// List
FXOldList::FXOldList(FXComposite *p,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h):
  FXScrollArea(p,opts,x,y,w,h){
  flags|=FLAG_ENABLED;
  target=tgt;
  message=sel;
  firstitem=NULL;
  lastitem=NULL;
  anchoritem=NULL;
  currentitem=NULL;
  font=getApp()->getNormalFont();
  sortfunc=NULL;
  textColor=getApp()->foreColor;
  selbackColor=getApp()->selbackColor;
  seltextColor=getApp()->selforeColor;
  totalWidth=0;
  totalHeight=0;
  }


// Create window
void FXOldList::create(){
  FXScrollArea::create();
  if(!textType){textType=getApp()->registerDragType(textTypeName);}
  font->create();
  }


// Detach window
void FXOldList::detach(){
  FXScrollArea::detach();
  textType=0;
  font->detach();
  }


// Propagate size change
void FXOldList::recalc(){ 
  FXScrollArea::recalc();
  flags|=FLAG_RECALC;
  }


// Can have focus
FXbool FXOldList::canFocus() const { return TRUE; }


// Create item
FXOldListItem* FXOldList::createItem(){ return new FXOldListItem; }


// Destroy item
void FXOldList::deleteItem(FXOldListItem* item){ delete item; }


// Get number of items
FXint FXOldList::getNumItems() const {
  register FXOldListItem *item=firstitem;
  register FXint n=0;
  while(item){
    item=item->next;
    n++;
    }
  return n;
  }


// Get number of selected items
FXint FXOldList::getNumSelectedItems() const {
  register FXOldListItem *item=firstitem;
  register FXint n=0;
  while(item){
    if(item->state&LISTITEM_SELECTED) n++;
    item=item->next;
    }
  return n;
  }


// Get list of selected ite,s
FXOldListItem** FXOldList::getSelectedItems() const {
  FXOldListItem *item=firstitem;
  FXOldListItem **itemlist;
  FXint i=0;
  FXMALLOC(&itemlist,FXOldListItem*,getNumSelectedItems()+1);
  while(item){
    if(item->state&LISTITEM_SELECTED) itemlist[i++]=item;
    item=item->next;
    }
  itemlist[i]=NULL;
  return itemlist;
  }
 


// Get item X
FXint FXOldList::getItemX(const FXOldListItem* item) const { 
  if(!item){ fxerror("%s::getItemX: item is NULL.\n",getClassName()); } 
  return 0; 
  }


// Get item Y
FXint FXOldList::getItemY(const FXOldListItem* item) const { 
  if(!item){ fxerror("%s::getItemY: item is NULL.\n",getClassName()); } 
  return item->y; 
  }


// Get item width
FXint FXOldList::getItemWidth(const FXOldListItem* item) const { 
  if(!item){ fxerror("%s::getItemWidth: item is NULL.\n",getClassName()); }
  return content_w;
  }


// Get Item height
FXint FXOldList::getItemHeight(const FXOldListItem* item) const { 
  if(!item){ fxerror("%s::getItemHeight: item is NULL.\n",getClassName()); } 
  return LINE_SPACING+font->getFontHeight(); 
  }


// True if item is selected
FXbool FXOldList::isItemSelected(const FXOldListItem* item) const { 
  if(!item){ fxerror("%s::isItemSelected: item is NULL.\n",getClassName()); } 
  return (item->state&LISTITEM_SELECTED)!=0; 
  }


// True if item is current
FXbool FXOldList::isItemCurrent(const FXOldListItem* item) const { 
  if(!item){ fxerror("%s::isItemCurrent: item is NULL.\n",getClassName()); } 
  return (item->state&LISTITEM_CURRENT)!=0; 
  }


// True if item (partially) visible
FXbool FXOldList::isItemVisible(const FXOldListItem* item) const { 
  if(!item){ fxerror("%s::isItemVisible: item is NULL.\n",getClassName()); } 
  return (0 < (pos_y+getItemY(item)+getItemHeight(item))) && ((pos_y+getItemY(item))) < viewport_h; 
  }


// Make item fully visible
void FXOldList::makeItemVisible(FXOldListItem* item){ 
  register FXint y,h;
  if(!xid) return;
  if(item){
    y=getItemY(item);
    h=getItemHeight(item);
    if((pos_y+y) < 0){
      setPosition(pos_x,-y);
      update();
      }
    else if(viewport_h<=(pos_y+y+h)){
      setPosition(pos_x,viewport_h-y-h);
      update();
      }
    }
  }


// Get index of item
FXint FXOldList::indexOfItem(const FXOldListItem* item) const {
  register FXint index=-1;
  while(item){item=item->prev;index++;}
  return index;
  }


// Get item at index
FXOldListItem* FXOldList::itemAtIndex(FXint index) const {
  register FXOldListItem* item=firstitem;
  if(index<0) return NULL;
  while(index && item){item=item->next;index--;}
  return item;
  }


// Get item at position x,y
FXOldListItem* FXOldList::getItemAt(FXint,FXint y) const {
  register FXint h=LINE_SPACING+font->getFontHeight();
  register FXOldListItem* n=firstitem;
  register FXint yy=0;
  y=y-pos_y;
  while(n){
    if(y<yy) return NULL;
    yy+=h;
    if(y<yy) return n;
    n=n->next;
    }
  return NULL;
  }


// Get next item
FXOldListItem* FXOldList::getNextItem(const FXOldListItem* item) const { 
  if(!item){ fxerror("%s::getNextItem: item is NULL.\n",getClassName()); } 
  return item->next; 
  }


// Get previous item
FXOldListItem* FXOldList::getPrevItem(const FXOldListItem* item) const { 
  if(!item){ fxerror("%s::getPrevItem: item is NULL.\n",getClassName()); } 
  return item->prev; 
  }


// Sort items
void FXOldList::sort(FXOldListItem*& f1,FXOldListItem*& t1,FXOldListItem*& f2,FXOldListItem*& t2,int n){
  FXOldListItem *ff1,*tt1,*ff2,*tt2,*q;
  FXint m;
  if(f2==NULL){ 
    f1=NULL; 
    t1=NULL; 
    return; 
    }
  if(n>1){
    m=n/2;
    n=n-m;
    sort(ff1,tt1,f2,t2,n);  // 1 or more
    sort(ff2,tt2,f2,t2,m);  // 0 or more
    FXASSERT(ff1);
    if(ff2 && sortfunc(ff1,ff2)){
      f1=ff2;
      ff2->prev=NULL;
      ff2=ff2->next;
      }
    else{
      f1=ff1;
      ff1->prev=NULL;
      ff1=ff1->next;
      }
    t1=f1;
    t1->next=NULL;
    while(ff1 || ff2){
      if(ff1==NULL){ t1->next=ff2; ff2->prev=t1; t1=tt2; break; }
      if(ff2==NULL){ t1->next=ff1; ff1->prev=t1; t1=tt1; break; }
      if(sortfunc(ff1,ff2)){
        t1->next=ff2;
        ff2->prev=t1;
        t1=ff2;
        ff2=ff2->next;
        }
      else{
        t1->next=ff1;
        ff1->prev=t1;
        t1=ff1;
        ff1=ff1->next;
        }
      t1->next=NULL;
      }
    return;
    }
  FXASSERT(f2);
  f1=f2;
  t1=f2;
  f2=f2->next;
  while(f2){
    f2->prev=NULL;
    if(sortfunc(f2,t1)){
      t1->next=f2;
      f2->prev=t1;
      t1=f2;
      f2=f2->next;
      continue;
      }
    if(sortfunc(f1,f2)){
      q=f2;
      f2=f2->next;
      q->next=f1;
      f1->prev=q;
      f1=q;
      continue;
      }
    break;
    }
  FXASSERT(f1);
  FXASSERT(t1);
  f1->prev=NULL;
  t1->next=NULL;
  }


// Sort the items based on the sort function
void FXOldList::sortItems(){
  if(sortfunc){
    FXOldListItem* f=firstitem;
    FXOldListItem* l=lastitem;
    sort(firstitem,lastitem,f,l,getNumItems());
    recalc();
    }
  }


// Repaint
void FXOldList::updateItem(FXOldListItem* item){
  if(xid){update(pos_x+getItemX(item),pos_y+getItemY(item),getItemWidth(item),getItemHeight(item));}
  }


// All turned on
FXbool FXOldList::selectItemRange(FXOldListItem* beg,FXOldListItem* end){
  register FXbool selected=FALSE;
  if(beg && end){
    register FXOldListItem *item=beg;
    do{
      selected|=selectItem(item);
      if(item==end) break;
      item=item->next;
      }
    while(item);
    }
  return selected;
  }


// All turned off
FXbool FXOldList::deselectItemRange(FXOldListItem* beg,FXOldListItem* end){
  register FXbool deselected=FALSE;
  if(beg && end){
    register FXOldListItem *item=beg;
    do{
      deselected|=deselectItem(item);
      if(item==end) break;
      item=item->next;
      }
    while(item);
    }
  return deselected;
  }


// Toggle items
FXbool FXOldList::toggleItemRange(FXOldListItem* beg,FXOldListItem* end){
  register FXbool toggled=FALSE;
  if(beg && end){
    register FXOldListItem *item=beg;
    do{
      toggled|=toggleItem(item);
      if(item==end) break;
      item=item->next;
      }
    while(item);
    }
  return toggled;
  }


// Restore items to previous state
FXbool FXOldList::restoreItemRange(FXOldListItem* beg,FXOldListItem* end){
  register FXbool restored=FALSE;
  if(beg && end){
    register FXOldListItem *item=beg;
    do{
      restored|=restoreItem(item);
      if(item==end) break;
      item=item->next;
      }
    while(item);
    }
  return restored;
  }


// Select one item
FXbool FXOldList::selectItem(FXOldListItem* item){
  if(item){
    if(!(item->state&LISTITEM_SELECTED)){
      item->state|=LISTITEM_SELECTED;
      updateItem(item);
      return TRUE;
      }
    }
  return FALSE;
  }


// Deselect one item
FXbool FXOldList::deselectItem(FXOldListItem* item){
  if(item){
    if(item->state&LISTITEM_SELECTED){
      item->state&=~LISTITEM_SELECTED;
      updateItem(item);
      return TRUE;
      }
    }
  return FALSE;
  }


// toggle one item
FXbool FXOldList::toggleItem(FXOldListItem* item){
  if(item){
    item->state^=LISTITEM_SELECTED;
    updateItem(item);
    return TRUE;
    }
  return FALSE;
  }


// Restore items to previous state
FXbool FXOldList::restoreItem(FXOldListItem* item){
  if(item){
    if((item->state&LISTITEM_MARK) && !(item->state&LISTITEM_SELECTED)){
      item->state|=LISTITEM_SELECTED;
      updateItem(item);
      return TRUE;
      }
    if(!(item->state&LISTITEM_MARK) && (item->state&LISTITEM_SELECTED)){
      item->state&=~LISTITEM_SELECTED;
      updateItem(item);
      return TRUE;
      }
    }
  return FALSE;
  }


// Mark items
void FXOldList::markItems(){
  register FXOldListItem* item;
  for(item=firstitem; item; item=item->next){
    (item->state&LISTITEM_SELECTED)?item->state|=LISTITEM_MARK:item->state&=~LISTITEM_MARK;
    }
  }


// Recompute interior 
void FXOldList::recompute(){
  register FXOldListItem* item;
  register FXint w,h;
  totalHeight=0;
  totalWidth=0;
  h=LINE_SPACING+font->getFontHeight();
  for(item=firstitem; item; item=item->next){
    item->y=totalHeight;
    w=SIDE_SPACING;
    if(item->label.text()){w+=font->getTextWidth(item->label.text(),item->label.length());}
    if(w>totalWidth) totalWidth=w;
    totalHeight=totalHeight+h;
    }
  flags&=~FLAG_RECALC;
  }


// Determine content width of list
FXint FXOldList::getContentWidth(){ 
  if(flags&FLAG_RECALC) recompute();
  return totalWidth;
  }


// Determine content height of list
FXint FXOldList::getContentHeight(){
  if(flags&FLAG_RECALC) recompute();
  return totalHeight; 
  }


// Recalculate layout determines item locations and sizes
void FXOldList::layout(){
  
  // Calculate contents 
  FXScrollArea::layout();
  
  // Determine line size for scroll bars
  vertical->setLine(LINE_SPACING+font->getFontHeight());
  horizontal->setLine(1);
  
  // Force repaint
  update();
  
  // No more dirty
  flags&=~FLAG_DIRTY;
  }


// Gained focus
long FXOldList::onFocusIn(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onFocusIn(sender,sel,ptr);
  if(currentitem) updateItem(currentitem);
  return 1;
  }


// Lost focus
long FXOldList::onFocusOut(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onFocusOut(sender,sel,ptr);
  if(currentitem) updateItem(currentitem);
  return 1;
  }


// We have the selection
long FXOldList::onSelectionGained(FXObject* sender,FXSelector sel,void* ptr){
  if(FXScrollArea::onSelectionGained(sender,sel,ptr)) return 1;
  //update();
  return 1;
  }


// We lost the selection
long FXOldList::onSelectionLost(FXObject* sender,FXSelector sel,void* ptr){
  if(FXScrollArea::onSelectionLost(sender,sel,ptr)) return 1;
  deselectItemRange(firstitem,lastitem);
  return 1;
  }


// Draw dashed focus rectangle
void FXOldList::drawFocusRectangle(FXDCWindow& dc,FXint x,FXint y,FXint w,FXint h){
//   static const char onoff[]={1,2};
//   dc.setForeground(textColor);
//   dc.setDashes(0,onoff,2);
//   dc.setLineStyle(LINE_ONOFF_DASH);
//   dc.drawLine(x,y,x+w-1,y);
//   dc.drawLine(x,y,x,y+h-1);
//   dc.drawLine(x,y+h-1,x+w-1,y+h-1);
//   dc.drawLine(x+w-1,y,x+w-1,y+h-1);
//   dc.setLineStyle(LINE_SOLID);
  dc.setFillStyle(FILL_STIPPLED);
  dc.setStipple(STIPPLE_GRAY);
  dc.setForeground(textColor);
  dc.drawRectangle(x,y,w-1,h-1);
  dc.setFillStyle(FILL_SOLID);
  dc.setStipple(STIPPLE_NONE);
  }


// Draw item list
long FXOldList::onPaint(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXOldListItem* n=firstitem;
  FXint h=LINE_SPACING+font->getFontHeight();
  FXint x=pos_x;
  FXint y=pos_y;
  FXint eylo=event->rect.y-h;
  FXint eyhi=event->rect.y+event->rect.h;
  FXint xx;
  FXDCWindow dc(this,event);
  dc.setForeground(backColor);
  dc.fillRectangle(event->rect.x,event->rect.y,event->rect.w,event->rect.h);
  dc.setTextFont(font);
  while(n && y<eyhi){
    if(eylo<=y){
      if(n->label.text()){
        FXint len=n->label.length();
        if(n->state&LISTITEM_SELECTED){
          if(hasFocus() && (n->state&LISTITEM_CURRENT)){
            dc.setForeground(selbackColor);
            dc.fillRectangle(x+1,y+1,content_w-2,h-2);
            drawFocusRectangle(dc,x,y,content_w,h);
            }
          else{
            dc.setForeground(selbackColor);
            dc.fillRectangle(x,y,content_w,h);
            }
          dc.setForeground(seltextColor);
          if(options&OLDLIST_RIGHTJUSTIFIED){
            xx=x+content_w-font->getTextWidth(n->label.text(),len);
            dc.drawText(xx-(SIDE_SPACING/2),y+(LINE_SPACING/2)+font->getFontAscent(),n->label.text(),len);
            }
          else{
            dc.drawText(x+(SIDE_SPACING/2),y+(LINE_SPACING/2)+font->getFontAscent(),n->label.text(),len);
            }
          }
        else{
          if(hasFocus() && (n->state&LISTITEM_CURRENT)){
            drawFocusRectangle(dc,x,y,content_w,h);
            }
          dc.setForeground(textColor);
          if(options&OLDLIST_RIGHTJUSTIFIED){
            xx=x+content_w-font->getTextWidth(n->label.text(),len);
            dc.drawText(xx-(SIDE_SPACING/2),y+(LINE_SPACING/2)+font->getFontAscent(),n->label.text(),len);
            }
          else{
            dc.drawText(x+(SIDE_SPACING/2),y+(LINE_SPACING/2)+font->getFontAscent(),n->label.text(),len);
            }
          }
        }
      }
    y+=h;
    n=n->next;
    }
  return 1;
  }


// Test if a is before b
FXbool FXOldList::before(FXOldListItem* a,FXOldListItem* b){
  while(a && a->next!=b) a=a->next;
  return a!=NULL;
  }


// Extend selection
FXbool FXOldList::extendSelection(FXOldListItem* item){
  FXbool changes=FALSE;
  if(item && anchoritem){
      
    // Browse select mode
    if((options&OLDLIST_BROWSESELECT) && !(options&OLDLIST_SINGLESELECT)){
      changes|=deselectItemRange(firstitem,item->prev);
      changes|=deselectItemRange(item->next,lastitem);
      changes|=selectItem(item);
      }

    // Extended select mode
    else if(!(options&(OLDLIST_SINGLESELECT|OLDLIST_BROWSESELECT))){
      if(item==anchoritem){
        changes|=restoreItemRange(firstitem,anchoritem->prev);
        changes|=restoreItemRange(anchoritem->next,lastitem);
        }
      else if(before(item,anchoritem)){
        changes|=restoreItemRange(firstitem,item->prev);
        changes|=restoreItemRange(anchoritem->next,lastitem);
        if(anchoritem->state&LISTITEM_SELECTED){
          changes|=selectItemRange(item,anchoritem->prev);
          }
        else{
          changes|=deselectItemRange(item,anchoritem->prev);
          }
        }
      else{
        changes|=restoreItemRange(firstitem,anchoritem->prev);
        changes|=restoreItemRange(item->next,lastitem);
        if(anchoritem->state&LISTITEM_SELECTED){
          changes|=selectItemRange(anchoritem->next,item);
          }
        else{
          changes|=deselectItemRange(anchoritem->next,item);
          }
        }
      }
    }
  return changes;
  }


// Key Press
long FXOldList::onKeyPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXOldListItem *item;
  flags&=~FLAG_TIP;
  if(!isEnabled()) return 0;
  if(target && target->handle(this,MKUINT(message,SEL_KEYPRESS),ptr)) return 1;
  switch(event->code){
    case KEY_Up:
    case KEY_Down:
    case KEY_Home:
    case KEY_End:
      item=currentitem;
      switch(event->code){
        case KEY_Up:   
          if(item && item->prev) item=item->prev; 
          if(!item) item=lastitem; 
          break;
        case KEY_Down: 
          if(item && item->next) item=item->next; 
          if(!item) item=firstitem; 
          break;
        case KEY_Home: 
          item=firstitem; 
          break;
        case KEY_End:  
          item=lastitem; 
          break;
        }
      if(item && item!=currentitem){
        handle(this,MKUINT(0,SEL_CHANGED),(void*)item);
        if((options&OLDLIST_BROWSESELECT) && !(options&OLDLIST_SINGLESELECT)){
          handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
          }
        flags&=~FLAG_UPDATE;
        flags|=FLAG_KEY;
        }
      return 1;
    case KEY_space:
      handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
      flags&=~FLAG_UPDATE;
      flags|=FLAG_KEY;
      return 1;
    case KEY_KP_Enter:
    case KEY_Return:
      handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
      flags&=~FLAG_UPDATE;
      flags|=FLAG_KEY;
      return 1;
    case KEY_Page_Up:
      setPosition(pos_x,pos_y+verticalScrollbar()->getPage());
      return 1;
    case KEY_Page_Down:
      setPosition(pos_x,pos_y-verticalScrollbar()->getPage());
      return 1;
    case KEY_Control_L:
    case KEY_Control_R:
    case KEY_Shift_L:
    case KEY_Shift_R:
      flags|=FLAG_KEY;
      return 1;
    }
  return 0;
  }


// Key Release 
long FXOldList::onKeyRelease(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  if(!isEnabled()) return 0;
  flags|=FLAG_UPDATE;
  if(target && target->handle(this,MKUINT(message,SEL_KEYRELEASE),ptr)) return 1;
  switch(event->code){
    case KEY_Up:
    case KEY_Down:
    case KEY_Home:
    case KEY_End:
      if(flags&FLAG_KEY){
        if((options&OLDLIST_BROWSESELECT) && !(options&OLDLIST_SINGLESELECT)){
          handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
          handle(this,MKUINT(0,SEL_CLICKED),(void*)currentitem);
          }
        flags&=~FLAG_KEY;
        }
      return 1;
    case KEY_space:
      handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
      handle(this,MKUINT(0,SEL_CLICKED),(void*)currentitem);
      flags&=~FLAG_KEY;
      return 1;
    case KEY_KP_Enter:
    case KEY_Return:
      handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
      handle(this,MKUINT(0,SEL_DOUBLECLICKED),(void*)currentitem);
      flags&=~FLAG_KEY;
      return 1;
    case KEY_Page_Up:
    case KEY_Page_Down:
      return 1;
    case KEY_Shift_L:
    case KEY_Shift_R:
    case KEY_Control_L:
    case KEY_Control_R:
      flags&=~FLAG_KEY;
      return 1;
    }
  return 0;
  }


// Automatic scroll
long FXOldList::onAutoScroll(FXObject* sender,FXSelector sel,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXOldListItem *item;
  FXint xx,yy;
    
  // Scroll the window
  FXScrollArea::onAutoScroll(sender,sel,ptr);
  
  // Validated position
  xx=event->win_x; if(xx<0) xx=0; else if(xx>=viewport_w) xx=viewport_w-1;
  yy=event->win_y; if(yy<0) yy=0; else if(yy>=viewport_h) yy=viewport_h-1;
  
  // Find item
  item=getItemAt(xx,yy);
  
  // Got item and different from last time
  if(item && item!=currentitem){
    
    // Make it the current item
    handle(this,MKUINT(0,SEL_CHANGED),(void*)item);
 
    // In autoselect, pretend we pressed it
    if(options&OLDLIST_AUTOSELECT){
      handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
      }
 
    // Extend the selection to it
    else{
      extendSelection(item);
      }
    }
  return 1;
  }


// Mouse moved
long FXOldList::onMotion(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXOldListItem *item;
  if((flags&FLAG_PRESSED) || (options&OLDLIST_AUTOSELECT)){
    
    // Start auto scrolling?
    if(startAutoScroll(event->win_x,event->win_y,FALSE)) return 1;
    
    // Find item
    item=getItemAt(event->win_x,event->win_y);
    
    // Got an item different from before
    if(item && item!=currentitem){
      
      // Make it the current item
      handle(this,MKUINT(0,SEL_CHANGED),(void*)item);
      
      // In autoselect, pretend we pressed it
      if(options&OLDLIST_AUTOSELECT){
        handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
        }
      
      // Extend the selection to it
      else{
        extendSelection(item);
        }
      return 1;
      }
    }
  return 0;
  }
 

// Pressed button
long FXOldList::onLeftBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXOldListItem *item;
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONPRESS),ptr)) return 1;
    if(options&OLDLIST_AUTOSELECT) return 1;
    if(event->click_count==1){
      item=getItemAt(event->win_x,event->win_y);
      if(item!=currentitem){
        handle(this,MKUINT(0,SEL_CHANGED),(void*)item);
        }
      handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
      }
    flags|=FLAG_PRESSED;
    flags&=~FLAG_UPDATE;
    return 1;
    }
  return 0;
  }


// Released button
long FXOldList::onLeftBtnRelease(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  if(isEnabled()){
    ungrab();
    flags&=~FLAG_PRESSED;
    flags|=FLAG_UPDATE;
    stopAutoScroll();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONRELEASE),ptr)) return 1;
    if(event->click_count==1){
      handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
      handle(this,MKUINT(0,SEL_CLICKED),(void*)currentitem);
      }
    else if(event->click_count==2){
      handle(this,MKUINT(0,SEL_DOUBLECLICKED),(void*)currentitem);
      }
    else if(event->click_count==3){
      handle(this,MKUINT(0,SEL_TRIPLECLICKED),(void*)currentitem);
      }
    return 1;
    }
  return 0;
  }


// The widget lost the grab for some reason
long FXOldList::onUngrabbed(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onUngrabbed(sender,sel,ptr);
  flags&=~FLAG_PRESSED;
  flags&=~FLAG_CHANGED;
  flags|=FLAG_UPDATE;
  stopAutoScroll();
  return 1;
  }


// Current item changed
long FXOldList::onChanged(FXObject*,FXSelector,void* ptr){
  makeItemVisible((FXOldListItem*)ptr);
  setCurrentItem((FXOldListItem*)ptr);
  if(target) target->handle(this,MKUINT(message,SEL_CHANGED),ptr);
  return 1;
  }


// Selected items
long FXOldList::onSelected(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_SELECTED),ptr);
  }


// Deselected items
long FXOldList::onDeselected(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_DESELECTED),ptr);
  }


// Clicked in list
long FXOldList::onClicked(FXObject*,FXSelector,void* ptr){
  FXOldListItem **selectedlist,**deselectedlist,*item;
  FXint numselected,numdeselected,i,j;
  
  // Notify target of the click
  if(target){
    
    // Clicked message indicates a click anywhere in the widget
    if(target->handle(this,MKUINT(message,SEL_CLICKED),ptr)) return 1;
    
    // Command message indicates click on an item
    if(ptr && target->handle(this,MKUINT(message,SEL_COMMAND),ptr)) return 1;
    }
    
  // Find out number of items whose selection state changed
  for(item=firstitem,numselected=numdeselected=0; item; item=item->next){
    if((item->state&LISTITEM_SELECTED) && !(item->state&LISTITEM_HISTORY)) numselected++;
    if(!(item->state&LISTITEM_SELECTED) && (item->state&LISTITEM_HISTORY)) numdeselected++;
    }
 
  // Make some room
  FXMALLOC(&selectedlist,FXOldListItem*,numselected+1);
  FXMALLOC(&deselectedlist,FXOldListItem*,numdeselected+1);
 
  // Add items to the proper lists
  for(item=firstitem,i=j=0; item; item=item->next){
    if((item->state&LISTITEM_SELECTED) && !(item->state&LISTITEM_HISTORY)) selectedlist[i++]=item;
    if(!(item->state&LISTITEM_SELECTED) && (item->state&LISTITEM_HISTORY)) deselectedlist[j++]=item;
    }
 
  // Close off lists
  selectedlist[i]=NULL;
  deselectedlist[j]=NULL;
 
  // Tell the target about the newly selected items
  handle(this,MKUINT(0,SEL_SELECTED),selectedlist);
 
  // Tell the target about the newly deselected items
  handle(this,MKUINT(0,SEL_DESELECTED),deselectedlist);
 
  // Free the lists
  FXFREE(&selectedlist);
  FXFREE(&deselectedlist);
  return 1;
  }


// Double clicked in list; ptr may or may not point to an item
long FXOldList::onDoubleClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_DOUBLECLICKED),ptr);
  }


// Triple clicked in list; ptr may or may not point to an item
long FXOldList::onTripleClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_TRIPLECLICKED),ptr);
  }


// Button or Key activate
long FXOldList::onActivate(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  register FXOldListItem *item;
  FXDragType types[2];
  
  // Remember previous state
  for(item=firstitem; item; item=item->next){
    if(item->state&LISTITEM_SELECTED)
      item->state|=LISTITEM_HISTORY;
    else
      item->state&=~LISTITEM_HISTORY;
    }
  
  // Have currentitem
  if(currentitem){
    
    // Multiple selection mode
    if((options&OLDLIST_SINGLESELECT)&&(options&OLDLIST_BROWSESELECT)){
      if(isItemSelected(currentitem)){
        deselectItem(currentitem);
        }
      else{
        selectItem(currentitem);
        }
      markItems();
      setAnchorItem(currentitem);
      }

    // Browse select mode
    else if(options&OLDLIST_BROWSESELECT){
      deselectItemRange(firstitem,currentitem->prev);
      deselectItemRange(currentitem->next,lastitem);
      selectItem(currentitem);
      setAnchorItem(currentitem);
      }

    // Single selection mode
    else if(options&OLDLIST_SINGLESELECT){
      if(isItemSelected(currentitem)){
        deselectItemRange(firstitem,lastitem);
        }
      else{
        deselectItemRange(firstitem,currentitem->prev);
        deselectItemRange(currentitem->next,lastitem);
        selectItem(currentitem);
        }
      setAnchorItem(currentitem);
      }

    // Add selection mode
    else if(event->state&SHIFTMASK){
      if(anchoritem){
        selectItem(anchoritem);
        markItems();
        extendSelection(currentitem);
        }
      else{
        setAnchorItem(currentitem);
        selectItem(currentitem);
        markItems();
        }
      }
    
    // Toggle selection mode
    else if(event->state&CONTROLMASK){
      setAnchorItem(currentitem);
      toggleItem(currentitem);
      markItems();
      }
    
    // Extended selection mode
    else{
      deselectItemRange(firstitem,currentitem->prev);
      deselectItemRange(currentitem->next,lastitem);
      selectItem(currentitem);
      setAnchorItem(currentitem);
      markItems();
      }
    }
  
  // No currentitem & not multiple or browse select mode
  else if(!(options&OLDLIST_BROWSESELECT)){
    
    // If not add or toggle mode, turn them off
    if(!(event->state&(SHIFTMASK|CONTROLMASK))){
      deselectItemRange(firstitem,lastitem);
      }
    setAnchorItem(currentitem);
    }
  
  // If we want selection, see wether to acquire or release it
  if(options&OLDLIST_WANTSELECTION){
    if(getNumSelectedItems()){
      types[0]=stringType;
      types[1]=textType;
      acquireSelection(types,2);
      }
    else{
      releaseSelection();
      }
    }
  return 1;
  }
  

// Button or Key deactivate
long FXOldList::onDeactivate(FXObject*,FXSelector,void*){
  setAnchorItem(currentitem);
  return 1;
  }


// Add item as first
FXOldListItem* FXOldList::addItemFirst(const FXString& text,void* ptr){
  FXOldListItem *item=createItem();
  item->prev=NULL;
  item->next=firstitem;
  if(item->next) item->next->prev=item; else lastitem=item;
  firstitem=item;
  item->label=text;
  item->state=0;
  item->data=ptr;
  item->y=0;
  recalc();
  return item;
  }


// Add item as last
FXOldListItem* FXOldList::addItemLast(const FXString& text,void* ptr){
  FXOldListItem *item=createItem();
  item->prev=lastitem;
  item->next=NULL;
  if(item->prev) item->prev->next=item; else firstitem=item;
  lastitem=item;
  item->label=text;
  item->state=0;
  item->data=ptr;
  item->y=0;
  recalc();
  return item;
  }


// Add item after other
FXOldListItem* FXOldList::addItemAfter(FXOldListItem* other,const FXString& text,void* ptr){
  if(other==NULL){ fxerror("%s::addItemAfter: other item is NULL\n",getClassName()); }
  FXOldListItem *item=createItem();
  item->prev=other;
  item->next=other->next;
  other->next=item;
  if(item->next) item->next->prev=item; else lastitem=item; 
  item->label=text;
  item->state=0;
  item->data=ptr;
  item->y=0;
  recalc();
  return item;
  }


// Add item before other
FXOldListItem* FXOldList::addItemBefore(FXOldListItem* other,const FXString& text,void* ptr){
  if(other==NULL){ fxerror("%s::addItemBefore: other item is NULL\n",getClassName()); }
  FXOldListItem *item=createItem();
  item->next=other;
  item->prev=other->prev;
  other->prev=item;
  if(item->prev) item->prev->next=item; else firstitem=item;
  item->label=text;
  item->state=0;
  item->data=ptr;
  item->y=0;
  recalc();
  return item;
  }


// Remove node from list
void FXOldList::removeItem(FXOldListItem* item){
  if(item){
    if(item->prev) item->prev->next=item->next; else firstitem=item->next;
    if(item->next) item->next->prev=item->prev; else lastitem=item->prev;
    if(currentitem==item) currentitem=NULL;
    if(anchoritem==item) anchoritem=NULL;
    deleteItem(item);
    recalc();
    }
  }


// Remove items [fm..to]
void FXOldList::removeItems(FXOldListItem* fm,FXOldListItem* to){
  if(fm && to){
    register FXOldListItem *item;
    if(fm->prev) fm->prev->next=to->next; else firstitem=to->next;
    if(to->next) to->next->prev=fm->prev; else lastitem=fm->prev;
    do{
      item=fm;
      if(currentitem==item) currentitem=NULL;
      if(anchoritem==item) anchoritem=NULL;
      fm=fm->next;
      deleteItem(item);
      }
    while(item!=to);
    recalc();
    }
  }


// Remove all items
void FXOldList::removeAllItems(){
  register FXOldListItem *item;
  while(firstitem){
    item=firstitem;
    firstitem=firstitem->next;
    deleteItem(item);
    }
  firstitem=NULL;
  lastitem=NULL;
  currentitem=NULL;
  anchoritem=NULL;
  recalc();
  }


// Set current item
void FXOldList::setCurrentItem(FXOldListItem* item){
  if(item!=currentitem){ 
    if(item){
      item->state|=LISTITEM_CURRENT;
      updateItem(item);
      }
    if(currentitem){
      currentitem->state&=~LISTITEM_CURRENT;
      updateItem(currentitem);
      }
    currentitem=item;
    }
  }


// Set anchor item
void FXOldList::setAnchorItem(FXOldListItem* item){
  anchoritem=item;
  }


void FXOldList::setItemText(FXOldListItem* item,const FXString& text){
  if(item==NULL){ fxerror("%s::setItemText: item is NULL\n",getClassName()); }
  item->label=text;
  recalc();
  }


FXString FXOldList::getItemText(const FXOldListItem* item) const {
  if(item==NULL){ fxerror("%s::getItemText: item is NULL\n",getClassName()); }
  return item->label;
  }


void FXOldList::setItemData(FXOldListItem* item,void* ptr){
  if(item==NULL){ fxerror("%s::setItemData: item is NULL\n",getClassName()); }
  item->data=ptr;
  }


void* FXOldList::getItemData(const FXOldListItem* item) const {
  if(item==NULL){ fxerror("%s::getItemData: item is NULL\n",getClassName()); }
  return item->data;
  }


// Find item by prefix of length len
FXOldListItem* FXOldList::findItem(const FXString& text,FXuint len) const {
  register FXOldListItem* item;
  for(item=firstitem; item; item=item->next){
    if(compare(text,item->label,len)==0) return item;
    }
  return NULL;
  }


// Change the font
void FXOldList::setFont(FXFont* fnt){
  if(!fnt){ fxerror("%s::setFont: NULL font specified.\n",getClassName()); }
  font=fnt;
  recalc();
  }


// Set text color
void FXOldList::setTextColor(FXColor clr){
  textColor=clr;
  update();
  }


// Set select background color
void FXOldList::setSelBackColor(FXColor clr){
  selbackColor=clr;
  update();
  }


// Set selected text color
void FXOldList::setSelTextColor(FXColor clr){
  seltextColor=clr;
  update();
  }


// Change list style
void FXOldList::setListStyle(FXuint style){
  options=(options&~OLDLIST_MASK) | (style&OLDLIST_MASK);
  }


// Get list style
FXuint FXOldList::getListStyle() const { 
  return (options&OLDLIST_MASK); 
  }


// Change help text
void FXOldList::setHelpText(const FXString& text){
  help=text;
  }


// Save data
void FXOldList::save(FXStream& store) const {
  FXScrollArea::save(store);
  store << font;
  store << textColor;
  store << selbackColor;
  store << seltextColor;
  store << totalWidth;
  store << totalHeight;
  store << help;
  }


// Load data
void FXOldList::load(FXStream& store){ 
  FXScrollArea::load(store);
  store >> font;
  store >> textColor;
  store >> selbackColor;
  store >> seltextColor;
  store >> totalWidth;
  store >> totalHeight;
  store >> help;
  }


// Clean up
FXOldList::~FXOldList(){
  removeAllItems();
  firstitem=(FXOldListItem*)-1;
  lastitem=(FXOldListItem*)-1;
  anchoritem=(FXOldListItem*)-1;
  currentitem=(FXOldListItem*)-1;
  font=(FXFont*)-1;
  }

