package datarep.beans;

import java.awt.*;
import java.awt.event.*;
import java.util.Vector;

public class Knob extends Canvas implements MouseMotionListener {

	int min, max, value, radius;
	int tmin, tmax;
	Image offscreen;
	Color color = SystemColor.control;

	public Knob() {
		this(0,100,50,20);
	}

	public Knob(int min, int max, int value, int radius) {
		this.min = min;
		this.max = max;
		this.value = value;
		this.radius = radius;
		addMouseMotionListener(this);
	}

	public void setMinimum(int min) {
		this.min = min;
		repaint();
	}

	public void setMaximum(int max) {
		this.max = max;
		repaint();
	}

	public void setValue(int value) {
		this.value = value;
		repaint();
	}

	public void setRadius(int radius) {
		this.radius = radius;
		invalidate();
		try { getParent().validate(); } catch ( Exception e ) {}
		repaint();
	}

	public void setKnobColor(Color color) {
		if ( color != null ) this.color = color;
		repaint();
	}

	public int getMinimum() {
		return min;
	}

	public int getMaximum() {
		return max;
	}

	public int getValue() {
		return value;
	}

	public int getRadius() {
		return radius;
	}

	public Color getKnobColor() {
		return color;
	}

	public void invalidate() {
		offscreen = null;
		super.invalidate();
	}

	public void update(Graphics g) {
		if ( offscreen == null ) offscreen = createImage(getSize().width,getSize().height);
		Graphics og = offscreen.getGraphics();
		paint(og);
		og.dispose();
		g.drawImage(offscreen,0,0,this);
	}

	public void paint(Graphics g) {
		Dimension sz = getSize();
		g.setColor(getBackground());
		g.fillRect(0,0,sz.width,sz.height);

		g.setColor(Color.black);
		g.drawOval(0,0,sz.width-1,sz.height-1);
		g.setColor(Color.darkGray);
		g.drawOval(0,0,sz.width-2,sz.height-2);
		g.setColor(Color.darkGray);
		g.drawOval(0,0,sz.width-3,sz.height-3);
		g.setColor(Color.gray);
		g.drawOval(0,0,sz.width-4,sz.height-4);
		g.setColor(Color.gray);
		g.drawOval(0,0,sz.width-5,sz.height-5);
		g.setColor(color);
		g.fillOval(0,0,sz.width-6,sz.height-6);
		g.setColor(Color.lightGray);
		g.drawOval(0,0,sz.width-6,sz.height-6);
		g.setColor(Color.black);
		g.drawOval(0,0,sz.width-1,sz.height-1);

		tmin = (int)(getSize().width/2*0.6);
		tmax = (int)(getSize().width/2*0.8);
		double theta = (value-min)*2*Math.PI/(max-min);
		g.drawLine(
			(int)(sz.width/2-tmin*Math.sin(theta)),
			(int)(sz.height/2+tmin*Math.cos(theta)),
			(int)(sz.width/2-tmax*Math.sin(theta)),
			(int)(sz.height/2+tmax*Math.cos(theta))
		);

		FontMetrics fm = g.getFontMetrics();
		int x = (sz.width - fm.stringWidth(String.valueOf(value)))/2;
		int y = (fm.getAscent() + (sz.height - (fm.getAscent()
			 + fm.getDescent()))/2);
		g.drawString(String.valueOf(value),x,y);

	}

	public void mouseMoved(MouseEvent e) {
	}
	public void mouseDragged(MouseEvent e) {
		int dx = getSize().width/2 - e.getX();
		int dy = getSize().height/2 - e.getY();
		double theta = (Math.atan(((double)dy)/dx)+Math.PI/2);
		if ( dx < 0 ) theta += Math.PI;
		value = min + (int)(theta/Math.PI/2*(max-min));
		repaint();
		postKnobTurn();
	}

	public Dimension getPreferredSize() {
		return new Dimension(2*radius+1,2*radius+1);
	}

	public Dimension getMinimumSize() {
		return new Dimension(2*radius+1,2*radius+1);
	}

	Vector turnListeners = new Vector(2,1);

	public synchronized void addKnobTurnListener(KnobTurnListener l) {
		if ( l != null && !turnListeners.contains(l) ) turnListeners.addElement(l);
	}

	public synchronized void removeKnobTurnListener(KnobTurnListener l) {
		turnListeners.removeElement(l);
	}

	private void postKnobTurn() {
		for ( int i = 0; i < turnListeners.size(); i++ ) 
			((KnobTurnListener)turnListeners.elementAt(i)).knobValueChanged(
				new KnobTurnEvent(
					this,
					value,
					min,
					max
				)
			);
	}


	public static void main(String arg[]) {
		Frame f = new Frame("Test of Knob");
		f.setBackground(SystemColor.control);
		f.add(new Knob());
		f.pack();
		f.show();
	}
}
