package sra.smalltalk;

import java.io.Writer;
import java.io.IOException;
import java.lang.IndexOutOfBoundsException;

/**
 * StInterval class
 * 
 * 	@author:    NISHIHARA Satoshi
 * 	@created:   2000/01/13 (by NISHIHARA Satoshi)
 * 	@updated:   2000/01/21 (by NISHIHARA Satoshi)
 * 	@version:   3.1
 * 	@JDK:       1.1.6 or higher
 * 	@copyright: 1999-2000 SRA (Software Research Associates, Inc.)
 */
public class StInterval extends StObject {
	/**
	 * initial number in the progression.
	 */
	private double start;
	/**
	 * last number in the progression.
	 */
	private double stop;
	/**
	 * increment number for determining the next number in the progression.
	 */
	private double step = 1.0;
	public StInterval() {
		super();
	}
	/**
	 * Create a new StInterval.
	 * @param startNumber double
	 * @param stopNumber double
	 */
	public StInterval(double startNumber, double stopNumber) {
		super();
		this.setFrom_to_by_(startNumber, stopNumber, 1);
	}
	/**
	 * Create a new StInterval.
	 * @param startNumber double
	 * @param stopNumber double
	 * @param stepNumber double
	 */
	public StInterval(double startNumber, double stopNumber, double stepNumber) {
		super();
		this.setFrom_to_by_(startNumber, stopNumber, stepNumber);
	}
	/**
	 * Computes the number of the index of this progression.
	 * note: not 1 but 0 origin.
	 *
	 * @param int
	 * @return double
	 */
	public double at_(int index) {
		if ((index >= 1) && (index <= this.size() - 1)) {
			double value = start;
			for (int i = 0; i < index; i++) {
				value += step;
			}
			return value;
			//return start + (step * (index - 1));
		}
		throw new IndexOutOfBoundsException();
	}
	/**
	 * Enumerate all elements of the receiver and evaluate the StBlockClosure.
	 *
	 * @param aBlock,sra.smalltalk.StBlockClosure
	 * @return double[]
	 */
	public double[] collect_(StBlockClosure aBlock) {
		double[] result = new double[this.size()];
		double nextValue = start;
		for (int i = 0; i < result.length; i++) {
			result[i] = ((Double) (aBlock.value_(new Double(nextValue)))).doubleValue();
			nextValue += step;
		}
		return result;
	}
	/**
	 * Enumerate all elements of the receiver and evaluate the StBlockClosure.
	 *
	 * @param aBlock,sra.smalltalk.StBlockClosure
	 */
	public void do_(StBlockClosure aBlock) {
		double value = start;
		if (step < 0) {
			while (stop <= value) {
				aBlock.value_(new Double(value));
				value += step;
			}
		} else {
			while (stop >= value) {
				aBlock.value_(new Double(value));
				value += step; 
			}
		}
	}
	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 *
	 * @return boolean
	 * @param anObject java.lang.Object
	 * @see java.util.Hashtable
	 */
	public boolean equals(Object other) {
		if (other.getClass() == this.getClass()) {
			StInterval aStInterval = (StInterval) other;
			return (
				(this.start == (aStInterval.start)) && 
				(this.step == (aStInterval.increment())) && 
				(this.size() == aStInterval.size())
			);
		}
		return false;
	}
	/**
	 * initinal number of this progression.
	 *
	 * @return double
	 */
	public double first() {
		return start;
	}
	/**
	 * Create a new StInterval.
	 * @param startNumber double
	 * @param stopNumber double
	 */
	public static StInterval From_to_(double startNumber, double stopNumber) {
		StInterval interval = new StInterval();
		interval.setFrom_to_by_(startNumber, stopNumber, 1);
		return interval;
	}
	/**
	 * Create a new StInterval.
	 * @param startNumber double
	 * @param stopNumber double
	 */
	public static StInterval From_to_by_(double startNumber, double stopNumber, double stepNumber) {
		StInterval interval = new StInterval();
		interval.setFrom_to_by_(startNumber, stopNumber, stepNumber);
		return interval;
	}
	// Hash Function

	/**
	 * Computes a hash code for this object.
	 *
	 * @return int
	 * @see java.util.Hashtable
	 */
	public int hashCode() {
		long bitsStart = Double.doubleToLongBits(start);
		int startHash = (int) (bitsStart ^ (bitsStart >> 32));
		long bitsStop = Double.doubleToLongBits(stop);
		int stopHash = (int) (bitsStop ^ (bitsStop >> 32));
		return (((startHash >> 2) | stopHash) >> 1 | this.size());
	}
	/**
	 * increment number of this progression.
	 *
	 * @return double
	 */
	public double increment() {
		return step;
	}
	/**
	 * Computes the last number of this progression.
	 *
	 * @return double
	 */
	public double last() {
		return stop - ((stop - start) % step);
	}
	/**
	 * Print my string representation on aWriter.
	 *
	 * @param aWriter java.io.Writer
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("(" + start + " to: " + stop);
		if (step != 1) {
			aWriter.write(" by: " + step);
		}
		aWriter.write(")");
	}
	/**
	 * Enumerate all elements of the receiver and evaluate the StBlockClosure.
	 *
	 * @param aBlock,sra.smalltalk.StBlockClosure
	 */
	public void reverseDo_(StBlockClosure aBlock) {
		double value = stop;
		if (step < 0) {
			while (start >= value) {
				aBlock.value_(new Double(value));
				value -= step;
			}
		} else {
			while (start <= value) {
				aBlock.value_(new Double(value));
				value -= step; 
			}
		}
	}
	/**
	 * private
	 *
	 * @param double
	 * @param double
	 * @param double
	 * @return sra.smalltalk.StInterval
	 */
	private StInterval setFrom_to_by_(double startNumber, double stopNumber, double stepNumber) {
		this.start = startNumber;
		this.stop = stopNumber;
		this.step = stepNumber;
		return this;
	}
	/**
	 * Computes the size of this progression.
	 *
	 * @return int
	 */
	public int size() {
		int size = 0;
		if (step < 0) {
			if (!(start < stop)) {
				size = (int) ((stop - start) / step) + 1;
			}
		} else {
			if (!(stop < start)) {
				size = (int) ((stop - start) / step) + 1;
			}
		}
		return size;
	}
	/**
	 * get initial number in the progression.
	 *
	 * @return double
	 */
	public double start() {
		return this.start;
	}
	/**
	 * set initial number in the progression.
	 *
	 * @param double
	 * @return double
	 */
	public double start_(double startValue) {
		return this.start = startValue;
	}
	/**
	 * get increment number for determining the next number in the progression.
	 *
	 * @return double
	 */
	public double step() {
		return this.step;
	}
	/**
	 * set increment number for determining 
	 * the next number in the progression.
	 *
	 * @param double
	 * @return double
	 */
	public double step_(double stepValue) {
		return this.step = stepValue;
	}
	/**
	 * get last number in the progression.
	 *
	 * @return double
	 */
	public double stop() {
		return this.stop;
	}
	/**
	 * set last number in the progression.
	 *
	 * @param double
	 * @return double
	 */
	public double stop_(double stopValue) {
		return this.stop = stopValue;
	}
	/**
	 * Store my string representation on the writer.
	 * @param aWriter java.io.Writer
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		this.printOn_(aWriter);
	}
}