package sra.smalltalk;

import java.io.*;
import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import java.util.*;

/**
 * StImage class
 * 
 * 	@author:    Hirotsugu Kondo
 * 	@created:   1998/11/10 (by Hirotsugu Kondo)
 * 	@updated:   2000/04/05 (by Mitsuhiro Asada)
 * 	@version:   3.1
 * 	@JDK:       1.1.6 or higher
 * 	@copyright: 1999-2000 SRA (Software Research Associates, Inc.)
 * 
 * 	$Id: StImage.java,v 3.4 2000/04/07 05:41:09 nisinaka Exp $
 */
public class StImage extends StObject implements Serializable {
	private Image image;
	private int width;
	private int height;
	private int[] bits;

	public static final int WriteZeros = 0;
	public static final int And = 1;
	public static final int Over = 3;
	public static final int Erase = 4;
	public static final int Reverse = 6;
	public static final int Under = 7;
	public static final int ReverseUnder = 13;
	public static final int WriteOnes = 15;
	public static final int Paint = 16;
	/**
	 * StImage constructor comment.
	 * @Param int,int
	 */
	public StImage(int width, int height) {
		super();
		int size = width * height;
		int[] newBits = new int[size];
		for (int index = 0; index < size; index++) {
			newBits[index] = -1;
		}
		this.width_(width);
		this.height_(height);
		this.bits_(newBits);
	}
	/**
	 * StImage constructor comment.
	 * @Param int,int,int[]
	 */
	public StImage(int width, int height, int[] bits) {
		super();
		this.width_(width);
		this.height_(height);
		this.bits_(bits);
	}
	/**
	 * StImage constructor comment.
	 */
	public StImage(Image image) {
		this.image = image;
		Component medium = new Canvas();
		int width = image.getWidth(medium);
		int height = image.getHeight(medium);
		this.width_(width);
		this.height_(height);
		int[] pixels = new int[width * height];
		PixelGrabber grabber = new PixelGrabber(image, 0, 0, width, height, pixels, 0, width);
		try {
			grabber.grabPixels();
		} catch (InterruptedException e) {
		}
		this.bits_(pixels);
	}
	/**
	 * @return int
	 * @param int
	 */
	private int _atIndex_(int index) {
		return this.bits[index];
	}
	/**
	 * @param int,int
	 */
	private void _atIndex_put_(int index, int pixelValue) {
		this.bits[index] = pixelValue;
		this.image = null;
	}
	/**
	 * @param int,int,int
	 * @return int
	 */
	public int _combinationedColorFrom_to_in_(int sourceColor, int destinationColor, int combinationRule) {
		if (combinationRule == WriteZeros) {
			return 0xFFFFFFFF;
		}
		if (combinationRule == And) {
			return sourceColor & destinationColor;
		}
		if (combinationRule == Over) {
			return sourceColor;
		}
		if (combinationRule == Under) {
			return sourceColor | destinationColor;
		}
		if (combinationRule == WriteOnes) {
			return 0xFF000000;
		}
		return sourceColor;
	}
	/**
	 * Answer image representing my contents using the palette aPalette .
	 * Use ErrorDiffusionImageRender to render the new image.
	 * ErrorDiffusion generate halftones using the Floyd-Steinberg errod diffusion algorithm.
	 * 
	 * @return sra.smalltalk.StImage
	 * @param aPalette java.awt.image.ColorModel
	 */
	public StImage _convertToPalette_RenderedByErrorDiffusion(IndexColorModel aPalette) {
		StImage newImage = new StImage(this.width(), this.height(), (int[]) this.bits().clone());
		Hashtable colorDictionary = new Hashtable();
		for (int y = 0; y < this.height(); y++) {
			for (int x = 0; x < newImage.width(); x++) {
				int i = y * newImage.width() + x;
				Integer pixel = new Integer(newImage.bits()[i]);
				Color currentColor = new Color(pixel.intValue());
				if (colorDictionary.containsKey(pixel)) {
					newImage.bits()[i] = ((Integer) colorDictionary.get(pixel)).intValue();
				} else {
					int fit = StColorValue._MaxDistanceSquared() + 1;
					int fitRGB = 0xFF000000;
					for (int j = 0; j < aPalette.getMapSize(); j++) {
						int pixelRGB = aPalette.getRGB(j);
						Color newColor = new Color(pixelRGB);
						Integer d = StColorValue._DistanceSquaredFrom_ifLessThan_(newColor, currentColor, fit);
						if (d != null) {
							fit = d.intValue();
							fitRGB = pixelRGB;
						}
					}
					colorDictionary.put(pixel, new Integer(fitRGB));
					newImage.bits()[i] = fitRGB;
				}
				Color actualColor = new Color(newImage.bits()[i]);
				float redError = (currentColor.getRed() - actualColor.getRed()) / 16.0f;
				float greenError = (currentColor.getGreen() - actualColor.getGreen()) / 16.0f;
				float blueError = (currentColor.getBlue() - actualColor.getBlue()) / 16.0f;
				if (x < newImage.width() - 1) {
					Color rightColor = new Color(newImage.atX_y_(x + 1, y));
					rightColor = new Color(Math.min(255, Math.max(0, rightColor.getRed() + (int) (redError * 7))), Math.min(255, Math.max(0, rightColor.getGreen() + (int) (greenError * 7))), Math.min(255, Math.max(0, rightColor.getBlue() + (int) (blueError * 7))));
					newImage.bits()[i + 1] = rightColor.getRGB();
				}
				if (y < newImage.height() - 1) {
					if (x > 0) {
						Color leftDownColor = new Color(newImage.atX_y_(x - 1, y + 1));
						leftDownColor = new Color(Math.min(255, Math.max(0, leftDownColor.getRed() + (int) (redError * 3))), Math.min(255, Math.max(0, leftDownColor.getGreen() + (int) (greenError * 3))), Math.min(255, Math.max(0, leftDownColor.getBlue() + (int) (blueError * 3))));
						newImage.atX_y_put_(x - 1, y + 1, leftDownColor.getRGB());
					}
					Color centerDownColor = new Color(newImage.atX_y_(x, y + 1));
					centerDownColor = new Color(Math.min(255, Math.max(0, centerDownColor.getRed() + (int) (redError * 5))), Math.min(255, Math.max(0, centerDownColor.getGreen() + (int) (greenError * 5))), Math.min(255, Math.max(0, centerDownColor.getBlue() + (int) (blueError * 5))));
					newImage.atX_y_put_(x, y + 1, centerDownColor.getRGB());
					if (x < newImage.width() - 1) {
						Color rightDownColor = new Color(newImage.atX_y_(x + 1, y + 1));
						rightDownColor = new Color(Math.min(255, Math.max(0, rightDownColor.getRed() + (int) (redError))), Math.min(255, Math.max(0, rightDownColor.getGreen() + (int) (greenError))), Math.min(255, Math.max(0, rightDownColor.getBlue() + (int) (blueError))));
						newImage.atX_y_put_(x + 1, y + 1, rightDownColor.getRGB());
					}
				}
			}
		}
		return newImage;
	}
	/**
	 * Answer image representing my contents using the palette aPalette .
	 * Use NearistPatintImageRender to render the new image.
	 * 
	 * @return sra.smalltalk.StImage
	 * @param aPalette java.awt.image.ColorModel
	 */
	public StImage _convertToPalette_RenderedByNearistPaint(IndexColorModel aPalette) {
		StImage newImage = new StImage(this.width(), this.height());
		Hashtable colorDictionary = new Hashtable();
		for (int i = 0; i < this.bits().length; i++) {
			Integer pixel = new Integer(this.bits()[i]);
			if (colorDictionary.containsKey(pixel)) {
				newImage.bits()[i] = ((Integer) colorDictionary.get(pixel)).intValue();
			} else {
				int fit = StColorValue._MaxDistanceSquared() + 1;
				int fitRGB = 0xFF000000;
				Color oldColor = new Color(pixel.intValue());
				for (int j = 0; j < aPalette.getMapSize(); j++) {
					int pixelRGB = aPalette.getRGB(j);
					Color newColor = new Color(pixelRGB);
					Integer d = StColorValue._DistanceSquaredFrom_ifLessThan_(newColor, oldColor, fit);
					if (d != null) {
						fit = d.intValue();
						fitRGB = pixelRGB;
					}
				}
				colorDictionary.put(pixel, new Integer(fitRGB));
				newImage.bits()[i] = fitRGB;
			}
		}
		return newImage;
	}
	/**
	 * display myself for new window
	 */
	public void _display() {
		StImage.Display_(this);
	}
	/**
	 * 
	 */
	private Image _makeImage() {
		int width = this.width();
		int height = this.height();
		int[] pixels = new int[width * height];
		for (int i = 0; i < width * height; i++) {
			pixels[i] = this.bits[i];
		}
		int offset = 0;
		int scansize = width;
		MemoryImageSource source = new MemoryImageSource(width, height, pixels, offset, scansize);
		image = Toolkit.getDefaultToolkit().createImage(source);
		return image;
	}
	/**
	 * @param Point
	 * @return int
	 */
	public int atPoint_(Point point) {
		int index = point.y * this.width() + point.x;
		return this.bits[index];
	}
	/**
	 * @param Point,int
	 */
	public void atPoint_put_(Point point, int pixelValue) {
		int index = point.y * this.width() + point.x;
		this._atIndex_put_(index, pixelValue);
	}
	/**
	 * @return int
	 * @param int,int
	 */
	public int atX_y_(int x, int y) {
		int index = y * this.width() + x;
		return this._atIndex_(index);
	}
	/**
	 * @param int,int,int
	 */
	public void atX_y_put_(int x, int y, int pixelValue) {
		int index = y * this.width() + x;
		this._atIndex_put_(index, pixelValue);
	}
	/**
	 * @return int[]
	 */
	public int[] bits() {
		return this.bits;
	}
	/**
	 * @param int[]
	 */
	public void bits_(int[] bits) {
		this.bits = bits;
	}
	/**
	 * @return Rectangle
	 */
	public Rectangle bounds() {
		return new Rectangle(0, 0, this.width, this.height);
	}
	/**
	 * Answer image representing my contents using the palette aPalette .
	 * 
	 * @return sra.smalltalk.StImage
	 * @param aPalette java.awt.image.ColorModel
	 */
	public StImage convertToPalette_(IndexColorModel aPalette) {
		return this._convertToPalette_RenderedByNearistPaint(aPalette);
	}
	/**
	 *
	 * @return StImage
	 * @param Rectangle,Point,StImage,int
	 */
	public StImage copy_from_in_rule_(Rectangle destRectangle, Point sourcePt, StImage sourceImage, int combinationRule) {
		int width = destRectangle.width;
		int height = destRectangle.height;
		int destOriginX = destRectangle.x;
		int destOriginY = destRectangle.y;
		int destCornerX = destRectangle.x + width - 1;
		int destCornerY = destRectangle.y + height - 1;
		int sourceOriginX = sourcePt.x;
		int sourceOriginY = sourcePt.y;
		int sourceCornerX = sourcePt.x + width - 1;
		int sourceCornerY = sourcePt.y + height - 1;
		if (this.bounds().contains(destRectangle.getLocation()) == false) {
			return null;
		}
		if (this.bounds().contains(destCornerX, destCornerY) == false) {
			return null;
		}
		if (sourceImage.bounds().contains(sourcePt) == false) {
			return null;
		}
		if (sourceImage.bounds().contains(sourceCornerX, sourceCornerY) == false) {
			return null;
		}
		for (int y = 0; y < height; y++) {
			for (int x = 0; x < width; x++) {
				Point sourcePoint = new Point(x + sourceOriginX, y + sourceOriginY);
				Point destinationPoint = new Point(x + destOriginX, y + destOriginY);
				int newPaint = this._combinationedColorFrom_to_in_(sourceImage.atPoint_(sourcePoint), this.atPoint_(destinationPoint), combinationRule);
				this.atPoint_put_(destinationPoint, newPaint);
			}
		}
		return this;
	}
	/**
	 * @param Point
	 * @return StImage
	 */
	public StImage copyEmpty_(Point newExtent) {
		return new StImage(newExtent.x, newExtent.y);
	}
	/**
	 * Display image for new window
	 *
	 */
	public static void Display_(StImage anImage) {
		Frame aFrame = new Frame();
		aFrame.setTitle("Display");
		aFrame.setLocation(100, 100);
		aFrame.setLayout(new BorderLayout());
		aFrame.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				e.getWindow().dispose();
			}
		});
		final StImage _image = anImage;
		Canvas newCanvas = new Canvas() {
			public void paint(Graphics g) {
				_image.displayOn_(g);
				return;
			}
		};
		newCanvas.setBounds(0, 0, anImage.width(), anImage.height());
		aFrame.add(newCanvas);
		aFrame.pack();
		aFrame.show();
	}
	/**
	 * Update the view.
	 *
	 * @param aGraphics java.awt.Graphics
	 */
	public void displayOn_(Graphics aGraphics) {
		this.displayOn_at_(aGraphics, new Point(0, 0));
	}
	/**
	 * Update the view.
	 *
	 * @param aGraphics java.awt.Graphics,point
	 */
	public void displayOn_at_(Graphics aGraphics, Point aPoint) {
		Graphics graphics = aGraphics.create();
		try {
			Canvas medium = new Canvas();
			MediaTracker mt = new MediaTracker(medium);
			mt.addImage(this.image(), 0);
			graphics.drawImage(this.image(), aPoint.x, aPoint.y, medium);
		} finally {
			graphics.dispose();
		}
	}
	/**
	 * @return Point
	 */
	public Point extent() {
		return new Point(this.width, this.height);
	}
	/**
	 * @return int
	 */
	public int height() {
		return this.height;
	}
	/**
	 * @param int
	 */
	public void height_(int height) {
		this.height = height;
	}
	/**
	 * @return Image
	 */
	public Image image() {
		if (this.image != null) {
			return this.image;
		}
		this.image = this._makeImage();
		return this.image;
	}
	/**
	 * @return int[]
	 */
	public int[] packedRowAt_(int rowIndex) {
		int width = this.width();
		int[] bits = this.bits();
		int[] rowBytes = new int[width];
		for (int index = 0; index < width; index++) {
			rowBytes[index] = bits[width * rowIndex + index];
		}
		return rowBytes;
	}
	/**
	 * @param int,int[]
	 */
	public void packedRowAt_into_(int rowIndex, int[] anArray) {
		int width = this.width();
		int[] bits = this.bits();
		for (int index = 0; index < width; index++) {
			bits[width * rowIndex + index] = anArray[index];
		}
	}
	/**
	 * @param double,double
	 * @return Point
	 */
	public Point scaledExtent_(double scaleX, double scaleY) {
		Double x = new Double(this.width * 1.0 / scaleX);
		Double y = new Double(this.height * 1.0 / scaleY);
		return new Point(x.intValue(), y.intValue());
	}
	/**
	 * Store my string representation on the writer.
	 * @param aWriter java.io.Writer
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		aWriter.write("(Image extent: ");
		aWriter.write(String.valueOf(width));
		aWriter.write('@');
		aWriter.write(String.valueOf(height));
		aWriter.write(" depth: ");
		aWriter.write("24"); // for now. :-)
		aWriter.write(" bitPerPixel: ");
		aWriter.write("24"); // for now. :-)
		aWriter.write(" palette: ");
		aWriter.write("(FixedPalette redShift: 16 redMask: 255 greenShift: 8 greenMask: 255 blueShift: 0 blueMask: 255)"); // for now. :-)
		aWriter.write(" usingBits: ");
		(new StByteArray(bits)).storeOn_(aWriter);
		aWriter.write(')');
	}
	/**
	 *
	 * @return StImage
	 * @param Rectangle,Point,StImage,int
	 */
	public StImage tile_from_in_rule_(Rectangle destRectangle, Point sourcePt, StImage sourceImage, int combinationRule) {
		int width = destRectangle.width;
		int height = destRectangle.height;
		int sourceWidth = sourceImage.width();
		int sourceHeight = sourceImage.height();
		int destOriginX = destRectangle.x;
		int destOriginY = destRectangle.y;
		int destCornerX = destRectangle.x + width - 1;
		int destCornerY = destRectangle.y + height - 1;
		int sourceOriginX = sourcePt.x;
		int sourceOriginY = sourcePt.y;
		int sourceCornerX = sourcePt.x + width - 1;
		int sourceCornerY = sourcePt.y + height - 1;
		if (this.bounds().contains(destRectangle.getLocation()) == false) {
			return null;
		}
		if (this.bounds().contains(destCornerX, destCornerY) == false) {
			return null;
		}
		for (int y = 0; y < height; y++) {
			for (int x = 0; x < width; x++) {
				Point sourcePoint = new Point((x + sourceOriginX) % sourceWidth, (y + sourceOriginY) % sourceHeight);
				Point destinationPoint = new Point(x + destOriginX, y + destOriginY);
				int newPaint = this._combinationedColorFrom_to_in_(sourceImage.atPoint_(sourcePoint), this.atPoint_(destinationPoint), combinationRule);
				this.atPoint_put_(destinationPoint, newPaint);
			}
		}
		return this;
	}
	/**
	 * @param Point
	 * @return java.awt.Color
	 */
	public Color valueAtPoint_(Point point) {
		int pixelValue = this.atPoint_(point);
		ColorModel colorModel = ColorModel.getRGBdefault();
		int red = colorModel.getRed(pixelValue);
		int green = colorModel.getGreen(pixelValue);
		int blue = colorModel.getBlue(pixelValue);
		return new Color(red, green, blue);
	}
	/**
	 * @param Point,Color
	 */
	public void valueAtPoint_put_(Point point, Color color) {
		this.atPoint_put_(point, (color.getRGB()));
	}
	/**
	 * @return int
	 */
	public int width() {
		return this.width;
	}
	/**
	 * @param int
	 */
	public void width_(int width) {
		this.width = width;
	}
	/**
	 * Store my image on the stream.
	 *
	 * @param out java.io.ObjectOutputStream
	 */
	private void writeObject(ObjectOutputStream out) throws IOException {
		image = null;
		out.defaultWriteObject();
	}
}
