package sra.smalltalk;

import java.io.*;
import java.util.*;

/**
 * StByteArray class
 * 
 * 	@author:    He Weijie
 * 	@created:   1998/09/20 (by xiao lai)
 * 	@updated:   1998/10/19 (by nisinaka)
 * 	@version:   3.1
 * 	@JDK:       1.1.6 or higher
 * 	@copyright: 1999-2000 SRA (Software Research Associates, Inc.)
 * 
 * 	$Id: StByteArray.java,v 3.1 2000/04/07 05:41:09 nisinaka Exp $
 */
public class StByteArray extends StObject {
	/**
	 * The array.
	 */
	protected byte[] byteArray = null;
	/**
	 * Constructor Method
	 */
	public StByteArray() {
		this(0);
	}
	/**
	 * Create a new instance of StByteArray and initialize it with the array of bytes.
	 *
	 * @param bytes byte[]
	 */
	public StByteArray(byte[] bytes) {
		byteArray = new byte[bytes.length];
		for (int i = 0; i < bytes.length; i++) {
			byteArray[i] = bytes[i];
		}
	}
	/**
	 * Create a new instance of StByteArray and initialize it with the array of bytes.
	 *
	 * @param ints int[]
	 */
	public StByteArray(int[] ints) {
		byteArray = new byte[ints.length];
		for (int i = 0; i < ints.length; i++) {
			byteArray[i] = (byte) ints[i];
		}
	}
	/**
	 * Constainer Method
	 */
	public StByteArray(int size) {
		byteArray = new byte[size];
		for (int i = 0; i < size; i++) {
			byteArray[i] = 0;
		}
	}
	/**
	 * Convert this StByteArray as an array of byte.
	 *
	 * @return byte[]
	 */
	public byte[] _asBytes() {
		byte[] newArray = new byte[byteArray.length];
		for (int i = 0; i < byteArray.length; i++) {
			newArray[i] = byteArray[i];
		}
		return newArray;
	}
	/**
	 * Convert this StByteArray as an array of int.
	 *
	 * @return int[]
	 */
	public int[] _asInts() {
		int[] ints = new int[byteArray.length];
		for (int i = 0; i < byteArray.length; i++) {
			ints[i] = (int) byteArray[i] & 0xff;
		}
		return ints;
	}
	/**
	 * Encode the receiver into a printable string.
	 *
	 * @return java.lang.String
	 */
	public String asPackedString() {
		int size = this.size();
		StByteArray result = new StByteArray((size + 2) / 3 * 4);
		if (size > 300) {
			// fast RasterOp algorithm is not supported.
			// use slower algorithm.
			this.slowEncodeFrom_into_startingAt_(1, result, 1);
		} else {
			this.slowEncodeFrom_into_startingAt_(1, result, 1);
		}
		return new String(result._asBytes());
	}
	/**
	 * Answer the byte value at the specified place.
	 *
	 * @return byte
	 * @param index int
	 */
	public byte at_(int index) {
		return byteArray[index-1];
	}
	/**
	 * Put the byte value at the specified place.
	 *
	 * @param index int
	 * @param value byte
	 */
	public void at_put_(int index, byte value) {
		byteArray[index-1] = value;
	}
	/**
	 * Create a StByteArray from a packed string.
	 *
	 * @param string sra.smalltalk.StString
	 * @return sra.smalltalk.StByteArray
	 */
	public static StByteArray FromPackedString_(String packedString) {
		int size = packedString.length();
		if (size == 0) {
			return new StByteArray();
		}
		int last = packedString.charAt(size - 1);
		int resultSize = Math.round(((float) size) / 4) * 3;
		int extra;
		if (last >= 96) {
			resultSize = resultSize - 3 + last - 96;
			extra = 1;
		} else {
			extra = 0;
		}
		StByteArray result = new StByteArray(resultSize);
		if (size > 400) {
			// The fast RasterOp algorithm is not implemented yet.
			// Do it with the slower algorithm.
			SlowDecodeFromFrom_Starting_Into_(1, packedString, 1, result);
//			FastDecodeFrom_Into_(packedString, result);
//			SlowDecodeFromFrom_Starting_Into_(size - Math.round((float)extra/16)*12+1 , packedString , size - Math.round(((float)extra)/16)*16+1 , result);
		} else {
			SlowDecodeFromFrom_Starting_Into_(1, packedString, 1, result);
		}
		return result;
	}
	/**
	 * Answer the size of the byte array.
	 *
	 * @return int
	 */
	public final int size() {
		return byteArray.length;
	}
	/**
	 * Decode a string into a ByteArray.
	 *
	 * @param int start
	 * @param source sra.smalltalk.StString
	 * @param int index
	 * @param dest sra.smalltalk.StByteArray
	 */
	private static void SlowDecodeFromFrom_Starting_Into_(int start, String source, int index, StByteArray dest) {
		int w;
		int to = start;
		int from = index;
		int stop = dest.size();
		while (to <= stop) {
			w = ((source.charAt(from - 1)) & 63) << 18;
			w += ((source.charAt(from)) & 63) << 12;
			w += ((source.charAt(from + 1)) & 63) << 6;
			w += (source.charAt(from + 2)) & 63;
			from += 4;
			dest.at_put_(to, (byte) (w >> 16));
			if (to < stop) {
				dest.at_put_(to + 1, (byte) ((w >> 8) & 255));
				if (to + 1 < stop) {
					dest.at_put_(to + 2, (byte) (w & 255));
				}
			}
			to = to + 3;
		}
	}
	/**
	 * Encode the receiver into printable characters.
	 *
	 * @param start int
	 * @param dest sra.smalltalk.StByteArray
	 * @param index int
	 */
	private void slowEncodeFrom_into_startingAt_(int start, StByteArray dest, int index) {
		int from = start;
		int to = index;
		int stop = this.size();
		int w;
		while (from <= stop) {
			w = this.at_(from) << 16;
			if (from < stop) {
				w += this.at_(from + 1) << 8;
				if (from + 1 < stop) {
					w += this.at_(from + 2);
				}
			}
			w = w ^ 8521760;
			from += 3;
			dest.at_put_(to, (byte) ((w >> 18) + 32));
			dest.at_put_(to + 1, (byte) (((w >> 12) & 63) + 32));
			dest.at_put_(to + 2, (byte) (((w >> 6) & 63) + 32));
			dest.at_put_(to + 3, (byte) ((w & 63) + 32));
			to += 4;
		}
		if (stop % 3 != 0) {
			dest.at_put_(to - 1, (byte) (stop % 3 + 96));
		}
	}
	/**
	 * Store my string representation on the writer.
	 * @param aWriter java.io.Writer
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		aWriter.write("(ByteArray fromPackedString: '");
		aWriter.write(this.asPackedString());
		aWriter.write("')");
	}
}