package sra.isr.image.searcher;

import java.util.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import sra.smalltalk.*;
import sra.isr.system.support.*;
import sra.isr.geometry.basic.*;
import sra.isr.image.browser.*;
import sra.isr.image.objects.*;
import sra.isr.image.support.*;
import sra.isr.interfaces.area.*;
import sra.isr.interfaces.button.*;
import sra.isr.interfaces.colors.*;
import sra.isr.interfaces.freehand.*;
import sra.isr.interfaces.gauge.*;
import sra.isr.interfaces.nineColors.*;
import sra.isr.interfaces.progress.*;
import sra.isr.interfaces.spot.*;

/**
 * IsrImageSearcher class
 * 
 * 	@author:        Mitsuhiro Asada
 * 	@created:       1999/12/21 (by Mitsuhiro Asada)
 * 	@updated:       2000/03/08 (by Mitsuhiro Asada)
 * 	@ImageSearcher: 019
 * 	@JDK:           1.1.6 or higher
 * 	@copyright:     2000 SRA (Software Research Associates, Inc.)
 * 
 * 	$Id: IsrImageSearcher.java,v 1.13 2000/04/07 04:24:16 m-asada Exp $
 */
public class IsrImageSearcher extends StApplicationModel {
	protected IsrImageFolder imageFolder;
	protected IsrAreaModel iconAreaModel;
	protected IsrButtonModel iconAreaSwitch;
	protected Isr9ColorsModel nineColorsModel;
	protected IsrButtonModel nineColorsSwitch;
	protected IsrColorChoiceSBH typicalColorModel;
	protected IsrButtonModel typicalColorSwitch;
	protected IsrGaugeModel averageGrayModel;
	protected IsrButtonModel averageGraySwitch;
	protected IsrSpotModel centerSpectraModel;
	protected IsrButtonModel centerSpectraSwitch;
	protected IsrFreehandModel freehandImageModel;
	protected IsrButtonModel freehandImageSwitch;
	protected IsrGaugeModel cutOffModel;
	protected IsrImageBrowser imageBrowser;
	private static Image OffImage;
	private static Image OnImage;
	//
	protected PopupMenu menu;
	/**
	 * Return average gray button image.
	 * 
	 * @return java.awt.Image
	 */
	private Image _averageGrayButtonImage() {
		String imageString = 
		"0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000000000000100000000000000000001000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000000"+
		"0011100100010011100010100011100011110011100000000101000100010110100001010110000111001011000011100001110000"+
		"0000010100010100010011000000010100100100010000000101000100011011010001011001000000101100100100010010001000"+
		"0011110010100111110010000011110111100111110000000101000100010010010001010001000111101000100100000011111000"+
		"0100010010100100000010000100010100000100000000000101000100010010010001010001001000101000100100000010000000"+
		"0100010001000100010010000100010111100100010000000101001100010010010001010001001000101000100100010010001000"+
		"0011110001000011100010000011110100010011100000000100110100010010010001010001000111101000100011100001110000"+
		"0000000000000000000000000000000011100000000000000000000000000000000000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
		String maskString =
		"0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000000000001110000000000000000011100000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000000000001110000000000000000011100000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000000000001110000000000000000011100000000000000000000000000000000000"+
		"0111111110111111110111110111110111111111110000001111101110111111110011111111001111111111100111110011111000"+
		"0111111110111111111111110111111111111111111000001111101110111111111011111111101111111111111111111111111100"+
		"0111111111111111111111110111111111111111111000001111101110111111111011111111101111111111111111111111111100"+
		"1111111111111111111111101111111111111111111000001111101110111111111011111111111111111111111110111111111100"+
		"1111111111111111111111001111111111111111111000001111111110111111111011111011111111111101111110111111111100"+
		"1111111111111111111111001111111111111111111000001111111110111111111011111011111111111101111111111111111100"+
		"1111111011101111111111001111111111111111111000001111111110111111111011111011111111111101111111111111111100"+
		"0111111011100111110111000111111111111111110000001111111110111111111011111011101111111101110111110011111000"+
		"0000000000000000000000000000000111110000000000000000000000000000000000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
		return _DecodeImageString_maskString_fore_back_width_height_(imageString, maskString, this.foreColor(), this.backColor(), 106, 16);
	}
	/**
	 * Return center spectra button image.
	 * 
	 * @return java.awt.Image
	 */
	private Image _centerSpectraButtonImage() {
		String imageString = 
		"00000000000000000000000000000000000000000000000000000000000000000000000000000000000000" + 
		"00000000000000000000000000000000000000000000000000000000000000000000000000000000000000" + 
		"00000000000000000000000000000000000000000000000000000000000000000000000000000000000000" + 
		"00000000000000000000000000000000000000000000000000000000000000000000000000000000000000" + 
		"00000000000000000000000100000000000000000000000000000000000000000000001000000000000000" + 
		"00000000000000000000000100000000000000000000000000000000000000000000001000000000000000" + 
		"00111000011100010110001110001110001010000000111001111000111000011100011100101000111000" + 
		"01000100100010011001000100010001001100000001000101000101000100100010001000110000000100" + 
		"01000000111110010001000100011111001000000000100001000101111100100000001000100000111100" + 
		"01000000100000010001000100010000001000000000011001000101000000100000001000100001000100" + 
		"01000100100010010001000100010001001000000001000101000101000100100010001000100001000100" +
		"00111000011100010001000010001110001000000000111001111000111000011100000100100000111100" + 
		"00000000000000000000000000000000000000000000000001000000000000000000000000000000000000" + 
		"00000000000000000000000000000000000000000000000001000000000000000000000000000000000000" + 
		"00000000000000000000000000000000000000000000000000000000000000000000000000000000000000" + 
		"00000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
		String maskString = 
		"00000000000000000000000000000000000000000000000000000000000000000000000000000000000000" + 
		"00000000000000000000000000000000000000000000000000000000000000000000000000000000000000" + 
		"00000000000000000000000000000000000000000000000000000000000000000000000000000000000000" + 
		"00000000000000000000001110000000000000000000000000000000000000000000011100000000000000" + 
		"00000000000000000000001110000000000000000000000000000000000000000000011100000000000000" + 
		"01111100111110111111011111011111011111000001111111111101111100111110111111111101111100" + 
		"11111111111111111111111111111111111111000011111111111111111111111111111111111101111110" + 
		"11111111111111111111111111111111111111000011111111111111111111111111111111111101111110" + 
		"11101111111111111111101110111111111110000011111111101111111111110111011101111011111110" + 
		"11101111111111111011101110111111111100000011111111101111111111110111011101110011111110" + 
		"11111111111111111011101111111111111100000011111111111111111111111111011111110011111110" + 
		"11111111111111111011101111111111111100000011111111111111111111111111011111110011111110" + 
		"01111100111110111011100111011111011100000001111111111101111100111110001111110001111110" + 
		"00000000000000000000000000000000000000000000000011100000000000000000000000000000000000" + 
		"00000000000000000000000000000000000000000000000011100000000000000000000000000000000000" + 
		"00000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
		return _DecodeImageString_maskString_fore_back_width_height_(imageString, maskString, this.foreColor(), this.backColor(), 86, 16);
	}
	/**
	 * Return cut off button image.
	 * 
	 * @return java.awt.Image
	 */
	private Image _cutOffButtonImage() {
		String imageString = 
		"0000000000000000000000000000000000000000" + 
		"0000000000000000000000000000000000000000" + 
		"0000000000000000000000000000000000000000" + 
		"0000000000000000000000000000000001000100" + 
		"0000000000000000100000000000000010001000" + 
		"0000000000000000100000000000000010001000" +
		"0011100010001001110000000111000111011100" + 
		"0100010010001000100000001000100010001000" + 
		"0100000010001000100000001000100010001000" + 
		"0100000010001000100000001000100010001000" + 
		"0100010010011000100000001000100010001000" + 
		"0011100001101000010000000111000010001000" + 
		"0000000000000000000000000000000000000000" + 
		"0000000000000000000000000000000000000000" + 
		"0000000000000000000000000000000000000000" + 
		"0000000000000000000000000000000000000000";
		String maskString = 
		"0000000000000000000000000000000000000000" + 
		"0000000000000000000000000000000000000000" + 
		"0000000000000000000000000000000011101110" + 
		"0000000000000001110000000000000111111110" + 
		"0000000000000001110000000000000111111110" + 
		"0111110111011111111000001111101111111110" + 
		"1111111111011111111000011111111111111110" + 
		"1111111111011111111000011111111111111110" + 
		"1110111111011101110000011101110111011100" + 
		"1110111111111101110000011101110111011100" + 
		"1111111111111101111000011111110111011100" + 
		"1111111111111101111000011111110111011100" + 
		"0111110011111100111000001111100111011100" + 
		"0000000000000000000000000000000000000000" + 
		"0000000000000000000000000000000000000000" + 
		"0000000000000000000000000000000000000000";
		return _DecodeImageString_maskString_fore_back_width_height_(imageString, maskString, this.foreColor2(), this.backColor2(), 40, 16);
	}
	/**
	 * Decode image from string.
	 *
	 * @return java.awt.Image
	 * @param imageString java.lang.String
	 * @param maskString java.lang.String
	 * @param foreColor java.awt.Color
	 * @param backColor java.awt.Color
	 * @param x int
	 * @param y int
	 */
	private Image _DecodeImageString_maskString_fore_back_width_height_(String imageString, String maskString, Color foreColor, Color backColor, int x, int y) {
		int[] imagePixels = new int[x * y];
		int imagePixel, maskPixel, pixel;
		for (int i = 0; i < x * y; i++) {
			imagePixel = Integer.parseInt(imageString.substring(i, i + 1), 2);
			maskPixel = Integer.parseInt(maskString.substring(i, i + 1), 2);
			if (imagePixel != 0) {
				pixel = foreColor.getRGB();
			} else
				if (maskPixel != 0) {
					pixel = backColor.getRGB();
				} else {
					pixel = 0x000000;
				}
			imagePixels[i] = pixel;
		};
		Image image = Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(x, y, imagePixels, 0, x));
		return image;
	}
	/**
	 * Return freehand image button image.
	 * 
	 * @return java.awt.Image
	 */
	private Image _freehandImageButtonImage() {
		String imageString = 
		"000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"000100000000000000000000100000000000000000000000100000010000000000000000000000000000000"+
		"001000000000000000000000100000000000000000000000100000000000000000000000000000000000000"+
		"001000000000000000000000100000000000000000000000100000000000000000000000000000000000000"+
		"011101010001110000111000101100001110010110000111100000010010110100001110001111001110000"+
		"001001100010001001000100110010000001011001001000100000010011011010000001010010010001000"+
		"001001000011111001111100100010001111010001001000100000010010010010001111011110011111000"+
		"001001000010000001000000100010010001010001001000100000010010010010010001010000010000000"+
		"001001000010001001000100100010010001010001001000100000010010010010010001011110010001000"+
		"001001000001110000111000100010001111010001000111100000010010010010001111010001001110000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000001110000000000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
		String maskString =
		"000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"001110000000000000000001110000000000000000000001110000111000000000000000000000000000000"+
		"011110000000000000000001110000000000000000000001110000111000000000000000000000000000000"+
		"011110000000000000000001110000000000000000000001110000111000000000000000000000000000000"+
		"111111111011111001111101111110011111111111001111110000111111111110011111011111111111000"+
		"111111111111111111111111111111011111111111111111110000111111111111011111111111111111100"+
		"111111111111111111111111111111011111111111111111110000111111111111011111111111111111100"+
		"011111110111111111111111111111111111111111111101110000111111111111111111111111111111100"+
		"011111100111111111111111110111111111111011111101110000111111111111111111111111111111100"+
		"011111100111111111111111110111111111111011111111110000111111111111111111111111111111100"+
		"011111100111111111111111110111111111111011111111110000111111111111111111111111111111100"+
		"011111100011111001111101110111011111111011101111110000111111111111011111111111111111000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000011111000000000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
		return _DecodeImageString_maskString_fore_back_width_height_(imageString, maskString, this.foreColor(), this.backColor(), 87, 16);
	}
	/**
	 * Return icon area button image.
	 * 
	 * @return java.awt.Image
	 */
	private Image _iconAreaButtonImage() {
		String imageString = 
		"000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"001000000000000000000000000000000000000000000000000000000000000000000000000"+
		"000000000000000000000000000000000000000000000000000100000000000000000010000"+
		"000000000000000000000000000000000000000000000000000100000000000000000010000"+
		"001001011010000111000111100111000000001110001000101110001110001011000111000"+
		"001001101101000000101001001000100000010001000101000100010001001100100010000"+
		"001001001001000111101111001111100000011111000010000100011111001000100010000"+
		"001001001001001000101000001000000000010000000010000100010000001000100010000"+
		"001001001001001000101111001000100000010001000101000100010001001000100010000"+
		"001001001001000111101000100111000000001110001000100010001110001000100001000"+
		"000000000000000000000111000000000000000000000000000000000000000000000000000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000000";
		String maskString =
		"000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"011100000000000000000000000000000000000000000000000000000000000000000000000"+
		"011100000000000000000000000000000000000000000000001110000000000000000111000"+
		"011100000000000000000000000000000000000000000000001110000000000000000111000"+
		"011111111111001111101111111111100000011111011101111111011111011111101111100"+
		"011111111111101111111111111111110000111111111111111111111111111111111111100"+
		"011111111111101111111111111111110000111111111111111111111111111111111111100"+
		"011111111111111111111111111111110000111111101111101110111111111111110111000"+
		"011111111111111111111111111111110000111111101111101110111111111101110111000"+
		"011111111111111111111111111111110000111111111111111111111111111101110111100"+
		"011111111111111111111111111111110000111111111111111111111111111101110111100"+
		"011111111111101111111111111111100000011111011101110111011111011101110011100"+
		"000000000000000000001111100000000000000000000000000000000000000000000000000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000000"+
		"000000000000000000000000000000000000000000000000000000000000000000000000000";
		return _DecodeImageString_maskString_fore_back_width_height_(imageString, maskString, this.foreColor(), this.backColor(), 75, 16);
	}
	/**
	 * Return image searcher button image.
	 * 
	 * @return java.awt.Image
	 */
	private Image _imageSearchButtonImage() {
		String imageString = 
		"0000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000"+
		"0000000000000000000000000000000010000000"+
		"0000000000000000000000000000000010000000"+
		"0000000000000000000000000000000010000000"+
		"0011100011100001110010100011100010110000"+
		"0100010100010000001011000100010011001000"+
		"0010000111110001111010000100000010001000"+
		"0001100100000010001010000100000010001000"+
		"0100010100010010001010000100010010001000"+
		"0011100011100001111010000011100010001000"+
		"0000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000";
		String maskString =
		"0000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000"+
		"0000000000000000000000000000000111000000"+
		"0000000000000000000000000000000111000000"+
		"0000000000000000000000000000000111000000"+
		"0111110111110011111111110111110111111000"+
		"1111111111111011111111111111111111111100"+
		"1111111111111011111111111111111111111100"+
		"1111111111111111111111101110111111111100"+
		"1111111111111111111111001110111111011100"+
		"1111111111111111111111001111111111011100"+
		"1111111111111111111111001111111111011100"+
		"0111110111110011111111000111110111011100"+
		"0000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000";
		return _DecodeImageString_maskString_fore_back_width_height_(imageString, maskString, this.foreColor2(), this.backColor2(), 40, 16);
	}
	/**
	 * Return menu button image.
	 * 
	 * @return java.awt.Image
	 */
	private Image _menuButtonImage() {
		String imageString = 
		"0000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000"+
		"0111000000000000000000000000000000000000"+
		"0010000000000000000000000000000000000000"+
		"0010000000000000000000000000000000000000"+
		"0010010110100001110001111001110000111000"+
		"0010011011010000001010010010001001000100"+
		"0010010010010001111011110011111000100000"+
		"0010010010010010001010000010000000011000"+
		"0010010010010010001011110010001001000100"+
		"0111010010010001111010001001110000111000"+
		"0000000000000000000001110000000000000000"+
		"0000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000";
		String maskString =
		"0000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000"+
		"1111100000000000000000000000000000000000"+
		"1111100000000000000000000000000000000000"+
		"1111100000000000000000000000000000000000"+
		"0111111111110011111011111111111001111100"+
		"0111111111111011111111111111111111111110"+
		"0111111111111011111111111111111111111110"+
		"0111111111111111111111111111111111111110"+
		"0111111111111111111111111111111111111110"+
		"1111111111111111111111111111111111111110"+
		"1111111111111111111111111111111111111110"+
		"1111111111111011111111111111111001111100"+
		"0000000000000000000011111000000000000000"+
		"0000000000000000000000000000000000000000"+
		"0000000000000000000000000000000000000000";
		return _DecodeImageString_maskString_fore_back_width_height_(imageString, maskString, this.foreColor2(), this.backColor2(), 40, 16);
	}
	/**
	 * Return nine colors button image.
	 * 
	 * @return java.awt.Image
	 */
	private Image _nineColorsButtonImage() {
		String imageString = 
		"00000000000000000000000000000000000000000000000000000000000000000"+
		"00000000000000000000000000000000000000000000000000000000000000000"+
		"00000000000000000000000000000000000000000000000000000000000000000"+
		"00000000010000000000000000000000000000000000100000000000000000000"+
		"00000000000000000000000000000000000000000000100000000000000000000"+
		"00000000000000000000000000000000000000000000100000000000000000000"+
		"01011000010101100001110000000011100001110000100111000101000111000"+
		"01100100010110010010001000000100010010001000101000100110001000100"+
		"01000100010100010011111000000100000010001000101000100100000100000"+
		"01000100010100010010000000000100000010001000101000100100000011000"+
		"01000100010100010010001000000100010010001000101000100100001000100"+
		"01000100010100010001110000000011100001110000100111000100000111000"+
		"00000000000000000000000000000000000000000000000000000000000000000"+
		"00000000000000000000000000000000000000000000000000000000000000000"+
		"00000000000000000000000000000000000000000000000000000000000000000"+
		"00000000000000000000000000000000000000000000000000000000000000000";
		String maskString =
		"00000000000000000000000000000000000000000000000000000000000000000"+
		"00000000000000000000000000000000000000000000000000000000000000000"+
		"00000000111000000000000000000000000000000001110000000000000000000"+
		"00000000111000000000000000000000000000000001110000000000000000000"+
		"00000000111000000000000000000000000000000001110000000000000000000"+
		"11111100111111110011111000000111110011111001111111101111101111100"+
		"11111110111111111111111100001111111111111101111111111111111111110"+
		"11111110111111111111111100001111111111111101111111111111111111110"+
		"11111110111111111111111100001110111111011101111101111111011111110"+
		"11101110111110111111111100001110111111011101111101111110011111110"+
		"11101110111110111111111100001111111111111101111111111110011111110"+
		"11101110111110111111111100001111111111111101111111111110011111110"+
		"11101110111110111011111000000111110011111001111111101110001111100"+
		"00000000000000000000000000000000000000000000000000000000000000000"+
		"00000000000000000000000000000000000000000000000000000000000000000"+
		"00000000000000000000000000000000000000000000000000000000000000000";
		return _DecodeImageString_maskString_fore_back_width_height_(imageString, maskString, this.foreColor(), this.backColor(), 65, 16);
	}
	/**
	 * Return typical color button image.
	 * 
	 * @return java.awt.Image
	 */
	private Image _typicalColorButtonImage() {
		String imageString = 
		"00000000000000000000000000000000000000000000000000000000000000000000000"+
		"00000000000000000000000000000000000000000000000000000000000000000000000"+
		"00000000000000000000000000000000000000000000000000000000000000000000000"+
		"00000000000000000001000000000000000100000000000000000000100000000000000"+
		"00100000000000000000000000000000000100000000000000000000100000000000000"+
		"00100000000000000000000000000000000100000000000000000000100000000000000"+
		"01110010001011110001001110000111000100000011100001110000100111000101000"+
		"00100010001010001001010001000000100100000100010010001000101000100110000"+
		"00100001010010001001010000000111100100000100000010001000101000100100000"+
		"00100001010010001001010000001000100100000100000010001000101000100100000"+
		"00100000100010001001010001001000100100000100010010001000101000100100000"+
		"00010000100011110001001110000111100100000011100001110000100111000100000"+
		"00000001000010000000000000000000000000000000000000000000000000000000000"+
		"00000010000010000000000000000000000000000000000000000000000000000000000"+
		"00000000000000000000000000000000000000000000000000000000000000000000000"+
		"00000000000000000000000000000000000000000000000000000000000000000000000";
		String maskString =
		"00000000000000000000000000000000000000000000000000000000000000000000000"+
		"00000000000000000000000000000000000000000000000000000000000000000000000"+
		"00000000000000000011100000000000001110000000000000000001110000000000000"+
		"01110000000000000011100000000000001110000000000000000001110000000000000"+
		"01110000000000000011100000000000001110000000000000000001110000000000000"+
		"11111111011111111011111111001111101110000111110011111001111111101111100"+
		"11111111011111111111111111101111111110001111111111111101111111111111100"+
		"11111111111111111111111111101111111110001111111111111101111111111111100"+
		"01110111111111011111111011111111111110001110111111011101111101111111000"+
		"01110011111111011111111011111111111110001110111111011101111101111110000"+
		"01111011111111111111111111111111111110001111111111111101111111111110000"+
		"01111011110111111111111111111111111110001111111111111101111111111110000"+
		"00111111110111111011111111001111111110000111110011111001111111101110000"+
		"00000111100111000000000000000000000000000000000000000000000000000000000"+
		"00000111000111000000000000000000000000000000000000000000000000000000000"+
		"00000000000000000000000000000000000000000000000000000000000000000000000";
		return _DecodeImageString_maskString_fore_back_width_height_(imageString, maskString, this.foreColor(), this.backColor(), 71, 16);
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 *
	 * @return boolean 
	 */
	public boolean allSwitchOff() {
		return (
			(!this.typicalColorSwitch().value()) &&
			(!this.averageGraySwitch().value()) && 
			(!this.nineColorsSwitch().value()) && 
			(!this.iconAreaSwitch().value()) && 
			(!this.centerSpectraSwitch().value()) && 
			(!this.freehandImageSwitch().value())
		);
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * (Smalltalk version only)
	 * 
	 * @return java.lang.Object
	 * @param asserBlock sra.smalltalk.StBlockClosure
	 * @param doBlock sra.smalltalk.StBlockClosure
	 * @param ensureBlock sra.smalltalk.StBlockClosure
	 */
	private Object assert_do_ensure_(StBlockClosure asserBlock, StBlockClosure doBlock, StBlockClosure ensureBlock) {
		/* asserBlock.value();
		Object result = null;
		try {
			result = doBlock.value();
		} finally {
			ensureBlock.value();
		}
		return result; */
		return null;
	}
	/**
	 * Answer average gray parameter.
	 *
	 * @return java.awt.Color 
	 */
	public Color averageGray() {
		return StColorValue.Brightness_(Math.max(0, Math.min(this.averageGrayModel().value() / 100, 1.0)));
	}
	/**
	 * Answer my average gray button.
	 * 
	 * @return sra.jun.goodies.button.JunButtonModel
	 */
	public IsrButtonModel averageGrayButton() {
		final IsrImageSearcher self = this;
		StBlockClosure block = new StBlockClosure() {
			public Object value_(Object o) {
				self.averageGrayButtonAction();
				return null;
			}
		};
		return IsrButtonModel.Value_visual_action_(false, this._averageGrayButtonImage(), block);
	}
	/**
	 * Set the average gray button action.
	 */
	public void averageGrayButtonAction() {
		IsrGaugeModel model = this.averageGrayModel();
		Frame[] frames = model.builder().frames();
		for (int i = 0; i < frames.length; i++) {
			if (frames[i].isShowing()) {
				frames[i].toFront();
			} else {
				frames[i].show();
			}
		}
		if (frames.length == 0) {
			Frame window = model.open();
			window.setTitle("Luminance");
		}
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 *
	 * @return sra.isr.interface.gauge.IsrGaugeModel 
	 */
	public IsrGaugeModel averageGrayModel() {
		if (averageGrayModel == null) {
			double[] array = {0.0, 100.0, 10.0, 1.0};
			averageGrayModel = IsrGaugeModel.Value_range_(0.0, array);
		}
		return averageGrayModel;
	}
	/**
	 * Answer my average gray switch button.
	 * 
	 * @return sra.jun.goodies.button.JunButtonModel
	 */
	public IsrButtonModel averageGraySwitch() {
		if (averageGraySwitch == null) {
			final IsrImageSearcher self = this;
			StBlockClosure block = new StBlockClosure() {
				public Object value_(Object o) {
					self.averageGraySwitchAction();
					return null;
				}
			};
			averageGraySwitch = IsrButtonModel.Value_visual_action_(false, this.offImage(), block);
		}
		return averageGraySwitch;
	}
	/**
	 * Set the average gray switch button action.
	 */
	public void averageGraySwitchAction() {
		IsrButtonModel averageGraySwitch = this.averageGraySwitch();
		averageGraySwitch.value_(!averageGraySwitch.value());
		if (averageGraySwitch.value() == true) {
			averageGraySwitch.visual_(this.onImage());
		} else {
			averageGraySwitch.visual_(this.offImage());
		}
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return java.awt.Color
	 */
	protected Color backColor() {
		return Color.getHSBColor((float) 0.0, (float) 0.0, (float) 0.9);
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return java.awt.Color
	 */
	protected Color backColor2() {
		return Color.getHSBColor((float) 0.0, (float) 0.0, (float) 0.95);
	}
	/**
	 * Answer center spectra parameter.
	 * 
	 * @return sra.jun.geometry.basic.Jun2dPoint
	 */
	public Isr2dPoint centerSpectra() {
		return this.centerSpectraModel().xy();
	}
	/**
	 * Answer my center spectra button.
	 * 
	 * @return sra.jun.goodies.button.JunButtonModel
	 */
	public IsrButtonModel centerSpectraButton() {
		final IsrImageSearcher self = this;
		StBlockClosure block = new StBlockClosure() {
			public Object value_(Object o) {
				self.centerSpectraButtonAction();
				return null;
			}
		};
		return IsrButtonModel.Value_visual_action_(false, this._centerSpectraButtonImage(), block);
	}
	/**
	 * Set the center spectra button action.
	 */
	public void centerSpectraButtonAction() {
		IsrSpotModel model = this.centerSpectraModel();
		Frame[] frames = model.builder().frames();
		for (int i = 0; i < frames.length; i++) {
			if (frames[i].isShowing()) {
				frames[i].toFront();
			} else {
				frames[i].show();
			}
		}
		if (frames.length == 0) {
			Frame window = model.open();
			window.setTitle("Center");
		}
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return sra.isr.interfaces.spot.IsrSpotModel
	 */
	public IsrSpotModel centerSpectraModel() {
		if (centerSpectraModel == null) {
			centerSpectraModel = IsrSpotModel.Xy_(new Isr2dPoint(0.5, 0.5));
		}
		return centerSpectraModel;
	}
	/**
	 * Answer my center spectra switch button.
	 * 
	 * @return sra.jun.goodies.button.JunButtonModel
	 */
	public IsrButtonModel centerSpectraSwitch() {
		if (centerSpectraSwitch == null) {
			final IsrImageSearcher self = this;
			StBlockClosure block = new StBlockClosure() {
				public Object value_(Object o) {
					self.centerSpectraSwitchAction();
					return null;
				}
			};
			centerSpectraSwitch = IsrButtonModel.Value_visual_action_(false, this.offImage(), block);
		}
		return centerSpectraSwitch;
	}
	/**
	 * Set the center spectra switch button action.
	 */
	public void centerSpectraSwitchAction() {
		IsrButtonModel centerSpectraSwitch = this.centerSpectraSwitch();
		centerSpectraSwitch.value_(!centerSpectraSwitch.value());
		if (centerSpectraSwitch.value() == true) {
			centerSpectraSwitch.visual_(this.onImage());
		} else {
			centerSpectraSwitch.visual_(this.offImage());
		}
	}
	/**
	 * Answer cut off parameter.
	 * 
	 * @return double
	 */
	public double cutOff() {
		return Math.max(0, Math.min(this.cutOffModel().value(), 1.0));
	}
	/**
	 * Answer my cut off button.
	 * 
	 * @return sra.jun.goodies.button.JunButtonModel
	 */
	public IsrButtonModel cutOffButton() {
		final IsrImageSearcher self = this;
		StBlockClosure block = new StBlockClosure() {
			public Object value_(Object o) {
				self.cutOffButtonAction();
				return null;
			}
		};
		return IsrButtonModel.Value_visual_action_(false, this._cutOffButtonImage(), block);
	}
	/**
	 * Set the cut off button action.
	 */
	public void cutOffButtonAction() {
		IsrGaugeModel model = this.cutOffModel();
		Frame[] frames = model.builder().frames();
		for (int i = 0; i < frames.length; i++) {
			if (frames[i].isShowing()) {
				frames[i].toFront();
			} else {
				frames[i].show();
			}
		}
		if (frames.length == 0) {
			Frame window = model.open();
			window.setTitle("Cut");
		}
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return sra.isr.interfaces.gauge.IsrGaugeModel
	 */
	public IsrGaugeModel cutOffModel() {
		if (cutOffModel == null) {
			double[] array = {0.0, 1.0, 10.0, 0.01};
			cutOffModel = IsrGaugeModel.Value_range_(0.0, array);
		}
		return cutOffModel;
	}
	/**
	 * Answer a default view.
	 *
	 * @return sra.smalltalk.StView
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new IsrImageSearcherViewAwt();
		} else {
			return this.CreateView("sra.isr.image.searcher.IsrImageSearcherViewSwing");
		}
	}
	/**
	 * Create new instance with directory.
	 *
	 * @param aString java.lang.String
	 * @return sra.isr.image.searcher.IsrImageSearcher
	 */
	public static IsrImageSearcher DirectoryName_(String aString) {
		return ImageFolder_(IsrImageFolder.DirectoryName_(aString));
	}
	/**
	 * Cleanup this On/Off image.
	 */
	public static void FlushImages() {
		OnImage = null;
		OffImage = null;
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return java.awt.Color
	 */
	protected Color foreColor() {
		return Color.blue;
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return java.awt.Color
	 */
	protected Color foreColor2() {
		return StColorValue.Brown;
	}
	/**
	 * Answer freehand image parameter.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage freehandImage() {
		return this.freehandImageModel().getImage();
	}
	/**
	 * Answer my freehand image button.
	 * 
	 * @return sra.jun.goodies.button.JunButtonModel
	 */
	public IsrButtonModel freehandImageButton() {
		final IsrImageSearcher self = this;
		StBlockClosure block = new StBlockClosure() {
			public Object value_(Object o) {
				self.freehandImageButtonAction();
				return null;
			}
		};
		return IsrButtonModel.Value_visual_action_(false, this._freehandImageButtonImage(), block);
	}
	/**
	 * Set the freehand image button action.
	 */
	public void freehandImageButtonAction() {
		IsrFreehandModel model = this.freehandImageModel();
		Frame[] frames = model.builder().frames();
		for (int i = 0; i < frames.length; i++) {
			if (frames[i].isShowing()) {
				frames[i].toFront();
			} else {
				frames[i].show();
			}
		}
		if (frames.length == 0) {
			Frame window = model.open();
			window.setTitle("Freehand");
		}
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return sra.isr.interfaces.freehand.IsrFreehandModel
	 */
	public IsrFreehandModel freehandImageModel() {
		if (freehandImageModel == null) {
			freehandImageModel = new IsrFreehandModel();
		}
		return freehandImageModel;
	}
	/**
	 * Answer my freehand image switch button.
	 * 
	 * @return sra.jun.goodies.button.JunButtonModel
	 */
	public IsrButtonModel freehandImageSwitch() {
		if (freehandImageSwitch == null) {
			final IsrImageSearcher self = this;
			StBlockClosure block = new StBlockClosure() {
				public Object value_(Object o) {
					self.freehandImageSwitchAction();
					return null;
				}
			};
			freehandImageSwitch = IsrButtonModel.Value_visual_action_(false, this.offImage(), block);
		}
		return freehandImageSwitch;
	}
	/**
	 * Set the freehand image switch button action.
	 */
	public void freehandImageSwitchAction() {
		IsrButtonModel freehandImageSwitch = this.freehandImageSwitch();
		freehandImageSwitch.value_(!freehandImageSwitch.value());
		if (freehandImageSwitch.value() == true) {
			freehandImageSwitch.visual_(this.onImage());
		} else {
			freehandImageSwitch.visual_(this.offImage());
		}
	}
	/**
	 * Create new instance with user parameter.
	 * 
	 * @return sra.isr.image.searcher.IsrImageSearcher or null
	 */
	public static IsrImageSearcher FromUser() {
		IsrImageFolder imageFolder = IsrImageFolder.FromUser();
		if (imageFolder == null) {
			return null;
		}
		IsrImageSearcher imageSearcher = ImageFolder_(imageFolder);
		return imageSearcher;
	}
	/**
	 * Answer icon area parameter.
	 * 
	 * @return java.awt.Rectangle
	 */
	public Rectangle iconArea() {
		return new Rectangle(0, 0, (int) Math.round(this.iconAreaModel().xy().x() * 100), (int) Math.round(this.iconAreaModel().xy().y() * 100));
	}
	/**
	 * Answer my icon area button.
	 * 
	 * @return sra.jun.goodies.button.JunButtonModel
	 */
	public IsrButtonModel iconAreaButton() {
		final IsrImageSearcher self = this;
		StBlockClosure block = new StBlockClosure() {
			public Object value_(Object o) {
				self.iconAreaButtonAction();
				return null;
			}
		};
		return IsrButtonModel.Value_visual_action_(false, this._iconAreaButtonImage(), block);
	}
	/**
	 * Set the icon area button action.
	 */
	public void iconAreaButtonAction() {
		IsrAreaModel model = this.iconAreaModel();
		Frame[] frames = model.builder().frames();
		for (int i = 0; i < frames.length; i++) {
			if (frames[i].isShowing()) {
				frames[i].toFront();
			} else {
				frames[i].show();
			}
		}
		if (frames.length == 0) {
			Frame window = model.open();
			window.setTitle("Extent");
		}
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return sra.isr.interfaces.area.IsrAreaModel
	 */
	public IsrAreaModel iconAreaModel() {
		if (iconAreaModel == null) {
			iconAreaModel = IsrAreaModel.Xy_(new Isr2dPoint(0.5, 0.5));
		}
		return iconAreaModel;
	}
	/**
	 * Answer my icon area switch button.
	 * 
	 * @return sra.jun.goodies.button.JunButtonModel
	 */
	public IsrButtonModel iconAreaSwitch() {
		if (iconAreaSwitch == null) {
			final IsrImageSearcher self = this;
			StBlockClosure block = new StBlockClosure() {
				public Object value_(Object o) {
					self.iconAreaSwitchAction();
					return null;
				}
			};
			iconAreaSwitch = IsrButtonModel.Value_visual_action_(false, this.offImage(), block);
		}
		return iconAreaSwitch;
	}
	/**
	 * Set the icon area switch button action.
	 */
	public void iconAreaSwitchAction() {
		IsrButtonModel iconAreaSwitch = this.iconAreaSwitch();
		iconAreaSwitch.value_(!iconAreaSwitch.value());
		if (iconAreaSwitch.value() == true) {
			iconAreaSwitch.visual_(this.onImage());
		} else {
			iconAreaSwitch.visual_(this.offImage());
		}
	}
	/**
	 * Answer my iamge browser.
	 * 
	 * @return sra.isr.image.browser.IsrImageBrowser
	 */
	public IsrImageBrowser imageBrowser() {
		if (imageBrowser == null) {
			imageBrowser = IsrImageBrowser.ImageObjects_(new IsrImageObject[0]);
			imageBrowser.slaveFlag_(true);
		}
		return imageBrowser;
	}
	/**
	 * Answer my iamge folder.
	 * 
	 * @return sra.isr.image.object.IsrImageFolder
	 */
	public IsrImageFolder imageFolder() {
		return imageFolder;
	}
	/**
	 * Create new instance with imageFolder.
	 *
	 * @param imageFolder sra.isr.image.object.IsrImageFolder
	 * @return sra.isr.image.searcher.IsrImageSearcher
	 */
	public static IsrImageSearcher ImageFolder_(IsrImageFolder imageFolder) {
		IsrImageSearcher model = new IsrImageSearcher();
		model.imageFolder_(imageFolder);
		return model;
	}
	/**
	 * Set my iamge folder.
	 * 
	 * @param anImageFolder sra.isr.image.object.IsrImageFolder
	 */
	public void imageFolder_(IsrImageFolder anImageFolder) {
		imageFolder = anImageFolder;
		this.imageObjects();
	}
	/**
	 * Answer my iamgeObjects.
	 * 
	 * @return sra.isr.image.object.IsrImageObject[]
	 */
	public IsrImageObject[] imageObjects() {
		return this.imageFolder().imageObjects();
	}
	/**
	 * Answer my image searcher button.
	 * 
	 * @return sra.jun.goodies.button.JunButtonModel
	 */
	public IsrButtonModel imageSearchButton() {
		final IsrImageSearcher self = this;
		StBlockClosure block = new StBlockClosure() {
			public Object value_(Object o) {
				self.imageSearchButtonAction();
				return null;
			}
		};
		return IsrButtonModel.Value_visual_action_(false, this._imageSearchButtonImage(), block);
	}
	/**
	 * Set the image searcher button action.
	 */
	public void imageSearchButtonAction() {
		this.search();
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 *
	 * @return sra.smalltalk.StImage
	 * @param aString java.lang.String
	 * @param foreColor java.awt.Color
	 * @param backbackColor java.awt.Color
	 */
	private StImage label_fore_back_(String aString, Color foreColor, Color backColor) {
		StImage figure = null;
		StImage shape = null;
		Image pixmap = null;
		Graphics gc = null;
		Point[] points;
		int i;
		String text = aString;
		FontMetrics fontMetrics = SystemResourceSupport.getFontMetrics(SystemResourceSupport.getFont());
		int width = fontMetrics.stringWidth(text);
		int base = fontMetrics.getAscent();
		int height = base + fontMetrics.getDescent();
		pixmap = SystemResourceSupport.createImage(width + 2, height + 2);
		try {
			gc = pixmap.getGraphics();
			gc.setColor(backColor);
			points = IsrImageProcessor.EightPoints();
			for (i = 0; i < points.length; i++) {
				gc.drawString(text, 1 + points[i].x, 1 + points[i].y + base);
			}
			gc.setColor(foreColor);
			gc.drawString(text, 1, 1 + base);
			figure = new StImage(pixmap);
		} finally {
			gc.dispose();
			pixmap.flush();
		}
		pixmap = SystemResourceSupport.createImage(width + 2, height + 2);
		try {
			gc = pixmap.getGraphics();
			gc.setColor(Color.black);
			points = IsrImageProcessor.EightPoints();
			for (i = 0; i < points.length; i++) {
				gc.drawString(text, 1 + points[i].x, 1 + points[i].y + base);
			}
			gc.drawString(text, 1, 1 + base);
			shape = new StImage(pixmap);
		} finally {
			gc.dispose();
			pixmap.flush();
		}
		shape = shape.convertToPalette_(IsrImageProcessor._WhiteBlackPalette());
		int white = Color.white.getRGB();
		int coverage = 0x00FFFFFF;
		for (int j = 0; j < shape.bits().length; j++) {
			if (shape.bits()[j] == white) {
				shape.bits()[j] = coverage;
			}
		}
		StImage iconImage = IsrImageProcessor._MakeImage_Shape_(figure, shape);
		return iconImage;
	}
	/**
	 * Create a popup menu of brush button.
	 *
	 * @return java.awt.PopupMenu
	 */
	protected PopupMenu menu() {
		StMenuModel rootMenu = new StMenuModel();
		//
		StMenu searcherMenu = new StMenu("Menu");
		searcherMenu.add(this.createMenuItem("open", "openImageSearcher"));
		searcherMenu.addSeparator();
		searcherMenu.add(this.createMenuItem("cut off", "setCutOff"));
		searcherMenu.add(this.createMenuItem("search", "searchImages"));
		searcherMenu.addSeparator();
		searcherMenu.add(this.createMenuItem("update", "updateImageSearcher"));
		rootMenu.add(searcherMenu);
		//
		StMenu menu = (StMenu) rootMenu.getItem("Menu");
		PopupMenu popupMenu = StMenuViewForAwt.CreatePopupMenu(menu);
		return popupMenu;
	}
	/**
	 * Answer my menu button.
	 * 
	 * @return sra.jun.goodies.button.JunButtonModel
	 */
	public IsrButtonModel menuButton() {
		Image buttonImage = null;
		final IsrImageSearcher self = this;
		StBlockClosure block = new StBlockClosure() {
			public Object value_value_(Object o1, Object o2) {
				IsrButtonModel model = (IsrButtonModel) o1;
				MouseEvent event = (MouseEvent) o2;
				self.menuButtonAction(event);
				return null;
			}
		};
		if ((this.imageFolder() == null) || (this.imageFolder().directoryName() == null)) {
			buttonImage = this._menuButtonImage();
		} else {
			String aString = this.imageFolder().directoryName();
			String text;
			int charCount = 24;
			if (aString.length() > charCount) {
				int half = charCount / 2;
				text = aString.substring(0, half - 1);
				text = text + "...";
				text = text + aString.substring(aString.length() - half + 2, aString.length());
			} else {
				text = aString;
			}
			buttonImage = this.label_fore_back_(text, this.foreColor2(), this.backColor2()).image();
		}
		return IsrButtonModel.Value_visual_action_(false, buttonImage, block);
	}
	/**
	 * Set the menu button action.
	 */
	public void menuButtonAction(MouseEvent event) {
		Point point = event.getPoint();
		Component component = event.getComponent();
		if (menu == null) {
			menu = this.menu();
			component.add(menu);
		}
		menu.show(component, point.x, point.y);
	}
	/**
	 * Answer nine colors parameter.
	 * 
	 * @return java.awt.Color[]
	 */
	public Color[] nineColors() {
		return this.nineColorsModel().colors();
	}
	/**
	 * Answer my nine colors button.
	 * 
	 * @return sra.jun.goodies.button.JunButtonModel
	 */
	public IsrButtonModel nineColorsButton() {
		final IsrImageSearcher self = this;
		StBlockClosure block = new StBlockClosure() {
			public Object value_(Object o) {
				self.nineColorsButtonAction();
				return null;
			}
		};
		return IsrButtonModel.Value_visual_action_(false, this._nineColorsButtonImage(), block);
	}
	/**
	 * Set the nine colors button action.
	 */
	public void nineColorsButtonAction() {
		Isr9ColorsModel model = this.nineColorsModel();
		Frame[] frames = model.builder().frames();
		for (int i = 0; i < frames.length; i++) {
			if (frames[i].isShowing()) {
				frames[i].toFront();
			} else {
				frames[i].show();
			}
		}
		if (frames.length == 0) {
			Frame window = model.open();
			window.setTitle("9 Colors");
		}
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return sra.isr.interfaces.nineColors.Isr9ColorsModel
	 */
	public Isr9ColorsModel nineColorsModel() {
		if (nineColorsModel == null) {
			nineColorsModel = Isr9ColorsModel.Colors_(new Color[9]);
		}
		return nineColorsModel;
	}
	/**
	 * Answer my nine colors switch button.
	 * 
	 * @return sra.jun.goodies.button.JunButtonModel
	 */
	public IsrButtonModel nineColorsSwitch() {
		if (nineColorsSwitch == null) {
			final IsrImageSearcher self = this;
			StBlockClosure block = new StBlockClosure() {
				public Object value_(Object o) {
					self.nineColorsSwitchAction();
					return null;
				}
			};
			nineColorsSwitch = IsrButtonModel.Value_visual_action_(false, this.offImage(), block);
		}
		return nineColorsSwitch;
	}
	/**
	 * Set the nine colors switch button action.
	 */
	public void nineColorsSwitchAction() {
		IsrButtonModel nineColorsSwitch = this.nineColorsSwitch();
		nineColorsSwitch.value_(!nineColorsSwitch.value());
		if (nineColorsSwitch.value() == true) {
			nineColorsSwitch.visual_(this.onImage());
		} else {
			nineColorsSwitch.visual_(this.offImage());
		}
	}
	/**
	 * Return off button image.
	 * 
	 * @return java.awt.Image
	 */
	protected Image offImage() {
		if (OffImage == null) {
			String imageString = 
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000";
			String maskString =
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000";
			OffImage = _DecodeImageString_maskString_fore_back_width_height_(imageString, maskString, this.foreColor(), this.backColor(), 8, 16);
		}
		return OffImage;
	}
	/**
	 * Return on button image.
	 * 
	 * @return java.awt.Image
	 */
	protected Image onImage() {
		if (OnImage == null) {
			String imageString = 
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"01000100"+
			"00101000"+
			"00010000"+
			"00010000"+
			"00101000"+
			"01000100"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000";
			String maskString =
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"00000000"+
			"11101110"+
			"11111110"+
			"11111110"+
			"01111100"+
			"01111100"+
			"11111110"+
			"11111110"+
			"11101110"+
			"00000000"+
			"00000000"+
			"00000000";
			OnImage = _DecodeImageString_maskString_fore_back_width_height_(imageString, maskString, this.foreColor(), this.backColor(), 8, 16);
		}
		return OnImage;
	}
	/**
	 * Open image browser with imageObjects.
	 *
	 * @param imageObjects sra.isr.image.object.IsrImageObject[]
	 */
	public void openImageBrowser_(IsrImageObject[] imageObjects) {
		IsrImageBrowser browser = this.imageBrowser();
		browser.imageObjects_(imageObjects);
		Frame[] frames = browser.builder().frames();
		for (int i = 0; i < frames.length; i++) {
			if (frames[i].isShowing()) {
				frames[i].toFront();
			} else {
				frames[i].show();
			}
		}
		if (frames.length == 0) {
			browser.open();
		}
	}
	/**
	 * Open new image searcher from Menu.
	 *
	 * @return sra.isr.image.searcher.IsrImageSearcher or null
	 */
	public IsrImageSearcher openImageSearcher() {
		IsrImageSearcher imageSearcher = IsrImageSearcher.FromUser();
		if (imageSearcher == null) {
			return null;
		}
		imageSearcher.open();
		return imageSearcher;
	}
	/**
	 * Open new image browser with imageObjects.
	 *
	 * @param imageObjects sra.isr.image.object.IsrImageObject[]
	 */
	public void openNewImageBrowser_(IsrImageObject[] imageObjects) {
		IsrImageBrowser browser = IsrImageBrowser.ImageObjects_(imageObjects);
		browser.open();
	}
	/**
	 * Open image browser with search images.
	 */
	public void search() {
		if (this.allSwitchOff()) {
			this.openImageBrowser_(this.imageObjects());
			return;
		}
		IsrImageObject[] imageObjects = this.searchAux();
		this.openImageBrowser_(imageObjects);
	}
	/**
	 * Answer search images.
	 *
	 * @return sra.isr.image.object.IsrImageObject[]
	 */
	public IsrImageObject[] searchAux() {
		int i, index, count;
		float similarity;
		Float[] similarities;
		IsrImageObject object;
		float[] collection = new float[this.imageObjects().length];
		for (i = 0; i < collection.length; i++) {
			collection[i] = 0.0f;
		}
		count = 0;
		if (this.typicalColorSwitch().value()) {
			similarities = this.similaritiesTypicalColor_(this.typicalColor());
			if (similarities == null) {
				return null;
			}
			for (i = 0; i < collection.length; i++) {
				collection[i] = collection[i] + similarities[i].floatValue();
			}
			count = count + 1;
		}
		if (this.averageGraySwitch().value()) {
			similarities = this.similaritiesAverageGray_(this.averageGray());
			if (similarities == null) {
				return null;
			}
			for (i = 0; i < collection.length; i++) {
				collection[i] = collection[i] + similarities[i].floatValue();
			}
			count = count + 1;
		}
		if (this.nineColorsSwitch().value()) {
			similarities = this.similaritiesNineColors_(this.nineColors());
			if (similarities == null) {
				return null;
			}
			for (i = 0; i < collection.length; i++) {
				collection[i] = collection[i] + similarities[i].floatValue();
			}
			count = count + 1;
		}
		if (this.iconAreaSwitch().value()) {
			similarities = this.similaritiesIconArea_(this.iconArea());
			if (similarities == null) {
				return null;
			}
			for (i = 0; i < collection.length; i++) {
				collection[i] = collection[i] + similarities[i].floatValue();
			}
			count = count + 1;
		}
		if (this.centerSpectraSwitch().value()) {
			similarities = this.similaritiesCenterSpectra_(this.centerSpectra());
			if (similarities == null) {
				return null;
			}
			for (i = 0; i < collection.length; i++) {
				collection[i] = collection[i] + similarities[i].floatValue();
			}
			count = count + 1;
		}
		if (this.freehandImageSwitch().value()) {
			similarities = this.similaritiesFreehandImage_(this.freehandImage());
			if (similarities == null) {
				return null;
			}
			for (i = 0; i < collection.length; i++) {
				collection[i] = collection[i] + similarities[i].floatValue();
			}
			count = count + 1;
		}
		StBlockClosure sortBlock = new StBlockClosure() {
			public Object value_value_(Object assoc1, Object assoc2) {
				return new Boolean(((Float) ((StAssociation) assoc1).key()).floatValue() >= ((Float) ((StAssociation) assoc2).key()).floatValue());
			}
		};
		StSortedCollection similarityCollection = new StSortedCollection(sortBlock);
		for (index = 0; index < collection.length; index++) {
			similarityCollection.add_(new StAssociation(new Float(collection[index] / count), new Integer(index)));
		}
		Object[] sortCollection = similarityCollection._asArray();
		Vector imageObjectVector = new Vector(this.imageObjects().length);
		float cutoff = (float) this.cutOff();
		for (i = 0; i < sortCollection.length; i++) {
			StAssociation assoc = (StAssociation) sortCollection[i];
			index = i;
			similarity = ((Float) assoc.key()).floatValue();
			object = this.imageObjects()[ ((Integer) assoc.value()).intValue()];
			if (similarity >= cutoff) {
				imageObjectVector.addElement(object);
				if (imageObjectVector.size() < 10) {
					if (imageObjectVector.size() < 2) {
						System.out.println();
					}
					System.out.print(String.valueOf(index) + ": ");
					System.out.print(((int) (similarity / 0.01)) * 0.01);
					if (imageObjectVector.size() % 3 == 0) {
						System.out.println();
					} else {
						System.out.print("  ");
					}
					System.out.flush();
				}
			}
		}
		IsrImageObject[] results = new IsrImageObject[imageObjectVector.size()];
		imageObjectVector.copyInto(results);
		return results;
	}
	/**
	 * Images search from Menu.
	 */
	public void searchImages() {
		this.imageSearchButtonAction();
	}
	/**
	 * Set cut off from Menu.
	 */
	public void setCutOff() {
		this.cutOffButtonAction();
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return Float[]
	 * @param grayValue java.awt.Color
	 */
	public Float[] similaritiesAverageGray_(Color grayValue) {
		final int size = this.imageObjects().length;
		final IsrImageSearcher self = this;
		final Vector collection = new Vector(size);
		final Color gray = grayValue;
		IsrProgress aProgress = new IsrProgress();
		aProgress.do_(new StBlockClosure() {
			public Object value_(Object anObject) {
				IsrProgress progress = (IsrProgress) anObject;
				int count = 1;
				progress.message_("average luminance ... ");
				for (int i = 0; i < self.imageObjects().length; i++) {
					IsrImageObject each = self.imageObjects()[i];
					float value = ((float) count) / size;
					progress.value_(value);
					float similarity = self.similarityAverageGray_with_(each, gray);
					collection.addElement(new Float(similarity));
					count = count + 1;
				}
				return null;
			}
		});
		Float[] results = new Float[collection.size()];
		collection.copyInto((Object[]) results);
		return results;
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return Float[]
	 * @param aPoint sra.isr.geometry.Isr2dPoint
	 */
	public Float[] similaritiesCenterSpectra_(Isr2dPoint aPoint) {
		final int size = this.imageObjects().length;
		final IsrImageSearcher self = this;
		final Vector collection = new Vector(size);
		final Isr2dPoint point = aPoint;
		IsrProgress aProgress = new IsrProgress();
		aProgress.do_(new StBlockClosure() {
			public Object value_(Object anObject) {
				IsrProgress progress = (IsrProgress) anObject;
				int count = 1;
				progress.message_("center spectra ... ");
				for (int i = 0; i < self.imageObjects().length; i++) {
					IsrImageObject each = self.imageObjects()[i];
					float value = ((float) count) / size;
					progress.value_(value);
					float similarity = self.similarityCenterSpectra_with_(each, point);
					collection.addElement(new Float(similarity));
					count = count + 1;
				}
				return null;
			}
		});
		Float[] results = new Float[collection.size()];
		collection.copyInto((Object[]) results);
		return results;
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return Float[]
	 * @param anImage sra.smalltalk.StImage
	 */
	public Float[] similaritiesFreehandImage_(StImage anImage) {
		final int size = this.imageObjects().length;
		final IsrImageSearcher self = this;
		final Vector collection = new Vector(size);
		final StImage image = anImage;
		IsrProgress aProgress = new IsrProgress();
		aProgress.do_(new StBlockClosure() {
			public Object value_(Object anObject) {
				IsrProgress progress = (IsrProgress) anObject;
				int count = 1;
				progress.message_("freehand image ... ");
				IsrImageObject imageAux = IsrImageAuxiliary.OriginalImage_(image);
				Color typicalColor = imageAux.typicalColor();
				Color[] nineColors = imageAux.nineColors();
				StImage xSpectrum = imageAux.xSpectrum();
				StImage ySpectrum = imageAux.ySpectrum();
				StImage renderingImage = imageAux.renderingImage();
				StImage binaryImage = imageAux.binaryImage();
				for (int i = 0; i < self.imageObjects().length; i++) {
					IsrImageObject each = self.imageObjects()[i];
					float value = ((float) count) / size;
					progress.value_(value);
					float similarity = self.similarityFreehandImage_typicalColor_nineColor_xSpectrum_ySpectrum_renderingImage_binaryImage_(each, typicalColor, nineColors, xSpectrum, ySpectrum, renderingImage, binaryImage);
					collection.addElement(new Float(similarity));
					count = count + 1;
				}
				return null;
			}
		});
		Float[] results = new Float[collection.size()];
		collection.copyInto((Object[]) results);
		return results;
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return Float[]
	 * @param aRectangle java.awt.Rectangle
	 */
	public Float[] similaritiesIconArea_(Rectangle aRectangle) {
		final int size = this.imageObjects().length;
		final IsrImageSearcher self = this;
		final Vector collection = new Vector(size);
		final Rectangle rectangle = aRectangle;
		IsrProgress aProgress = new IsrProgress();
		aProgress.do_(new StBlockClosure() {
			public Object value_(Object anObject) {
				IsrProgress progress = (IsrProgress) anObject;
				int count = 1;
				progress.message_("image area ... ");
				StImage image = new StImage(rectangle.width, rectangle.height);
				Vector array = IsrImageProcessor.Icon3_size_(image, IsrImageObject.DefaultSize());
				Rectangle box = (Rectangle) array.lastElement();
				for (int i = 0; i < self.imageObjects().length; i++) {
					IsrImageObject each = self.imageObjects()[i];
					float value = ((float) count) / size;
					progress.value_(value);
					float similarity = self.similarityIconArea_with_(each, box);
					collection.addElement(new Float(similarity));
					count = count + 1;
				}
				return null;
			}
		});
		Float[] results = new Float[collection.size()];
		collection.copyInto((Object[]) results);
		return results;
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return Float[]
	 * @param nineColors java.awt.Color[]
	 */
	public Float[] similaritiesNineColors_(Color[] nineColors) {
		final int size = this.imageObjects().length;
		final IsrImageSearcher self = this;
		final Vector collection = new Vector(size);
		final Color[] colors = nineColors;
		IsrProgress aProgress = new IsrProgress();
		aProgress.do_(new StBlockClosure() {
			public Object value_(Object anObject) {
				IsrProgress progress = (IsrProgress) anObject;
				int count = 1;
				progress.message_("nine colors ... ");
				for (int i = 0; i < self.imageObjects().length; i++) {
					IsrImageObject each = self.imageObjects()[i];
					float value = ((float) count) / size;
					progress.value_(value);
					float similarity = self.similarityNineColors_with_(each, colors);
					collection.addElement(new Float(similarity));
					count = count + 1;
				}
				return null;
			}
		});
		Float[] results = new Float[collection.size()];
		collection.copyInto((Object[]) results);
		return results;
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return Float[]
	 * @param nineColors java.awt.Color
	 */
	public Float[] similaritiesTypicalColor_(Color colorValue) {
		final int size = this.imageObjects().length;
		final IsrImageSearcher self = this;
		final Vector collection = new Vector(size);
		final Color color = colorValue;
		IsrProgress aProgress = new IsrProgress();
		aProgress.do_(new StBlockClosure() {
			public Object value_(Object anObject) {
				IsrProgress progress = (IsrProgress) anObject;
				int count = 1;
				progress.message_("typical color ... ");
				for (int i = 0; i < self.imageObjects().length; i++) {
					IsrImageObject each = self.imageObjects()[i];
					float value = ((float) count) / size;
					progress.value_(value);
					float similarity = self.similarityTypicalColor_with_(each, color);
					collection.addElement(new Float(similarity));
					count = count + 1;
				}
				return null;
			}
		});
		Float[] results = new Float[collection.size()];
		collection.copyInto((Object[]) results);
		return results;
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 *
	 * @return float
	 * @param imageObject sra.isr.image.objects.IsrImageObject
	 * @param grayValue java.awt.Color
	 */
	public float similarityAverageGray_with_(IsrImageObject imageObject, Color grayValue) {
		double luminance1 = StColorValue._GetLuminance(grayValue);
		double luminance2 = StColorValue._GetLuminance(imageObject.averageGray());
		double luminance = Math.max(luminance1, luminance2) - Math.min(luminance1, luminance2);
		float similarity = 1.0f - (float) luminance;
		similarity = Math.max(0.0f, Math.min(similarity, 1.0f));
		return similarity;
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 *
	 * @return float
	 * @param imageObject sra.isr.image.objects.IsrImageObject
	 * @param aPoint sra.isr.geometry.basic.Isr2dPoint
	 */
	public float similarityCenterSpectra_with_(IsrImageObject imageObject, Isr2dPoint aPoint) {
		Isr2dPoint point = imageObject.centerSpectra();
		double x = Math.max(aPoint.x(), point.x()) - Math.min(aPoint.x(), point.x());
		double y = Math.max(aPoint.y(), point.y()) - Math.min(aPoint.y(), point.y());
		float similarity = 1.0f - (float) (x + y);
		similarity = Math.max(0.0f, Math.min(similarity, 1.0f));
		return similarity;
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 *
	 * @return float
	 * @param imageObject sra.isr.image.objects.IsrImageObject
	 * @param grayValue java.awt.Color
	 */
	public float similarityFreehandImage_typicalColor_nineColor_xSpectrum_ySpectrum_renderingImage_binaryImage_(IsrImageObject imageObject, Color typicalColor, Color[] nineColors, Object[] xSpectrum, Object[] ySpectrum, StImage renderingImage, StImage binaryImage) {
		double luminance1 = StColorValue._GetLuminance(typicalColor);
		double luminance2 = StColorValue._GetLuminance(imageObject.averageGray());
		double luminance = Math.max(luminance1, luminance2) - Math.min(luminance1, luminance2);
		float similarity = 1.0f - (float) luminance;
		similarity = Math.max(0.0f, Math.min(similarity, 1.0f));
		return similarity;
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 *
	 * @return float
	 * @param imageObject sra.isr.image.objects.IsrImageObject
	 * @param typicalColor java.awt.Color
	 * @param nineColors java.awt.Color[]
	 * @param xSpectrum sra.smalltalk.StImage
	 * @param ySpectrum sra.smalltalk.StImage
	 * @param renderingImage sra.smalltalk.StImage
	 * @param binaryImage sra.smalltalk.StImage
	 */
	public float similarityFreehandImage_typicalColor_nineColor_xSpectrum_ySpectrum_renderingImage_binaryImage_(IsrImageObject imageObject, Color typicalColor, Color[] nineColors, StImage xSpectrum, StImage ySpectrum, StImage renderingImage, StImage binaryImage) {
		float similarity;
		float sum = 0.0f;
		int count = 0;
		similarity = this.similarityTypicalColor_with_(imageObject, typicalColor);
		sum = sum + similarity;
		count = count + 1;
		similarity = this.similarityNineColors_with_(imageObject, nineColors);
		sum = sum + similarity;
		count = count + 1;
		similarity = IsrImageProcessor.Similarity_with_(imageObject.xSpectrum(), xSpectrum);
		sum = sum + similarity;
		count = count + 1;
		similarity = IsrImageProcessor.Similarity_with_(imageObject.ySpectrum(), ySpectrum);
		sum = sum + similarity;
		count = count + 1;
		similarity = IsrImageProcessor.Similarity_with_(imageObject.renderingImage(), renderingImage);
		sum = sum + similarity;
		count = count + 1;
		similarity = IsrImageProcessor.Similarity_with_(imageObject.binaryImage(), binaryImage);
		sum = sum + similarity;
		count = count + 1;
		similarity = sum / count;
		similarity = Math.max(0.0f, Math.min(similarity, 1.0f));
		return similarity;
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 *
	 * @return float
	 * @param imageObject sra.isr.image.objects.IsrImageObject
	 * @param iconArea java.awt.Rectangle
	 */
	public float similarityIconArea_with_(IsrImageObject imageObject, Rectangle iconArea) {
		Rectangle area = imageObject.iconArea();
		double w = ((double) Math.min(iconArea.width, area.width)) / Math.max(iconArea.width, area.width);
		double h = ((double) Math.min(iconArea.height, area.height)) / Math.max(iconArea.height, area.height);
		float similarity = (float) ((w + h) / 2);
		similarity = Math.max(0.0f, Math.min(similarity, 1.0f));
		return similarity;
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 *
	 * @return float
	 * @param imageObject sra.isr.image.objects.IsrImageObject
	 * @param nineColors java.awt.Color[]
	 */
	public float similarityNineColors_with_(IsrImageObject imageObject, Color[] nineColors) {
		int index;
		Vector indexes = new Vector();
		Vector colors1 = new Vector();
		for (index = 0; index < nineColors.length; index++) {
			if (nineColors[index] != null) {
				indexes.addElement(new Integer(index));
				colors1.addElement(nineColors[index]);
			}
		}
		if (indexes.size() == 0) {
			return -0.0f;
		}
		Vector colors2 = new Vector(colors1.size());
		Enumeration enum = indexes.elements();
		while (enum.hasMoreElements()) {
			index = ((Integer) enum.nextElement()).intValue();
			colors2.addElement(imageObject.nineColors()[index]);
		}
		float sum = 0.0f;
		for (index = 0; index < colors2.size(); index++) {
			Color c1 = (Color) colors1.elementAt(index);
			Color c2 = (Color) colors2.elementAt(index);
			float r = (Math.max(c1.getRed(), c2.getRed()) - Math.min(c1.getRed(), c2.getRed())) / 255.0f;
			float g = (Math.max(c1.getGreen(), c2.getGreen()) - Math.min(c1.getGreen(), c2.getGreen())) / 255.0f;
			float b = (Math.max(c1.getBlue(), c2.getBlue()) - Math.min(c1.getBlue(), c2.getBlue())) / 255.0f;
			sum = sum + ((r + g + b) / 3.0f);
		}
		float similarity = 1.0f - (sum / indexes.size());
		similarity = Math.max(0.0f, Math.min(similarity, 1.0f));
		return similarity;
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 *
	 * @return float
	 * @param imageObject sra.isr.image.objects.IsrImageObject
	 * @param colorValue java.awt.Color
	 */
	public float similarityTypicalColor_with_(IsrImageObject imageObject, Color colorValue) {
		Color color = imageObject.typicalColor();
		float r = (Math.max(colorValue.getRed(), color.getRed()) - Math.min(colorValue.getRed(), color.getRed())) / 255.0f;
		float g = (Math.max(colorValue.getGreen(), color.getGreen()) - Math.min(colorValue.getGreen(), color.getGreen())) / 255.0f;
		float b = (Math.max(colorValue.getBlue(), color.getBlue()) - Math.min(colorValue.getBlue(), color.getBlue())) / 255.0f;
		float similarity = 1.0f - ((r + g + b) / 3.0f);
		similarity = Math.max(0.0f, Math.min(similarity, 1.0f));
		return similarity;
	}
	/**
	 * Answer typical color parameter.
	 * 
	 * @return java.awt.Color
	 */
	public Color typicalColor() {
		return this.typicalColorModel().color();
	}
	/**
	 * Answer my typical color button.
	 * 
	 * @return sra.jun.goodies.button.JunButtonModel
	 */
	public IsrButtonModel typicalColorButton() {
		final IsrImageSearcher self = this;
		StBlockClosure block = new StBlockClosure() {
			public Object value_(Object o) {
				self.typicalColorButtonAction();
				return null;
			}
		};
		return IsrButtonModel.Value_visual_action_(false, this._typicalColorButtonImage(), block);
	}
	/**
	 * Set the typical color button action.
	 */
	public void typicalColorButtonAction() {
		IsrColorChoiceSBH model = this.typicalColorModel();
		Frame[] frames = model.builder().frames();
		for (int i = 0; i < frames.length; i++) {
			if (frames[i].isShowing()) {
				frames[i].toFront();
			} else {
				frames[i].show();
			}
		}
		if (frames.length == 0) {
			Frame window = model.open();
			window.setTitle("Color");
		}
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return sra.isr.interface.colors.IsrColorChoiceSBH
	 */
	public IsrColorChoiceSBH typicalColorModel() {
		if (typicalColorModel == null) {
			typicalColorModel = new IsrColorChoiceSBH();
		}
		return typicalColorModel;
	}
	/**
	 * Answer my typical color switch button.
	 * 
	 * @return sra.isr.interfaces.button.IsrButtonModel
	 */
	public IsrButtonModel typicalColorSwitch() {
		if (typicalColorSwitch == null) {
			final IsrImageSearcher self = this;
			StBlockClosure block = new StBlockClosure() {
				public Object value_(Object o) {
					self.typicalColorSwitchAction();
					return null;
				}
			};
			typicalColorSwitch = IsrButtonModel.Value_visual_action_(false, this.offImage(), block);
		}
		return typicalColorSwitch;
	}
	/**
	 * Set the typical color switch button action.
	 */
	public void typicalColorSwitchAction() {
		IsrButtonModel typicalColorSwitch = this.typicalColorSwitch();
		typicalColorSwitch.value_(!typicalColorSwitch.value());
		if (typicalColorSwitch.value() == true) {
			typicalColorSwitch.visual_(this.onImage());
		} else {
			typicalColorSwitch.visual_(this.offImage());
		}
	}
	/**
	 * Update this image searcher from Menu.
	 *
	 * @return sra.isr.image.searcher.IsrImageSearcher or null
	 */
	public IsrImageSearcher updateImageSearcher() {
		if (!StDialog.Confirm_("Do you want to update?")) {
			return null;
		}
		this.imageFolder().updateImageObjects();
		this.imageBrowser().imageObjects_(this.imageObjects());
		return this;
	}
	/**
	 * Invoked when a window is in the process of being closed.
	 * The close operation can be overridden at this point.
	 */
	public void windowClosing(java.awt.event.WindowEvent e) {
		Window aWindow = e.getWindow();
		Point location = aWindow.getLocationOnScreen();
		Dimension size = aWindow.getSize();
		location = new Point(location.x + size.width, location.y);
		if (StDialog.Confirm_("Really close?", false, location) == true) {
			StApplicationModel[] models = new StApplicationModel[8];
			models[0] = this.typicalColorModel();
			models[1] = this.averageGrayModel();
			models[2] = this.nineColorsModel();
			models[3] = this.iconAreaModel();
			models[4] = this.centerSpectraModel();
			models[5] = this.freehandImageModel();
			models[6] = this.cutOffModel();
			models[7] = this.imageBrowser();
			for (int i = 0; i < models.length; i++) {
				StApplicationModel model = models[i];
				Frame[] frames = model.builder().frames();
				for (int j = 0; j < frames.length; j++) {
					frames[j].setVisible(false);
				}
			}
			super.windowClosing(e);
		}
	}
	/**
	 * Answer a window title.
	 * @return java.lang.String
	 */
	protected String windowTitle() {
		return "Image Searcher";
	}
}
