package sra.isr.image.support;

import java.awt.*;
import java.awt.image.*;
import java.util.*;
import java.io.*;
import sra.smalltalk.*;
import sra.isr.image.streams.*;
import sra.isr.geometry.basic.*;

/**
 * IsrImageProcessor class
 * 
 * 	@author:        Hirotsugu Kondo
 * 	@created:       1998/11/16 (by Hirotsugu Kondo)
 * 	@updated:       2000/02/16 (by Mitsuhiro Asada)
 * 	@ImageSearcher: 019
 * 	@JDK:           1.1.6 or higher
 * 	@copyright:     2000 SRA (Software Research Associates, Inc.)
 * 
 * 	$Id: IsrImageProcessor.java,v 1.16 2000/04/07 04:24:17 m-asada Exp $
 */
public class IsrImageProcessor {
	private static int[] Stipples;
	private static StImage[] Masks;
	private static IndexColorModel ColorPalette256;
	private static IndexColorModel GrayPalette256;
	private static IndexColorModel _WhiteBlackPalette;
	private static IndexColorModel _BlackWhitePalette;
	private static Point[] EightPoints;
	private static Point[] FourPoints;
	static {
		Stipples = _InitializeStipples();
		Masks = _InitializeMasks();
	}

	/**
	 * Answer the ColorModel for Black and White.
	 *
	 * @return java.awt.image.IndexColorModel
	 */
	public final static IndexColorModel _BlackWhitePalette() {
		if (_BlackWhitePalette == null) {
			byte[] palette = {(byte) 0, (byte) 255};
			_BlackWhitePalette = new IndexColorModel(1, 2, palette, palette, palette);
		}
		return _BlackWhitePalette;
	}
	/**
	 * 
	 * @return StImage[]
	 */
	private static StImage[] _InitializeMasks() {
		int[] map = new int[64];
		map[0] = 1;
		map[1] = 33;
		map[2] = 9;
		map[3] = 41;
		map[4] = 3;
		map[5] = 35;
		map[6] = 11;
		map[7] = 43;
		map[8] = 59;
		map[9] = 17;
		map[10] = 49;
		map[11] = 25;
		map[12] = 57;
		map[13] = 19;
		map[14] = 51;
		map[15] = 27;
		map[16] = 13;
		map[17] = 45;
		map[18] = 5;
		map[19] = 37;
		map[20] = 15;
		map[21] = 47;
		map[22] = 7;
		map[23] = 39;
		map[24] = 55;
		map[25] = 29;
		map[26] = 61;
		map[27] = 21;
		map[28] = 53;
		map[29] = 31;
		map[30] = 63;
		map[31] = 23;
		map[32] = 4;
		map[33] = 36;
		map[34] = 12;
		map[35] = 44;
		map[36] = 2;
		map[37] = 34;
		map[38] = 10;
		map[39] = 42;
		map[40] = 58;
		map[41] = 20;
		map[42] = 52;
		map[43] = 28;
		map[44] = 60;
		map[45] = 18;
		map[46] = 50;
		map[47] = 26;
		map[48] = 16;
		map[49] = 48;
		map[50] = 8;
		map[51] = 40;
		map[52] = 14;
		map[53] = 46;
		map[54] = 6;
		map[55] = 38;
		map[56] = 54;
		map[57] = 32;
		map[58] = 64;
		map[59] = 24;
		map[60] = 56;
		map[61] = 30;
		map[62] = 62;
		map[63] = 22;
		int pixelValue = Color.black.getRGB();
		StImage[] masks = new StImage[65];
		for (int index = 0; index < 65; index++) {
			StImage mask = new StImage(16, 16);
			if (index != 0) {
				int count = 0;
				for (int eachIndex = 0; eachIndex < map.length; eachIndex++) {
					int each = map[eachIndex];
					if (each <= index) {
						int x = count % 8;
						int y = count / 8;
						mask.atX_y_put_(x, y, pixelValue);
						mask.atX_y_put_(x + 8, y, pixelValue);
						mask.atX_y_put_(x, y + 8, pixelValue);
						mask.atX_y_put_(x + 8, y + 8, pixelValue);
					};
					count = count + 1;
				};
			};
			masks[index] = mask;
		};
		return masks;
	}
	/**
	 * 
	 * @return int[]
	 */
	private static int[] _InitializeStipples() {
		int[] stipples = new int[9];
		stipples[0] = Integer.parseInt("0000",16);
		stipples[1] = Integer.parseInt("8080",16);
		stipples[2] = Integer.parseInt("8888",16);
		stipples[3] = Integer.parseInt("A8A8",16);
		stipples[4] = Integer.parseInt("AAAA",16);
		stipples[5] = Integer.parseInt("EAEA",16);
		stipples[6] = Integer.parseInt("EEEE",16);
		stipples[7] = Integer.parseInt("FEFE",16);
		stipples[8] = Integer.parseInt("FFFF",16);
		return stipples;
	}
	/*
	 * @return sra.smalltalk.StImage
	 * @param figuer sra.smalltalk.StImage
	 * @param shape sra.smalltalk.StImage
	 */
	public static StImage _MakeImage_Shape_(StImage figure, StImage shape) {
		if (figure.extent().equals(shape.extent())) {
			ColorModel colorModel = ColorModel.getRGBdefault();
			int width = figure.width();
			int height = figure.height();
			int[] pixels = new int[width * height];
			int figurePixelValue, shapePixelValue, red, green, blue, alpha;
			for (int i = 0; i < width * height; i++) {
				figurePixelValue = figure.bits()[i];
				shapePixelValue = shape.bits()[i];
				red = colorModel.getRed(figurePixelValue);
				green = colorModel.getGreen(figurePixelValue);
				blue = colorModel.getBlue(figurePixelValue);
				alpha = colorModel.getAlpha(shapePixelValue);
				if (figurePixelValue != 0x00FFFFFF) {
					pixels[i] = alpha << 24 | red << 16 | green << 8 | blue << 0;
				} else
					if (shapePixelValue != 0x00FFFFFF) {
						pixels[i] = shapePixelValue;
					} else {
						pixels[i] = 0x00FFFFFF;
					}
			}
			int offset = 0;
			int scansize = width;
			return new StImage(width, height, pixels);
		}
		return null;
	}
	/**
	 * Answer the ColorModel for White and Black.
	 *
	 * @return java.awt.image.IndexColorModel
	 */
	public final static IndexColorModel _WhiteBlackPalette() {
		if (_WhiteBlackPalette == null) {
			byte[] palette = {(byte) 255, (byte) 0};
			_WhiteBlackPalette = new IndexColorModel(1, 2, palette, palette, palette);
		}
		return _WhiteBlackPalette;
	}
	/**
	 * Answer border following on anImage.
	 * 
	 * @return sra.smalltalk.StAssociation[]
	 * @param anImage sra.smalltalk.StImage
	 */
	public static StAssociation[] BorderFollowing_(StImage anImage) {
		int b1, b2, b3, b4, direction;
		StImage temp = anImage.convertToPalette_(_WhiteBlackPalette());
		StImage image = new StImage(temp.width() + 2, temp.height() + 2);
		image.copy_from_in_rule_(new Rectangle(1, 1, temp.width(), temp.height()), new Point(0, 0), temp, StImage.Over);
		Vector borders = new Vector();
		int colorBlackValue = Color.black.getRGB();
		int colorWhiteValue = Color.white.getRGB();
		for (int i = 0; i < image.width(); i++) {
			for (int j = 0; j < image.height(); j++) {
				if (image.atX_y_(i, j) == colorBlackValue) {
					b1 = image.atX_y_(i, j + 1);
					b2 = image.atX_y_(i - 1, j);
					b3 = image.atX_y_(i, j - 1);
					b4 = image.atX_y_(i + 1, j);
					if ((b1 | b2 | b3 | b4) == colorWhiteValue) {
						if (b4 == colorWhiteValue) {
							direction = 6; // in

						} else
							if (b3 == colorWhiteValue) {
								direction = 4; // out
							} else
								if (b2 == colorWhiteValue) {
									direction = 2; // out

								} else
									if (b1 == colorWhiteValue) {
										direction = 0; // in

									} else {
										direction = -1; // null

									}
						if (direction > 0) {
							Point[] collection = BorderFollowingPoint_direction_image_(new Point(i, j), direction, image);
							if (collection.length > 0) {
								borders.addElement(new StAssociation(new Integer(direction), collection));
							}
						}
					}
				}
			}
		}
		StAssociation[] answer = new StAssociation[borders.size()];
		borders.copyInto(answer);
		return answer;
	}
	/**
	 * @return java.awt.Point[]
	 * @param anImage sra.smalltalk.StImage
	 */
	public static Point[] BorderFollowingPoint_direction_image_(Point point, int ds, StImage image) {
		int i, iw, is, j, jw, js, d, n, nb, nc;
		int[] array1 = {0, -1, -1, -1, 0, 1, 1, 1};
		int[] array2 = {1, 1, 0, -1, -1, -1, 0, 1};
		int colorGrayValue = Color.gray.getRGB();
		int colorWhiteValue = Color.white.getRGB();
		boolean loop1, loop2;
		Point p;
		Vector border = new Vector();
		i = point.x;
		iw = point.x;
		is = point.x;
		j = point.y;
		jw = point.y;
		js = point.y;
		d = ds;
		n = (d + 4) & 7;
		d = -1;
		nb = -1;
		nc = 0;
		nc = nc + (image.atX_y_(i - 1, j - 1)) + 1;
		nc = nc + (image.atX_y_(i - 1, j)) + 1;
		nc = nc + (image.atX_y_(i - 1, j + 1)) + 1;
		nc = nc + (image.atX_y_(i, j - 1)) + 1;
		nc = nc + (image.atX_y_(i, j + 1)) + 1;
		nc = nc + (image.atX_y_(i + 1, j - 1)) + 1;
		nc = nc + (image.atX_y_(i + 1, j)) + 1;
		nc = nc + (image.atX_y_(i + 1, j + 1)) + 1;
		if (nc < 0) {
			loop1 = true;
			while (loop1) {
				i = iw;
				j = jw;
				image.atX_y_put_(iw, jw, colorGrayValue);
				p = new Point(iw - 1, jw - 1);
				border.addElement(p);
				n = (n + 4) & 7;
				loop2 = true;
				while (loop2) {
					n = (n + 1) & 7;
					iw = i + array1[n];
					jw = j + array2[n];
					loop2 = (image.atX_y_(iw, jw) == colorWhiteValue);
				}
				if (d < 0) {
					if (nb < 0) {
						nb = n;
					} else {
						d = nb;
					}
				}
				loop1 = ((i != is) | (j != js) | (n != d));
			}
		} else {
			image.atX_y_put_(i, j, colorGrayValue);
		}
		Point[] answer = new Point[border.size()];
		border.copyInto(answer);
		return answer;
	}
	/**
	 * @return Object[]
	 * @param sra,smalltalk.StImage
	 */
	public static Object[] ColorHistogram_(StImage anImage) {
		Hashtable dictionary = new Hashtable();
		for (int y = 0; y < anImage.height(); y++) {
			for (int x = 0; x < anImage.width(); x++) {
				Integer index = new Integer(anImage.atX_y_(x, y) & 0x00FFFFFF);
				if (dictionary.containsKey(index)) {
					dictionary.put(index, new Integer(1 + ((Integer) dictionary.get(index)).intValue()));
				} else {
					dictionary.put(index, new Integer(1));
				}
			}
		}
		StBlockClosure sortBlock = new StBlockClosure() {
			public Object value_value_(Object x, Object y) {
				Color cx = (Color) ((StAssociation) x).key();
				double dx = StColorValue._DistanceFrom(cx, Color.black);
				Color cy = (Color) ((StAssociation) y).key();
				double dy = StColorValue._DistanceFrom(cy, Color.black);
				if (cx == cy) {
					return new Boolean((cx.getRed() < cy.getRed()) && (cx.getGreen() < cy.getGreen()) && (cx.getBlue() < cy.getBlue()));
				} else {
					return new Boolean(dx < dy);
				}
			}
		};
		StSortedCollection histogram = new StSortedCollection(sortBlock);
		Enumeration enum = dictionary.keys();
		while (enum.hasMoreElements()) {
			Integer assocKey = (Integer) enum.nextElement();
			Integer assocValue = (Integer) dictionary.get(assocKey);
			Color color = new Color(assocKey.intValue());
			double value = assocValue.doubleValue() / (anImage.width() * anImage.height());
			StAssociation pair = new StAssociation(color, new Double(value));
			histogram.add_(pair);
		}
		return histogram._asArray();
	}
	/**
	 * Answer the ColorModel for 256 colors.
	 * 
	 * @return java.awt.image.IndexColorModel
	 */
	public final static IndexColorModel ColorPalette256() {
		if (ColorPalette256 == null) {
			Color[] colors = new Color[256];
			colors[0] = new Color(0, 0, 0);
			colors[1] = new Color(16, 16, 16);
			colors[2] = new Color(33, 33, 33);
			colors[3] = new Color(67, 67, 67);
			colors[4] = new Color(84, 84, 84);
			colors[5] = new Color(118, 118, 118);
			colors[6] = new Color(136, 136, 136);
			colors[7] = new Color(170, 170, 170);
			colors[8] = new Color(187, 187, 187);
			colors[9] = new Color(221, 221, 221);
			colors[10] = new Color(238, 238, 238);
			colors[11] = new Color(0, 0, 16);
			colors[12] = new Color(0, 0, 33);
			colors[13] = new Color(0, 0, 67);
			colors[14] = new Color(0, 0, 84);
			colors[15] = new Color(0, 0, 118);
			colors[16] = new Color(0, 0, 136);
			colors[17] = new Color(0, 0, 170);
			colors[18] = new Color(0, 0, 187);
			colors[19] = new Color(0, 0, 221);
			colors[20] = new Color(0, 0, 238);
			colors[21] = new Color(0, 16, 0);
			colors[22] = new Color(0, 33, 0);
			colors[23] = new Color(0, 67, 0);
			colors[24] = new Color(0, 84, 0);
			colors[25] = new Color(0, 118, 0);
			colors[26] = new Color(0, 136, 0);
			colors[27] = new Color(0, 170, 0);
			colors[28] = new Color(0, 187, 0);
			colors[29] = new Color(0, 221, 0);
			colors[30] = new Color(0, 238, 0);
			colors[31] = new Color(16, 0, 0);
			colors[32] = new Color(33, 0, 0);
			colors[33] = new Color(67, 0, 0);
			colors[34] = new Color(84, 0, 0);
			colors[35] = new Color(118, 0, 0);
			colors[36] = new Color(136, 0, 0);
			colors[37] = new Color(170, 0, 0);
			colors[38] = new Color(187, 0, 0);
			colors[39] = new Color(221, 0, 0);
			colors[40] = new Color(238, 0, 0);
			colors[41] = new Color(0, 0, 50);
			colors[42] = new Color(0, 0, 101);
			colors[43] = new Color(0, 0, 153);
			colors[44] = new Color(0, 0, 204);
			colors[45] = new Color(0, 0, 255);
			colors[46] = new Color(0, 50, 0);
			colors[47] = new Color(0, 50, 50);
			colors[48] = new Color(0, 50, 101);
			colors[49] = new Color(0, 50, 153);
			colors[50] = new Color(0, 50, 204);
			colors[51] = new Color(0, 50, 255);
			colors[52] = new Color(0, 101, 0);
			colors[53] = new Color(0, 101, 50);
			colors[54] = new Color(0, 101, 101);
			colors[55] = new Color(0, 101, 153);
			colors[56] = new Color(0, 101, 204);
			colors[57] = new Color(0, 101, 255);
			colors[58] = new Color(0, 153, 0);
			colors[59] = new Color(0, 153, 50);
			colors[60] = new Color(0, 153, 101);
			colors[61] = new Color(0, 153, 153);
			colors[62] = new Color(0, 153, 204);
			colors[63] = new Color(0, 153, 255);
			colors[64] = new Color(0, 204, 0);
			colors[65] = new Color(0, 204, 50);
			colors[66] = new Color(0, 204, 101);
			colors[67] = new Color(0, 204, 153);
			colors[68] = new Color(0, 204, 204);
			colors[69] = new Color(0, 204, 255);
			colors[70] = new Color(0, 255, 0);
			colors[71] = new Color(0, 255, 50);
			colors[72] = new Color(0, 255, 101);
			colors[73] = new Color(0, 255, 153);
			colors[74] = new Color(0, 255, 204);
			colors[75] = new Color(0, 255, 255);
			colors[76] = new Color(50, 0, 0);
			colors[77] = new Color(50, 0, 50);
			colors[78] = new Color(50, 0, 101);
			colors[79] = new Color(50, 0, 153);
			colors[80] = new Color(50, 0, 204);
			colors[81] = new Color(50, 0, 255);
			colors[82] = new Color(50, 50, 0);
			colors[83] = new Color(50, 50, 50);
			colors[84] = new Color(50, 50, 101);
			colors[85] = new Color(50, 50, 153);
			colors[86] = new Color(50, 50, 204);
			colors[87] = new Color(50, 50, 255);
			colors[88] = new Color(50, 101, 0);
			colors[89] = new Color(50, 101, 50);
			colors[90] = new Color(50, 101, 101);
			colors[91] = new Color(50, 101, 153);
			colors[92] = new Color(50, 101, 204);
			colors[93] = new Color(50, 101, 255);
			colors[94] = new Color(50, 153, 0);
			colors[95] = new Color(50, 153, 50);
			colors[96] = new Color(50, 153, 101);
			colors[97] = new Color(50, 153, 153);
			colors[98] = new Color(50, 153, 204);
			colors[99] = new Color(50, 153, 255);
			colors[100] = new Color(50, 204, 0);
			colors[101] = new Color(50, 204, 50);
			colors[102] = new Color(50, 204, 101);
			colors[103] = new Color(50, 204, 153);
			colors[104] = new Color(50, 204, 204);
			colors[105] = new Color(50, 204, 255);
			colors[106] = new Color(50, 255, 0);
			colors[107] = new Color(50, 255, 50);
			colors[108] = new Color(50, 255, 101);
			colors[109] = new Color(50, 255, 153);
			colors[110] = new Color(50, 255, 204);
			colors[111] = new Color(50, 255, 255);
			colors[112] = new Color(101, 0, 0);
			colors[113] = new Color(101, 0, 50);
			colors[114] = new Color(101, 0, 101);
			colors[115] = new Color(101, 0, 153);
			colors[116] = new Color(101, 0, 204);
			colors[117] = new Color(101, 0, 255);
			colors[118] = new Color(101, 50, 0);
			colors[119] = new Color(101, 50, 50);
			colors[120] = new Color(101, 50, 101);
			colors[121] = new Color(101, 50, 153);
			colors[122] = new Color(101, 50, 204);
			colors[123] = new Color(101, 50, 255);
			colors[124] = new Color(101, 101, 0);
			colors[125] = new Color(101, 101, 50);
			colors[126] = new Color(101, 101, 101);
			colors[127] = new Color(101, 101, 153);
			colors[128] = new Color(101, 101, 204);
			colors[129] = new Color(101, 101, 255);
			colors[130] = new Color(101, 153, 0);
			colors[131] = new Color(101, 153, 50);
			colors[132] = new Color(101, 153, 101);
			colors[133] = new Color(101, 153, 153);
			colors[134] = new Color(101, 153, 204);
			colors[135] = new Color(101, 153, 255);
			colors[136] = new Color(101, 204, 0);
			colors[137] = new Color(101, 204, 50);
			colors[138] = new Color(101, 204, 101);
			colors[139] = new Color(101, 204, 153);
			colors[140] = new Color(101, 204, 204);
			colors[141] = new Color(101, 204, 255);
			colors[142] = new Color(101, 255, 0);
			colors[143] = new Color(101, 255, 50);
			colors[144] = new Color(101, 255, 101);
			colors[145] = new Color(101, 255, 153);
			colors[146] = new Color(101, 255, 204);
			colors[147] = new Color(101, 255, 255);
			colors[148] = new Color(153, 0, 0);
			colors[149] = new Color(153, 0, 50);
			colors[150] = new Color(153, 0, 101);
			colors[151] = new Color(153, 0, 153);
			colors[152] = new Color(153, 0, 204);
			colors[153] = new Color(153, 0, 255);
			colors[154] = new Color(153, 50, 0);
			colors[155] = new Color(153, 50, 50);
			colors[156] = new Color(153, 50, 101);
			colors[157] = new Color(153, 50, 153);
			colors[158] = new Color(153, 50, 204);
			colors[159] = new Color(153, 50, 255);
			colors[160] = new Color(153, 101, 0);
			colors[161] = new Color(153, 101, 50);
			colors[162] = new Color(153, 101, 101);
			colors[163] = new Color(153, 101, 153);
			colors[164] = new Color(153, 101, 204);
			colors[165] = new Color(153, 101, 255);
			colors[166] = new Color(153, 153, 0);
			colors[167] = new Color(153, 153, 50);
			colors[168] = new Color(153, 153, 101);
			colors[169] = new Color(153, 153, 153);
			colors[170] = new Color(153, 153, 204);
			colors[171] = new Color(153, 153, 255);
			colors[172] = new Color(153, 204, 0);
			colors[173] = new Color(153, 204, 50);
			colors[174] = new Color(153, 204, 101);
			colors[175] = new Color(153, 204, 153);
			colors[176] = new Color(153, 204, 204);
			colors[177] = new Color(153, 204, 255);
			colors[178] = new Color(153, 255, 0);
			colors[179] = new Color(153, 255, 50);
			colors[180] = new Color(153, 255, 101);
			colors[181] = new Color(153, 255, 153);
			colors[182] = new Color(153, 255, 204);
			colors[183] = new Color(153, 255, 255);
			colors[184] = new Color(204, 0, 0);
			colors[185] = new Color(204, 0, 50);
			colors[186] = new Color(204, 0, 101);
			colors[187] = new Color(204, 0, 153);
			colors[188] = new Color(204, 0, 204);
			colors[189] = new Color(204, 0, 255);
			colors[190] = new Color(204, 50, 0);
			colors[191] = new Color(204, 50, 50);
			colors[192] = new Color(204, 50, 101);
			colors[193] = new Color(204, 50, 153);
			colors[194] = new Color(204, 50, 204);
			colors[195] = new Color(204, 50, 255);
			colors[196] = new Color(204, 101, 0);
			colors[197] = new Color(204, 101, 50);
			colors[198] = new Color(204, 101, 101);
			colors[199] = new Color(204, 101, 153);
			colors[200] = new Color(204, 101, 204);
			colors[201] = new Color(204, 101, 255);
			colors[202] = new Color(204, 153, 0);
			colors[203] = new Color(204, 153, 50);
			colors[204] = new Color(204, 153, 101);
			colors[205] = new Color(204, 153, 153);
			colors[206] = new Color(204, 153, 204);
			colors[207] = new Color(204, 153, 255);
			colors[208] = new Color(204, 204, 0);
			colors[209] = new Color(204, 204, 50);
			colors[210] = new Color(204, 204, 101);
			colors[211] = new Color(204, 204, 153);
			colors[212] = new Color(204, 204, 204);
			colors[213] = new Color(204, 204, 255);
			colors[214] = new Color(204, 255, 0);
			colors[215] = new Color(204, 255, 50);
			colors[216] = new Color(204, 255, 101);
			colors[217] = new Color(204, 255, 153);
			colors[218] = new Color(204, 255, 204);
			colors[219] = new Color(204, 255, 255);
			colors[220] = new Color(255, 0, 0);
			colors[221] = new Color(255, 0, 50);
			colors[222] = new Color(255, 0, 101);
			colors[223] = new Color(255, 0, 153);
			colors[224] = new Color(255, 0, 204);
			colors[225] = new Color(255, 0, 255);
			colors[226] = new Color(255, 50, 0);
			colors[227] = new Color(255, 50, 50);
			colors[228] = new Color(255, 50, 101);
			colors[229] = new Color(255, 50, 153);
			colors[230] = new Color(255, 50, 204);
			colors[231] = new Color(255, 50, 255);
			colors[232] = new Color(255, 101, 0);
			colors[233] = new Color(255, 101, 50);
			colors[234] = new Color(255, 101, 101);
			colors[235] = new Color(255, 101, 153);
			colors[236] = new Color(255, 101, 204);
			colors[237] = new Color(255, 101, 255);
			colors[238] = new Color(255, 153, 0);
			colors[239] = new Color(255, 153, 50);
			colors[240] = new Color(255, 153, 101);
			colors[241] = new Color(255, 153, 153);
			colors[242] = new Color(255, 153, 204);
			colors[243] = new Color(255, 153, 255);
			colors[244] = new Color(255, 204, 0);
			colors[245] = new Color(255, 204, 50);
			colors[246] = new Color(255, 204, 101);
			colors[247] = new Color(255, 204, 153);
			colors[248] = new Color(255, 204, 204);
			colors[249] = new Color(255, 204, 255);
			colors[250] = new Color(255, 255, 0);
			colors[251] = new Color(255, 255, 50);
			colors[252] = new Color(255, 255, 101);
			colors[253] = new Color(255, 255, 153);
			colors[254] = new Color(255, 255, 204);
			colors[255] = new Color(255, 255, 255);
			byte[] rPalette = new byte[256];
			byte[] gPalette = new byte[256];
			byte[] bPalette = new byte[256];
			for (int index = 0; index < 256; index++) {
				rPalette[index] = ((byte) colors[index].getRed());
				gPalette[index] = ((byte) colors[index].getGreen());
				bPalette[index] = ((byte) colors[index].getBlue());
			}
			ColorPalette256 = new IndexColorModel(8, 256, rPalette, gPalette, bPalette);
		}
		return ColorPalette256;
	}
	/**
	 * Answer color palette image for 256 colors.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public static StImage ColorPalette256Image_(Point cellSize) {
		Image cellImage;
		StImage anImage = new StImage(cellSize.x * 32, cellSize.y * 8);
		Toolkit toolkit = Toolkit.getDefaultToolkit();
		IndexColorModel colorModel = ColorPalette256();
		int index = 0;
		int width = cellSize.x;
		int height = cellSize.y;
		int offset = 0;
		int scansize = width;
		for (int y = 0; y < 8; y++) {
			for (int x = 0; x < 32; x++) {
				int[] pixels = new int[width * height];
				for (int i = 0; i < pixels.length; i++) {
					pixels[i] = index;
				}
				MemoryImageSource source = new MemoryImageSource(width, height, colorModel, pixels, offset, scansize);
				cellImage = toolkit.createImage(source);
				anImage.copy_from_in_rule_(new Rectangle(x * width, y * height, width, height), new Point(0, 0), new StImage(cellImage), StImage.Over);
				cellImage.flush();
				index = index + 1;
			}
		}
		return anImage;
	}
	/**
	 * Answer compute image.
	 * 
	 * @return sra.smalltalk.StImage
	 * @param anImage sra.smalltalk.StImage
	 * @param aPoint java.awt.Point
	 */
	private static StImage ComputeImage_shrinkExtent_(StImage anImage, Point aPoint) {
		StInterval[] xIntervals = ComputeIntervals_divisionSize_(anImage.width(), aPoint.x);
		StInterval[] yIntervals = ComputeIntervals_divisionSize_(anImage.height(), aPoint.y);
		StImage shrunkenImage = new StImage(aPoint.x, aPoint.y);
		int yValue = 0;
		ColorModel colorModel = ColorModel.getRGBdefault();
		for (int i = 0; i < yIntervals.length; i++) {
			StInterval yInterval = yIntervals[i];
			int xValue = 0;
			for (int j = 0; j < xIntervals.length; j++) {
				StInterval xInterval = xIntervals[j];
				int redValue = 0;
				int greenValue = 0;
				int blueValue = 0;
				int countValue = 0;
				for (int y = (int) yInterval.start(); y <= (int) yInterval.stop(); y = y + (int) yInterval.step()) {
					for (int x = (int) xInterval.start(); x <= (int) xInterval.stop(); x = x + (int) xInterval.step()) {
						int pixelValue = anImage.atX_y_(x, y) & 0x00FFFFFF;
						redValue = redValue + colorModel.getRed(pixelValue);
						greenValue = greenValue + colorModel.getGreen(pixelValue);
						blueValue = blueValue + colorModel.getBlue(pixelValue);
						countValue = countValue + 1;
					}
				}
				Color colorValue;
				if (countValue > 0) {
					colorValue = new Color(Math.max(0, Math.min(Math.round(((float) redValue) / countValue), 0xFF)), Math.max(0, Math.min(Math.round(((float) greenValue) / countValue), 0xFF)), Math.max(0, Math.min(Math.round(((float) blueValue) / countValue), 0xFF)));
				} else {
					colorValue = new Color(0, 0, 0);
				}
				shrunkenImage.valueAtPoint_put_(new Point(xValue, yValue), colorValue);
				xValue = xValue + 1;
			}
			yValue = yValue + 1;
		}
		return shrunkenImage;
	}
	/**
	 * Answer intervals.
	 * 
	 * @return sra.smalltalk.StInterval[]
	 * @param anInteger int
	 * @param divisionSize int
	 */
	private static StInterval[] ComputeIntervals_divisionSize_(int anInteger, int divisionSize) {
		if (anInteger < divisionSize) {
			throw SmalltalkException.Error(Integer.toString(anInteger) + " is smaller than " + Integer.toString(divisionSize));
		}
		double stepValue = (double) anInteger / (double) divisionSize;
		int[] collection = new int[divisionSize + 1];
		int index = 0;
		for (double i = 0; i <= anInteger; i = i + stepValue) {
			collection[index] = Math.round((float) i);
			index = index + 1;
		}
		int[] startCollection = new int[collection.length - 1];
		int[] endCollection = new int[collection.length - 1];
		for (int j = 0; j < collection.length - 1; j++) {
			startCollection[j] = collection[j];
		}
		for (int k = 0; k < collection.length - 1; k++) {
			endCollection[k] = collection[k + 1] - 1;
		}
		StInterval[] array = new StInterval[startCollection.length];
		for (int l = 0; l < startCollection.length; l++) {
			StInterval interval = new StInterval(startCollection[l], endCollection[l], 1.0d);
			array[l] = interval;
		}
		return array;
	}
	/**
	 * Answer eight points.
	 *
	 * @return java.awt.Point[]
	 */
	public final static Point[] EightPoints() {
		if (EightPoints == null) {
			Point[] points = new Point[8];
			points[0] = new Point(-1, -1);
			points[1] = new Point(0, -1);
			points[2] = new Point(1, -1);
			points[3] = new Point(1, 0);
			points[4] = new Point(1, 1);
			points[5] = new Point(0, 1);
			points[6] = new Point(-1, 1);
			points[7] = new Point(-1, 0);
			EightPoints = points;
		}
		return EightPoints;
	}
	/**
	 * @return sra.smalltalk.StImage
	 * @param sra.smalltalk.StImage
	 */
	public static StImage Fill_(StImage anImage) {
		return Fill_rectangle_color_(anImage, anImage.bounds(), Color.black);
	}
	/**
	 * @return sra.smalltalk.StImage
	 * @param sra.smalltalk.StImage
	 * @param java.awt.Color
	 */
	public static StImage Fill_color_(StImage anImage, Color colorValue) {
		return Fill_rectangle_color_(anImage, anImage.bounds(), colorValue);
	}
	/**
	 * @return sra.smalltalk.StImage
	 * @param sra.smalltalk.StImage
	 * @param java.awt.Rectangle
	 */
	public static StImage Fill_rectangle_(StImage anImage, Rectangle aRectangle) {
		return Fill_rectangle_color_(anImage, aRectangle, Color.black);
	}
	/**
	 * @return sra.smalltalk.StImage
	 * @param sra.smalltalk.StImage
	 * @param java.awt.Rectangle
	 * @param java.awt.Color
	 */
	public static StImage Fill_rectangle_color_(StImage anImage, Rectangle aRectangle, Color colorValue) {
		StImage pattern = new StImage(16, 16);
		int value = colorValue.getRGB();
		for (int x = 0; x < pattern.width(); x++) {
			for (int y = 0; y < pattern.height(); y++) {
				pattern.atX_y_put_(x, y, value);
			}
		}
		return anImage.tile_from_in_rule_(aRectangle, new Point(0, 0), pattern, StImage.Over);
	}
	/**
	 * @return sra.smalltalk.StImage
	 * @param sra.smalltalk.StImage
	 * @param java.awt.Point
	 */
	public static StImage Fill_seed_(StImage anImage, Point aPoint) {
		return Fill_seed_color_(anImage, aPoint, Color.black);
	}
	/**
	 * @return sra.smalltalk.StImage
	 * @param sra.smalltalk.StImage
	 * @param java.awt.Point
	 * @param java.awt.Color
	 */
	public static StImage Fill_seed_color_(StImage anImage, Point aPoint, Color colorValue) {
		int x0, x1, y, direction, xLimit, d, xStart, xEnd;
		int colorIndex = colorValue.getRGB();
		if (colorIndex == anImage.atPoint_(aPoint)) {
			return anImage;
		}
		int whiteIndex = Color.white.getRGB();
		int width = anImage.width();
		int height = anImage.height();
		Vector x0Stack = new Vector();
		x0Stack.addElement(new Integer(LastOn_before_after_index_(anImage, aPoint, -1, whiteIndex) + 1));
		Vector x1Stack = new Vector();
		x1Stack.addElement(new Integer(FirstOn_after_before_index_(anImage, aPoint, width, whiteIndex) - 1));
		Vector yStack = new Vector();
		yStack.addElement(new Integer(aPoint.y));
		Vector directionStack = new Vector();
		directionStack.addElement(new Integer(0));
		while (yStack.size() > 0) {
			x0 = ((Integer) x0Stack.lastElement()).intValue();
			x0Stack.removeElementAt(x0Stack.size() - 1);
			x1 = ((Integer) x1Stack.lastElement()).intValue();
			x1Stack.removeElementAt(x1Stack.size() - 1);
			y = ((Integer) yStack.lastElement()).intValue();
			yStack.removeElementAt(yStack.size() - 1);
			direction = ((Integer) directionStack.lastElement()).intValue();
			directionStack.removeElementAt(directionStack.size() - 1);
			xLimit = Math.min(x1, width - 1);
			Point point = new Point(x0, y);
			while (point.x <= xLimit) {
				anImage.atPoint_put_(point, colorIndex);
				point = new Point(point.x + 1, point.y);
			}
			if ((direction != 1) && (y > 0)) {
				if (whiteIndex == anImage.atPoint_(new Point(x0, y - 1))) {
					xStart = LastOn_before_after_index_(anImage, new Point(x0, y - 1), -1, whiteIndex) + 1;
				} else {
					xStart = FirstOff_after_before_index_(anImage, new Point(x0, y - 1), x1 + 1, whiteIndex);
				}
				while (xStart <= x1) {
					xEnd = FirstOn_after_before_index_(anImage, new Point(xStart, y - 1), width, whiteIndex) - 1;
					x0Stack.addElement(new Integer(xStart));
					x1Stack.addElement(new Integer(xEnd));
					yStack.addElement(new Integer(y - 1));
					if ((x0 - 1 > xStart) || (x1 + 1 < xEnd)) {
						d = 0;
					} else {
						d = 2;
					}
					directionStack.addElement(new Integer(d));
					if (xEnd >= x1) {
						xStart = xEnd + 1;
					} else {
						xStart = FirstOff_after_before_index_(anImage, new Point(xEnd + 1, y - 1), x1 + 1, whiteIndex);
					}
				}
			}
			if ((direction != 2) && (y < height - 1)) {
				if (whiteIndex == anImage.atPoint_(new Point(x0, y + 1))) {
					xStart = LastOn_before_after_index_(anImage, new Point(x0, y + 1), -1, whiteIndex) + 1;
				} else {
					xStart = FirstOff_after_before_index_(anImage, new Point(x0, y + 1), x1 + 1, whiteIndex);
				}
				while (xStart <= x1) {
					xEnd = FirstOn_after_before_index_(anImage, new Point(xStart, y + 1), width, whiteIndex) - 1;
					x0Stack.addElement(new Integer(xStart));
					x1Stack.addElement(new Integer(xEnd));
					yStack.addElement(new Integer(y + 1));
					if ((x0 - 1 > xStart) || (x1 + 1 < xEnd)) {
						d = 0;
					} else {
						d = 1;
					}
					directionStack.addElement(new Integer(d));
					if (xEnd > x1) {
						xStart = xEnd;
					} else {
						xStart = FirstOff_after_before_index_(anImage, new Point(xEnd + 1, y + 1), x1 + 1, whiteIndex);
					}
				}
			}
		}
		return anImage;
	}
	/**
	 * @return int
	 * @param sra.smalltalk.StImage
	 * @param java.awt.Point
	 * @param int
	 * @param int
	 */
	private static int FirstOff_after_before_index_(StImage anImage, Point aPoint, int max, int paletteIndex) {
		if (!(aPoint.x < max)) {
			return max;
		}
		Rectangle b = anImage.bounds();
		int x = Math.min(Math.max(aPoint.x, 0), b.width - 1);
		int y = Math.min(Math.max(aPoint.y, 0), b.height - 1);
		if (paletteIndex != anImage.atX_y_(x, y)) {
			x = x + 1;
			while ((x < max) && (paletteIndex != anImage.atX_y_(x, y))) {
				x = x + 1;
			}
		}
		return Math.min(x, max);
	}
	/**
	 * @return int
	 * @param sra.smalltalk.StImage
	 * @param java.awt.Point
	 * @param int
	 * @param int
	 */
	private static int FirstOn_after_before_index_(StImage anImage, Point aPoint, int max, int paletteIndex) {
		Rectangle b = anImage.bounds();
		int x = Math.min(Math.max(aPoint.x, 0), b.width - 1);
		int y = Math.min(Math.max(aPoint.y, 0), b.height - 1);
		if (paletteIndex == anImage.atX_y_(x, y)) {
			x = x + 1;
			while ((x < max) && (paletteIndex == anImage.atX_y_(x, y))) {
				x = x + 1;
			}
		}
		return Math.min(x, max);
	}
	/**
	 * Clean up my palette
	 */
	public static void FlushPalettes() {
		ColorPalette256 = null;
		GrayPalette256 = null;
		_WhiteBlackPalette = null;
		_BlackWhitePalette = null;
	}
	/**
	 * Clean up my points
	 */
	public static void FlushPoints() {
		EightPoints = null;
		FourPoints = null;
	}
	/**
	 * Answer four points.
	 *
	 * @return java.awt.Point[]
	 */
	public final static Point[] FourPoints() {
		if (FourPoints == null) {
			Point[] points = new Point[4];
			points[0] = new Point(0, -1);
			points[1] = new Point(1, 0);
			points[2] = new Point(0, 1);
			points[3] = new Point(-1, 0);
			FourPoints = points;
		}
		return FourPoints;
	}
	/**
	 * Answer the ColorModel for gray 256 colors.
	 * 
	 * @return java.awt.image.IndexColorModel
	 */
	public final static IndexColorModel GrayPalette256() {
		if (GrayPalette256 == null) {
			Color[] colors = new Color[256];
			colors[0] = new Color(0, 0, 0);
			colors[1] = new Color(0, 0, 0);
			colors[2] = new Color(1, 1, 1);
			colors[3] = new Color(2, 2, 2);
			colors[4] = new Color(3, 3, 3);
			colors[5] = new Color(5, 5, 5);
			colors[6] = new Color(6, 6, 6);
			colors[7] = new Color(7, 7, 7);
			colors[8] = new Color(8, 8, 8);
			colors[9] = new Color(8, 8, 8);
			colors[10] = new Color(9, 9, 9);
			colors[11] = new Color(10, 10, 10);
			colors[12] = new Color(11, 11, 11);
			colors[13] = new Color(13, 13, 13);
			colors[14] = new Color(14, 14, 14);
			colors[15] = new Color(15, 15, 15);
			colors[16] = new Color(16, 16, 16);
			colors[17] = new Color(16, 16, 16);
			colors[18] = new Color(17, 17, 17);
			colors[19] = new Color(18, 18, 18);
			colors[20] = new Color(19, 19, 19);
			colors[21] = new Color(21, 21, 21);
			colors[22] = new Color(22, 22, 22);
			colors[23] = new Color(23, 23, 23);
			colors[24] = new Color(24, 24, 24);
			colors[25] = new Color(24, 24, 24);
			colors[26] = new Color(25, 25, 25);
			colors[27] = new Color(26, 26, 26);
			colors[28] = new Color(27, 27, 27);
			colors[29] = new Color(29, 29, 29);
			colors[30] = new Color(30, 30, 30);
			colors[31] = new Color(31, 31, 31);
			colors[32] = new Color(32, 32, 32);
			colors[33] = new Color(32, 32, 32);
			colors[34] = new Color(33, 33, 33);
			colors[35] = new Color(34, 34, 34);
			colors[36] = new Color(35, 35, 35);
			colors[37] = new Color(36, 36, 36);
			colors[38] = new Color(38, 38, 38);
			colors[39] = new Color(39, 39, 39);
			colors[40] = new Color(40, 40, 40);
			colors[41] = new Color(41, 41, 41);
			colors[42] = new Color(41, 41, 41);
			colors[43] = new Color(42, 42, 42);
			colors[44] = new Color(43, 43, 43);
			colors[45] = new Color(44, 44, 44);
			colors[46] = new Color(46, 46, 46);
			colors[47] = new Color(47, 47, 47);
			colors[48] = new Color(48, 48, 48);
			colors[49] = new Color(49, 49, 49);
			colors[50] = new Color(49, 49, 49);
			colors[51] = new Color(50, 50, 50);
			colors[52] = new Color(51, 51, 51);
			colors[53] = new Color(52, 52, 52);
			colors[54] = new Color(54, 54, 54);
			colors[55] = new Color(55, 55, 55);
			colors[56] = new Color(56, 56, 56);
			colors[57] = new Color(57, 57, 57);
			colors[58] = new Color(57, 57, 57);
			colors[59] = new Color(58, 58, 58);
			colors[60] = new Color(59, 59, 59);
			colors[61] = new Color(60, 60, 60);
			colors[62] = new Color(62, 62, 62);
			colors[63] = new Color(63, 63, 63);
			colors[64] = new Color(64, 64, 64);
			colors[65] = new Color(65, 65, 65);
			colors[66] = new Color(65, 65, 65);
			colors[67] = new Color(66, 66, 66);
			colors[68] = new Color(67, 67, 67);
			colors[69] = new Color(68, 68, 68);
			colors[70] = new Color(70, 70, 70);
			colors[71] = new Color(71, 71, 71);
			colors[72] = new Color(72, 72, 72);
			colors[73] = new Color(73, 73, 73);
			colors[74] = new Color(74, 74, 74);
			colors[75] = new Color(74, 74, 74);
			colors[76] = new Color(75, 75, 75);
			colors[77] = new Color(76, 76, 76);
			colors[78] = new Color(77, 77, 77);
			colors[79] = new Color(79, 79, 79);
			colors[80] = new Color(80, 80, 80);
			colors[81] = new Color(81, 81, 81);
			colors[82] = new Color(82, 82, 82);
			colors[83] = new Color(82, 82, 82);
			colors[84] = new Color(83, 83, 83);
			colors[85] = new Color(84, 84, 84);
			colors[86] = new Color(85, 85, 85);
			colors[87] = new Color(87, 87, 87);
			colors[88] = new Color(88, 88, 88);
			colors[89] = new Color(89, 89, 89);
			colors[90] = new Color(90, 90, 90);
			colors[91] = new Color(90, 90, 90);
			colors[92] = new Color(91, 91, 91);
			colors[93] = new Color(92, 92, 92);
			colors[94] = new Color(93, 93, 93);
			colors[95] = new Color(95, 95, 95);
			colors[96] = new Color(96, 96, 96);
			colors[97] = new Color(97, 97, 97);
			colors[98] = new Color(98, 98, 98);
			colors[99] = new Color(98, 98, 98);
			colors[100] = new Color(99, 99, 99);
			colors[101] = new Color(100, 100, 100);
			colors[102] = new Color(101, 101, 101);
			colors[103] = new Color(103, 103, 103);
			colors[104] = new Color(104, 104, 104);
			colors[105] = new Color(105, 105, 105);
			colors[106] = new Color(106, 106, 106);
			colors[107] = new Color(106, 106, 106);
			colors[108] = new Color(107, 107, 107);
			colors[109] = new Color(108, 108, 108);
			colors[110] = new Color(109, 109, 109);
			colors[111] = new Color(110, 110, 110);
			colors[112] = new Color(112, 112, 112);
			colors[113] = new Color(113, 113, 113);
			colors[114] = new Color(114, 114, 114);
			colors[115] = new Color(115, 115, 115);
			colors[116] = new Color(115, 115, 115);
			colors[117] = new Color(116, 116, 116);
			colors[118] = new Color(117, 117, 117);
			colors[119] = new Color(118, 118, 118);
			colors[120] = new Color(120, 120, 120);
			colors[121] = new Color(121, 121, 121);
			colors[122] = new Color(122, 122, 122);
			colors[123] = new Color(123, 123, 123);
			colors[124] = new Color(123, 123, 123);
			colors[125] = new Color(124, 124, 124);
			colors[126] = new Color(125, 125, 125);
			colors[127] = new Color(126, 126, 126);
			colors[128] = new Color(128, 128, 128);
			colors[129] = new Color(129, 129, 129);
			colors[130] = new Color(130, 130, 130);
			colors[131] = new Color(131, 131, 131);
			colors[132] = new Color(131, 131, 131);
			colors[133] = new Color(132, 132, 132);
			colors[134] = new Color(133, 133, 133);
			colors[135] = new Color(134, 134, 134);
			colors[136] = new Color(136, 136, 136);
			colors[137] = new Color(137, 137, 137);
			colors[138] = new Color(138, 138, 138);
			colors[139] = new Color(139, 139, 139);
			colors[140] = new Color(139, 139, 139);
			colors[141] = new Color(140, 140, 140);
			colors[142] = new Color(141, 141, 141);
			colors[143] = new Color(142, 142, 142);
			colors[144] = new Color(144, 144, 144);
			colors[145] = new Color(145, 145, 145);
			colors[146] = new Color(146, 146, 146);
			colors[147] = new Color(147, 147, 147);
			colors[148] = new Color(148, 148, 148);
			colors[149] = new Color(148, 148, 148);
			colors[150] = new Color(149, 149, 149);
			colors[151] = new Color(150, 150, 150);
			colors[152] = new Color(151, 151, 151);
			colors[153] = new Color(153, 153, 153);
			colors[154] = new Color(154, 154, 154);
			colors[155] = new Color(155, 155, 155);
			colors[156] = new Color(156, 156, 156);
			colors[157] = new Color(156, 156, 156);
			colors[158] = new Color(157, 157, 157);
			colors[159] = new Color(158, 158, 158);
			colors[160] = new Color(159, 159, 159);
			colors[161] = new Color(161, 161, 161);
			colors[162] = new Color(162, 162, 162);
			colors[163] = new Color(163, 163, 163);
			colors[164] = new Color(164, 164, 164);
			colors[165] = new Color(164, 164, 164);
			colors[166] = new Color(165, 165, 165);
			colors[167] = new Color(166, 166, 166);
			colors[168] = new Color(167, 167, 167);
			colors[169] = new Color(169, 169, 169);
			colors[170] = new Color(170, 170, 170);
			colors[171] = new Color(171, 171, 171);
			colors[172] = new Color(172, 172, 172);
			colors[173] = new Color(172, 172, 172);
			colors[174] = new Color(173, 173, 173);
			colors[175] = new Color(174, 174, 174);
			colors[176] = new Color(175, 175, 175);
			colors[177] = new Color(177, 177, 177);
			colors[178] = new Color(178, 178, 178);
			colors[179] = new Color(179, 179, 179);
			colors[180] = new Color(180, 180, 180);
			colors[181] = new Color(180, 180, 180);
			colors[182] = new Color(181, 181, 181);
			colors[183] = new Color(182, 182, 182);
			colors[184] = new Color(183, 183, 183);
			colors[185] = new Color(184, 184, 184);
			colors[186] = new Color(186, 186, 186);
			colors[187] = new Color(187, 187, 187);
			colors[188] = new Color(188, 188, 188);
			colors[189] = new Color(189, 189, 189);
			colors[190] = new Color(189, 189, 189);
			colors[191] = new Color(190, 190, 190);
			colors[192] = new Color(191, 191, 191);
			colors[193] = new Color(192, 192, 192);
			colors[194] = new Color(194, 194, 194);
			colors[195] = new Color(195, 195, 195);
			colors[196] = new Color(196, 196, 196);
			colors[197] = new Color(197, 197, 197);
			colors[198] = new Color(197, 197, 197);
			colors[199] = new Color(198, 198, 198);
			colors[200] = new Color(199, 199, 199);
			colors[201] = new Color(200, 200, 200);
			colors[202] = new Color(202, 202, 202);
			colors[203] = new Color(203, 203, 203);
			colors[204] = new Color(204, 204, 204);
			colors[205] = new Color(205, 205, 205);
			colors[206] = new Color(205, 205, 205);
			colors[207] = new Color(206, 206, 206);
			colors[208] = new Color(207, 207, 207);
			colors[209] = new Color(208, 208, 208);
			colors[210] = new Color(210, 210, 210);
			colors[211] = new Color(211, 211, 211);
			colors[212] = new Color(212, 212, 212);
			colors[213] = new Color(213, 213, 213);
			colors[214] = new Color(213, 213, 213);
			colors[215] = new Color(214, 214, 214);
			colors[216] = new Color(215, 215, 215);
			colors[217] = new Color(216, 216, 216);
			colors[218] = new Color(218, 218, 218);
			colors[219] = new Color(219, 219, 219);
			colors[220] = new Color(220, 220, 220);
			colors[221] = new Color(221, 221, 221);
			colors[222] = new Color(222, 222, 222);
			colors[223] = new Color(222, 222, 222);
			colors[224] = new Color(223, 223, 223);
			colors[225] = new Color(224, 224, 224);
			colors[226] = new Color(225, 225, 225);
			colors[227] = new Color(227, 227, 227);
			colors[228] = new Color(228, 228, 228);
			colors[229] = new Color(229, 229, 229);
			colors[230] = new Color(230, 230, 230);
			colors[231] = new Color(230, 230, 230);
			colors[232] = new Color(231, 231, 231);
			colors[233] = new Color(232, 232, 232);
			colors[234] = new Color(233, 233, 233);
			colors[235] = new Color(235, 235, 235);
			colors[236] = new Color(236, 236, 236);
			colors[237] = new Color(237, 237, 237);
			colors[238] = new Color(238, 238, 238);
			colors[239] = new Color(238, 238, 238);
			colors[240] = new Color(239, 239, 239);
			colors[241] = new Color(240, 240, 240);
			colors[242] = new Color(241, 241, 241);
			colors[243] = new Color(243, 243, 243);
			colors[244] = new Color(244, 244, 244);
			colors[245] = new Color(245, 245, 245);
			colors[246] = new Color(246, 246, 246);
			colors[247] = new Color(246, 246, 246);
			colors[248] = new Color(247, 247, 247);
			colors[249] = new Color(248, 248, 248);
			colors[250] = new Color(249, 249, 249);
			colors[251] = new Color(251, 251, 251);
			colors[252] = new Color(252, 252, 252);
			colors[253] = new Color(253, 253, 253);
			colors[254] = new Color(254, 254, 254);
			colors[255] = new Color(255, 255, 255);
			byte[] rPalette = new byte[256];
			byte[] gPalette = new byte[256];
			byte[] bPalette = new byte[256];
			for (int index = 0; index < 256; index++) {
				rPalette[index] = ((byte) colors[index].getRed());
				gPalette[index] = ((byte) colors[index].getGreen());
				bPalette[index] = ((byte) colors[index].getBlue());
			}
			GrayPalette256 = new IndexColorModel(8, 256, rPalette, gPalette, bPalette);
		}
		return GrayPalette256;
	}
	/**
	 * Answer gray palette image for 256 colors.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public static StImage GrayPalette256Image_(Point cellSize) {
		Image cellImage;
		StImage anImage = new StImage(cellSize.x * 32, cellSize.y * 8);
		Toolkit toolkit = Toolkit.getDefaultToolkit();
		IndexColorModel colorModel = GrayPalette256();
		int index = 0;
		int width = cellSize.x;
		int height = cellSize.y;
		int offset = 0;
		int scansize = width;
		for (int y = 0; y < 8; y++) {
			for (int x = 0; x < 32; x++) {
				int[] pixels = new int[width * height];
				for (int i = 0; i < pixels.length; i++) {
					pixels[i] = index;
				}
				MemoryImageSource source = new MemoryImageSource(width, height, colorModel, pixels, offset, scansize);
				cellImage = toolkit.createImage(source);
				anImage.copy_from_in_rule_(new Rectangle(x * width, y * height, width, height), new Point(0, 0), new StImage(cellImage), StImage.Over);
				cellImage.flush();
				index = index + 1;
			}
		}
		return anImage;
	}
	/**
	 * @return java.awt.Image
	 * @param double
	 */
	public static Image Halftone_(double halftoneScale) {
		double scale = halftoneScale;
		scale = Math.max(scale, 0);
		scale = Math.min(scale, 1);
		int index = new Double((Masks.length) * scale).intValue() + 1;
		return MaskAt_(index);
	}
	/**
	 * @return sra.smalltalk.StImage
	 * @param sra.smalltalk.StImage
	 * @param int
	 */
	public static StImage Icon_size_(StImage anImage, int sizeInteger) {
		return (StImage) Icon2_size_(anImage, sizeInteger).firstElement();
	}
	/**
	 * Answer icon image with area.
	 *
	 * @return java.util.Vector
	 * @param sra.smalltalk.StImage
	 * @param int
	 */
	public static Vector Icon2_size_(StImage anImage, int sizeInteger) {
		double factor = ((double) sizeInteger) / Math.max(anImage.width(), anImage.height());
		StImage image = Scale_factor_(anImage, new Isr2dPoint(factor, factor));
		int maxSize = Math.max(image.width(), image.height());
		Point extent = new Point(maxSize, maxSize);
		Rectangle area = image.bounds();
		area.translate((int) ((extent.x / 2.0d) - (image.bounds().width / 2.0d)), (int) ((extent.y / 2.0d) - (image.bounds().height / 2.0d)));
		int[] bits = new int[maxSize * maxSize];
		for (int i = 0; i < maxSize * maxSize; i++) {
			bits[i] = 0x00FFFFFF;
		}
		StImage figure = new StImage(extent.x, extent.y, (int[]) bits.clone());
		figure.copy_from_in_rule_(area, new Point(0, 0), image, StImage.Over);
		StImage shape = new StImage(extent.x, extent.y, (int[]) bits.clone());
		shape = Fill_rectangle_color_(shape, area, Color.black);
		StImage opaque = _MakeImage_Shape_(figure, shape);
		Vector array = new Vector();
		array.addElement(opaque);
		array.addElement(area);
		return array;
	}
	/**
	 * Answer icon image with area.
	 *
	 * @return java.util.Vector
	 * @param sra.smalltalk.StImage
	 * @param int
	 */
	public static Vector Icon3_size_(StImage anImage, int sizeInteger) {
		StImage image = null;
		Point extent = null;
		double factor = ((double) sizeInteger) / Math.max(anImage.width(), anImage.height());
		try {
			extent = new Point(Math.round((float) (factor * anImage.width())), Math.round((float) (factor * anImage.height())));
			image = ComputeImage_shrinkExtent_(anImage, extent);
		} catch (Exception e) {
			image = Scale_factor_(anImage, new Isr2dPoint(factor, factor));
		}
		int maxSize = Math.max(image.width(), image.height());
		extent = new Point(maxSize, maxSize);
		Rectangle area = image.bounds();
		area.translate((int) ((extent.x / 2.0d) - (image.bounds().width / 2.0d)), (int) ((extent.y / 2.0d) - (image.bounds().height / 2.0d)));
		int[] bits = new int[maxSize * maxSize];
		for (int i = 0; i < maxSize * maxSize; i++) {
			bits[i] = 0x00FFFFFF;
		}
		StImage figure = new StImage(extent.x, extent.y, (int[]) bits.clone());
		figure.copy_from_in_rule_(area, new Point(0, 0), image, StImage.Over);
		StImage shape = new StImage(extent.x, extent.y, (int[]) bits.clone());
		shape = Fill_rectangle_color_(shape, area, Color.black);
		StImage opaque = _MakeImage_Shape_(figure, shape);
		Vector array = new Vector();
		array.addElement(opaque);
		array.addElement(area);
		return array;
	}
	/**
	 * receives BMP or JPEG filename (full path), and return StImage 
	 * @return java.awt.Image
	 * @param String
	 */
	public static Image ImageFromFile_(String filename) {
		if (filename.toLowerCase().endsWith(".gif")) {
			System.err.println("It is not allowed to load GIF image.");
			return null;
		}
		StImage anImage = null;
		if (filename.toLowerCase().endsWith(".bmp")) {
			try {
				FileInputStream file = new FileInputStream(filename);
				IsrBmpImageStream stream = IsrBmpImageStream.On_(file);
				anImage = stream.nextImage();
				file.close();
			} catch (Exception e) {
				System.err.println("could not load image.");
				return null;
			}
		} else {
			Frame frame = new Frame();
			Toolkit toolKit = frame.getToolkit();
			Image image = toolKit.getImage(filename);
			MediaTracker mt = new MediaTracker(frame);
			mt.addImage(image, 0);
			try {
				mt.waitForAll();
			} catch (InterruptedException e) {
				System.err.println("image load interrupted.");
				return null;
			}
			int status = mt.statusAll(false);
			if ((status & MediaTracker.ABORTED) != 0 || (status & MediaTracker.ERRORED) != 0) {
				System.err.println("could not load image.");
				return null;
			}
			anImage = new StImage(image);
			image.flush();
		};
		return anImage.image();
	}
	/**
	 * open BMP or JPEG file with dialog and return StImage 
	 * @return java.awt.Image
	 */
	public static Image ImageFromFileDialog() {
		File file = StDialog.OpenFileDialog_("Select a image file");
		if (file == null) {
			return null;
		};
		return ImageFromFile_(file.getPath());
	}
	/**
	 * @return int
	 * @param sra.smalltalk.StImage
	 * @param java.awt.Point
	 * @param int
	 * @param int
	 */
	private static int LastOn_before_after_index_(StImage anImage, Point aPoint, int min, int paletteIndex) {
		Rectangle b = anImage.bounds();
		int x = Math.min(Math.max(aPoint.x, 0), b.width - 1);
		int y = Math.min(Math.max(aPoint.y, 0), b.height - 1);
		while ((x > min) && (paletteIndex == anImage.atX_y_(x, y))) {
			x = x - 1;
		}
		return x;
	}
	/**
	 * Answer Image thin
	 *
	 * @return sra.smalltalk.StImage
	 * @param sra.smalltalk.StImage
	 */
	public static StImage MakeThin_(StImage anImage) {
		int jsw, i1, j1, delta, ii1, ii2, jj1, jj2, i, im, ip, j, jm, jp, data;
		int colorWhiteValue0 = Color.white.getRGB();
		int colorBlackValue1 = Color.black.getRGB();
		int colorGrayValue3 = Color.gray.getRGB();
		int colorGrayValue5 = Color.darkGray.getRGB();
		int colorGrayValue7 = Color.lightGray.getRGB();
		StImage image = anImage.convertToPalette_(_WhiteBlackPalette());
		StImage map = new StImage(image.width(), image.height());
		for (int z = 0; z < map.bits().length; z++) {
			map.bits()[z] = image.bits()[z];
		}
		i1 = map.width() - 1;
		j1 = map.height() - 1;
		boolean loop = true;
		while (loop) {
			for (int x = 1; x < i1; x++) {
				for (int y = 1; y < j1; y++) {
					if (map.atX_y_(x, y) == colorBlackValue1) {
						if (map.atX_y_(x - 1, y) == colorWhiteValue0) {
							map.atX_y_put_(x, y, colorGrayValue3);
						} else {
							if (map.atX_y_(x, y - 1) == colorWhiteValue0) {
								map.atX_y_put_(x, y, colorGrayValue5);
							} else {
								if (map.atX_y_(x + 1, y) == colorWhiteValue0) {
									map.atX_y_put_(x, y, colorGrayValue3);
								} else {
									if (map.atX_y_(x, y + 1) == colorWhiteValue0) {
										map.atX_y_put_(x, y, colorGrayValue5);
									}
								}
							}
						}
					}
				}
			}
			jsw = 0;
			delta = 1;
			while (delta >= -1) {
				if (delta == 1) {
					ii1 = 1;
					ii2 = i1;
					jj1 = 1;
					jj2 = j1;
					data = colorGrayValue3;
				} else {
					ii1 = i1 - 1;
					ii2 = 0;
					jj1 = j1 - 1;
					jj2 = 0;
					data = colorGrayValue5;
				}
				i = ii1;
				while (i != ii2) {
					im = i - 1;
					ip = i + 1;
					j = jj1;
					while (j != jj2) {
						jm = j - 1;
						jp = j + 1;
						if (map.atX_y_(i, j) == data) {
							int a1, a2, a3, a4, a5, a6, a7, a8, ns, k;
							jsw = 1;
							if (map.atX_y_(i, jp) == colorWhiteValue0) {
								a1 = 1;
							} else {
								a1 = 0;
							}
							if (map.atX_y_(im, jp) == colorWhiteValue0) {
								a2 = 1;
							} else {
								a2 = 0;
							}
							if (map.atX_y_(im, j) == colorWhiteValue0) {
								a3 = 1;
							} else {
								a3 = 0;
							}
							if (map.atX_y_(im, jm) == colorWhiteValue0) {
								a4 = 1;
							} else {
								a4 = 0;
							}
							if (map.atX_y_(i, jm) == colorWhiteValue0) {
								a5 = 1;
							} else {
								a5 = 0;
							}
							if (map.atX_y_(ip, jm) == colorWhiteValue0) {
								a6 = 1;
							} else {
								a6 = 0;
							}
							if (map.atX_y_(ip, j) == colorWhiteValue0) {
								a7 = 1;
							} else {
								a7 = 0;
							}
							if (map.atX_y_(ip, jp) == colorWhiteValue0) {
								a8 = 1;
							} else {
								a8 = 0;
							}
							ns = a1 + a3 + a5 + a7;
							if ((ns + a2 + a4 + a6 + a8) <= 6) {
								k = ns - (a1 & a2 & a3) - (a3 & a4 & a5) - (a5 & a6 & a7) - (a7 & a8 & a1);
								if (k == 1) {
									map.atX_y_put_(i, j, colorWhiteValue0);
									image.atX_y_put_(i, j, colorWhiteValue0);
								} else {
									map.atX_y_put_(i, j, colorGrayValue7);
								}
							} else {
								map.atX_y_put_(i, j, colorGrayValue7);
							}
						}
						j = j + delta;
					}
					i = i + delta;
				}
				delta = delta - 2;
			}
			loop = (jsw > 0);
		}
		return image;
	}
	/**
	 * @return java.awt.Image
	 * @param int
	 *
	 *Example: Evaluate in scrapbook.
	 * sra.jun.goodies.images.IsrImageProcessor.MaskAt_(33);
	 *
	 */
	public static Image MaskAt_(int indexInteger) {
		int index = indexInteger;
		index = Math.max(index,1);
		index = Math.min(index,Masks.length);
		return Masks[index - 1].image();
	}
	/**
	 * 
	 * @return StImage
	 * @param image StImage
	 * @param point Jun2dPoint
	 */
	public static StImage Scale_factor_(StImage anImage, Isr2dPoint scaleFactor) {
		Isr2dPoint scale = new Isr2dPoint(1.0 / scaleFactor.x(), 1.0 / scaleFactor.y());
		StImage image = anImage.copyEmpty_(anImage.scaledExtent_(scale.x(), scale.y()));
		StImage wide = anImage.copyEmpty_(new Point(anImage.width(), new Double(anImage.height() * 1.0 / scale.y()).intValue()));
		for (int y = 0; y < wide.height(); y++) {
			int[] row = anImage.packedRowAt_(new Double(y * scale.y()).intValue());
			wide.packedRowAt_into_(y, row);
		};
		for (int x = 0; x < image.width(); x++) {
			image.copy_from_in_rule_(new Rectangle(x, 0, 1, wide.height()), new Point((new Double(x * scale.x()).intValue()), 0), wide, StImage.Over);
		};
		return image;
	}
	/**
	 * @return sra.smalltalk.StImage
	 * @param sra.smalltalk.StImage
	 */
	public static StImage Shape_(StImage anImage) {
		return Shape_color_(anImage, Color.black);
	}
	/**
	 * @return sra.smalltalk.StImage
	 * @param sra.smalltalk.StImage
	 * @param java.awt.Color
	 */
	public static StImage Shape_color_(StImage anImage, Color colorValue) {
		StImage copyImage = new StImage(anImage.width(), anImage.height());
		for (int i = 0; i < copyImage.bits().length; i++) {
			copyImage.bits()[i] = anImage.bits()[i];
		}
		StImage expandImage = new StImage(copyImage.width() + 2, copyImage.height() + 2);
		expandImage.copy_from_in_rule_(new Rectangle(1, 1, copyImage.width(), copyImage.height()), new Point(0, 0), copyImage, StImage.Over);
		Fill_seed_color_(expandImage, new Point(0, 0), colorValue);
		int width = copyImage.width();
		int height = copyImage.height();
		int index = colorValue.getRGB();
		for (int y = 0; y < height; y++) {
			for (int x = 0; x < width; x++) {
				Point point = new Point(x, y);
				if ((index != copyImage.atPoint_(point)) && (index != expandImage.atPoint_(new Point(x + 1, y + 1)))) {
					copyImage.atPoint_put_(point, index);
				}
			}
		}
		return copyImage;
	}
	/**
	 * Answer image similarity
	 * 
	 * @return float
	 * @param image1 sra.smalltalk.StImage
	 * @param image1 sra.smalltalk.StImage
	 */
	public static float Similarity_with_(StImage image1, StImage image2) {
		if (image1.bits().length <= 0) {
			return 0.0f;
		}
		StImage destination = image1.convertToPalette_(_WhiteBlackPalette());
		StImage source = image2.convertToPalette_(_WhiteBlackPalette());
		destination.copy_from_in_rule_(destination.bounds(), new Point(0, 0), source, StImage.Reverse);
		int color0 = Color.white.getRGB();
		int color1 = Color.black.getRGB();
		int count = 0;
		for (int y = 0; y < destination.height(); y++) {
			for (int x = 0; x < destination.width(); x++) {
				int v = destination.atX_y_(x, y);
				if (v == color1) {
					count = count + 1;
				}
			}
		}
		float similarity = 1.0f - (((float) count) / destination.bits().length);
		similarity = Math.max(0.0f, Math.min(similarity, 1.0f));
		return similarity;
	}
	/**
	 * 
	 * @return int
	 * @param double
	 */
	public static int Stipple_(double stippleScale) {
		double scale = stippleScale;
		scale = Math.max(scale, 0);
		scale = Math.min(scale, 1);
		int index = new Double((Stipples.length - 1) * scale).intValue() + 1;
		return StippleAt_(index);
	}
	/**
	 * @return int
	 * @param int
	 */
	public static int StippleAt_(int indexInteger) {
		int index = indexInteger;
		index = Math.max(index,1);
		index = Math.min(index,Stipples.length);
		return Stipples[index - 1];
	}
	/**
	 * Answer trace border on anImage.
	 * 
	 * @return Object[]
	 * @param anImage sra.smalltalk.StImage
	 */
	public static Object[] TraceBorder_(StImage anImage) {
		StAssociation[] collection = BorderFollowing_(anImage);
		Object[] array = new Object[collection.length];
		for (int i = 0; i < collection.length; i++) {
			array[i] = (collection[i]).value();
		}
		return array;
	}
	/**
	 * Answer x-spectrum on anImage.
	 * 
	 * @return (sra.smalltalk.StAssociation[]) Object[]
	 * @param anImage sra.smalltalk.StImage
	 */
	public static Object[] XSpectrum_(StImage anImage) {
		Integer key;
		double value;
		Color color;
		Hashtable dictionary = new Hashtable();
		for (int x = 0; x < anImage.width(); x++) {
			key = new Integer(x);
			value = 0.0d;
			for (int y = 0; y < anImage.height(); y++) {
				color = anImage.valueAtPoint_(new Point(x, y));
				value = value + (1.0d - StColorValue._GetBrightness(color));
			}
			if (dictionary.containsKey(key)) {
				value = value + ((Double) (dictionary.get(key))).doubleValue();
			}
			dictionary.put(key, new Double(value));
		}
		StBlockClosure sortBlock = new StBlockClosure() {
			public Object value_value_(Object obj1, Object obj2) {
				int obj1Number = ((Number) ((StAssociation) obj1).key()).intValue();
				int obj2Number = ((Number) ((StAssociation) obj2).key()).intValue();
				return new Boolean(obj1Number <= obj2Number);
			}
		};
		StSortedCollection spectrum = new StSortedCollection(sortBlock);
		Enumeration enum = dictionary.keys();
		while (enum.hasMoreElements()) {
			Integer assocKey = (Integer) enum.nextElement();
			Double assocValue = (Double) dictionary.get(assocKey);
			value = assocValue.doubleValue() / (double) anImage.height();
			StAssociation pair = new StAssociation(assocKey, new Double(value));
			spectrum.add_(pair);
		}
		return spectrum._asArray();
	}
	/**
	 * Answer y-spectrum on anImage.
	 * 
	 * @return (sra.smalltalk.StAssociation[]) Object[]
	 * @param anImage sra.smalltalk.StImage
	 */
	public static Object[] YSpectrum_(StImage anImage) {
		Integer key;
		double value;
		Color color;
		Hashtable dictionary = new Hashtable();
		for (int y = 0; y < anImage.height(); y++) {
			key = new Integer(y);
			value = 0.0d;
			for (int x = 0; x < anImage.width(); x++) {
				color = anImage.valueAtPoint_(new Point(x, y));
				value = value + (1.0d - StColorValue._GetBrightness(color));
			}
			if (dictionary.containsKey(key)) {
				value = value + ((Double) (dictionary.get(key))).doubleValue();
			}
			dictionary.put(key, new Double(value));
		}
		StBlockClosure sortBlock = new StBlockClosure() {
			public Object value_value_(Object obj1, Object obj2) {
				int obj1Number = ((Number) ((StAssociation) obj1).key()).intValue();
				int obj2Number = ((Number) ((StAssociation) obj2).key()).intValue();
				return new Boolean(obj1Number <= obj2Number);
			}
		};
		StSortedCollection spectrum = new StSortedCollection(sortBlock);
		Enumeration enum = dictionary.keys();
		while (enum.hasMoreElements()) {
			Integer assocKey = (Integer) enum.nextElement();
			Double assocValue = (Double) dictionary.get(assocKey);
			value = assocValue.doubleValue() / (double) anImage.width();
			StAssociation pair = new StAssociation(assocKey, new Double(value));
			spectrum.add_(pair);
		}
		return spectrum._asArray();
	}
}
