package sra.isr.image.objects;

import java.awt.*;
import java.awt.image.*;
import java.io.*;
import java.util.Vector;
import sra.smalltalk.*;
import sra.isr.image.support.*;
import sra.isr.geometry.basic.*;

/**
 * IsrImageObject class
 * 
 * 	@author:        Mitsuhiro Asada
 * 	@created:       1999/12/27 (by Mitsuhiro Asada)
 * 	@updated:       2000/04/05 (by Mitsuhiro Asada)
 * 	@ImageSearcher: 019
 * 	@JDK:           1.1.6 or higher
 * 	@copyright:     2000 SRA (Software Research Associates, Inc.)
 * 
 * 	$Id: IsrImageObject.java,v 1.18 2000/04/07 04:24:16 m-asada Exp $
 */
public class IsrImageObject extends StObject implements Serializable {
	protected String fileName;
	protected IsrImageFolder imageFolder;
	protected long modifiedDate;
	protected StImage iconImage;
	protected Rectangle iconArea;
	protected Color[] nineColors;
	protected Color averageColor;
	protected Color typicalColor;
	protected Isr2dPoint centerSpectra;
	protected StImage xSpectrum;
	protected StImage ySpectrum;
	protected StImage renderingImage;
	/**
	 * Answer my file name.
	 * 
	 * @return java.io.File
	 */
	public File asFilename() {
		if (this.imageFolder() == null) {
			return new File(this.fileName());
		}
		return this.imageFolder().asFilenameWith_(this.fileName());
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * (Smalltalk version only)
	 * 
	 * @return java.lang.Object
	 * @param asserBlock sra.smalltalk.StBlockClosure
	 * @param doBlock sra.smalltalk.StBlockClosure
	 * @param ensureBlock sra.smalltalk.StBlockClosure
	 */
	private Object assert_do_ensure_(StBlockClosure asserBlock, StBlockClosure doBlock, StBlockClosure ensureBlock) {
		/* asserBlock.value();
		Object result = null;
		try {
			result = doBlock.value();
		} finally {
			ensureBlock.value();
		}
		return result; */
		return null;
	}
	/**
	 * Answer average color on my image.
	 * 
	 * @return java.awt.Color
	 */
	public Color averageColor() {
		if (averageColor == null) {
			StImage image = this.iconImage();
			Rectangle area = this.iconArea();
			Color color = null;
			int red = 0;
			int green = 0;
			int blue = 0;
			for (int y = area.y; y < (area.y + area.height); y++) {
				for (int x = area.x; x < (area.x + area.width); x++) {
					color = image.valueAtPoint_(new Point(x, y));
					red = red + color.getRed();
					green = green + color.getGreen();
					blue = blue + color.getBlue();
				}
			}
			int areaSize = area.width * area.height;
			if (areaSize == 0) {
				color = Color.white;
			} else {
				red = Math.round(((float) red) / areaSize);
				green = Math.round(((float) green) / areaSize);
				blue = Math.round(((float) blue) / areaSize);
				color = new Color(red, green, blue);
			}
			averageColor = color;
		}
		return averageColor;
	}
	/**
	 * Answer average gray color on my image.
	 * 
	 * @return java.awt.Color
	 */
	public Color averageGray() {
		return StColorValue.Brightness_(StColorValue._GetBrightness(this.averageColor()));
	}
	/**
	 * Answer binary(white and black) image.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage binaryImage() {
		StImage image = this.iconImage().convertToPalette_(IsrImageProcessor._WhiteBlackPalette());
		return image;
	}
	/**
	 * Answer my image center spectra.
	 * 
	 * @return sra.isr.geometry.basic.Isr2dPoint
	 */
	public Isr2dPoint centerSpectra() {
		if (centerSpectra == null) {
			Object[] xs = IsrImageProcessor.XSpectrum_(this.iconImage());
			if (xSpectrum == null) {
				xSpectrum = this.xSpectrum_height_(xs, this.iconImage().height());
			}
			Object[] ys = IsrImageProcessor.YSpectrum_(this.iconImage());
			if (ySpectrum == null) {
				ySpectrum = this.ySpectrum_width_(ys, this.iconImage().width());
			}
			double xTotal = 0.0d;
			for (int i = 0; i < xs.length; i++) {
				xTotal = xTotal + ((Double) ((StAssociation) xs[i]).value()).doubleValue();
			}
			double yTotal = 0.0d;
			for (int j = 0; j < ys.length; j++) {
				yTotal = yTotal + ((Double) ((StAssociation) ys[j]).value()).doubleValue();
			}
			double count = 0.0d;
			double x = -1.0d; // null
			for (int k = 0; k < xs.length; k++) {
				count = count + ((Double) ((StAssociation) xs[k]).value()).doubleValue();
				if ((count > (xTotal * 0.5)) && (x == -1.0d)) {
					x = ((Integer) ((StAssociation) xs[k]).key()).doubleValue() / (double) xs.length;
				}
			}
			if (x == -1.0d) {
				x = 0.5d;
			}
			count = 0.0d;
			double y = -1.0d; // null
			for (int l = 0; l < xs.length; l++) {
				count = count + ((Double) ((StAssociation) ys[l]).value()).doubleValue();
				if ((count > (yTotal * 0.5)) && (y == -1.0d)) {
					y = ((Integer) ((StAssociation) ys[l]).key()).doubleValue() / (double) ys.length;
				}
			}
			if (y == -1.0d) {
				y = 0.5d;
			}
			centerSpectra = new Isr2dPoint(x, y);
		}
		return centerSpectra;
	}
	/**
	 * Answer default icon size.
	 * 
	 * @return int
	 */
	public int defaultSize() {
		return this.DefaultSize();
	}
	/**
	 * Answer default icon size.
	 * 
	 * @return int
	 */
	public static int DefaultSize() {
		return 50;
	}
	/**
	 * Answer my file name.
	 * 
	 * @return java.lang.String
	 */
	public String fileName() {
		return this.fileName;
	}
	/**
	 * Create new instance with imageFilename and imageFolder.
	 * 
	 * @return sra.isr.image.objects.IsrImageObject
	 * @param aString java.lang.String
	 * @param anImageFolder sra.isr.image.objects.IsrImageFolder
	 */
	public static IsrImageObject FileName_imageFolder_(String aString, IsrImageFolder anImageFolder) {
		IsrImageObject self = new IsrImageObject();
		self.setFileName_imageFolder_(aString, anImageFolder);
		return self;
	}
	/**
	 * Cleanup my variables.
	 */
	private void flushCaches() {
		iconImage = null;
		iconArea = null;
		nineColors = null;
		averageColor = null;
		typicalColor = null;
		centerSpectra = null;
		xSpectrum = null;
		ySpectrum = null;
		renderingImage = null;
	}
	/**
	 * Answer gray image from my image.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage grayImage() {
		return this.iconImage().convertToPalette_(IsrImageProcessor.GrayPalette256());
	}
	/**
	 * Answer my icon size.
	 * 
	 * @return java.awt.Rectangle or null
	 */
	public Rectangle iconArea() {
		if (iconArea == null) {
			this.iconImage();
		}
		return iconArea;
	}
	/**
	 * Answer my icon image.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage iconImage() {
		if (iconImage == null) {
			Vector array = IsrImageProcessor.Icon3_size_(this.originalImage(), this.defaultSize());
			iconImage = ((StImage) array.firstElement()).convertToPalette_(IsrImageProcessor.ColorPalette256());
			iconArea = (Rectangle) array.lastElement();
			int[] bits = new int[this.defaultSize() * this.defaultSize()];
			for (int i = 0; i < bits.length; i++) {
				bits[i] = 0x00FFFFFF;
			}
			StImage shape = new StImage(this.defaultSize(), this.defaultSize(), bits);
			IsrImageProcessor.Fill_rectangle_color_(shape, iconArea, Color.black);
			iconImage = IsrImageProcessor._MakeImage_Shape_(iconImage, shape);
		}
		return iconImage;
	}
	/**
	 * Answer image attributes.
	 * (Smalltalk version used assert:do:ensure method.)
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage imageAttributes() {
		StImage anImage = null;
		Graphics gc = null;
		Rectangle box = null;
		int size = this.defaultSize();
		Image image = SystemResourceSupport.createImage(size * 4 + 5, size * 3 + 4);
		try {
			gc = image.getGraphics();
			gc.setColor(Color.gray);
			gc.fillRect(0, 0, size * 4 + 5, size * 3 + 4);
			box = this.iconImage().bounds();
			box.translate(size * 0 - box.x, size * 0 - box.y);
			this.iconImage().displayOn_at_(gc, new Point(box.x + 1, box.y + 1));
			box.translate(size * 1 - box.x, size * 0 - box.y);
			this.grayImage().displayOn_at_(gc, new Point(box.x + 2, box.y + 1));
			box.translate(size * 2 - box.x, size * 0 - box.y);
			this.ySpectrum().displayOn_at_(gc, new Point(box.x + 3, box.y + 1));
			box.translate(size * 3 - box.x, size * 0 - box.y);
			this.binaryImage().displayOn_at_(gc, new Point(box.x + 4, box.y + 1));
			box.translate(size * 0 - box.x, size * 1 - box.y);
			this.imageNineColors().displayOn_at_(gc, new Point(box.x + 1, box.y + 2));
			box.translate(size * 1 - box.x, size * 1 - box.y);
			this.xSpectrum().displayOn_at_(gc, new Point(box.x + 2, box.y + 2));
			box.translate(size * 2 - box.x, size * 1 - box.y);
			this.imageCenterSpectra().displayOn_at_(gc, new Point(box.x + 3, box.y + 2));
			box.translate(size * 3 - box.x, size * 1 - box.y);
			this.renderingImage().displayOn_at_(gc, new Point(box.x + 4, box.y + 2));
			box.translate(size * 0 - box.x, size * 2 - box.y);
			this.imageTypicalColor().displayOn_at_(gc, new Point(box.x + 1, box.y + 3));
			box.translate(size * 1 - box.x, size * 2 - box.y);
			this.imageTypicalGray().displayOn_at_(gc, new Point(box.x + 2, box.y + 3));
			box.translate(size * 2 - box.x, size * 2 - box.y);
			this.imageAverageColor().displayOn_at_(gc, new Point(box.x + 3, box.y + 3));
			box.translate(size * 3 - box.x, size * 2 - box.y);
			this.imageAverageGray().displayOn_at_(gc, new Point(box.x + 4, box.y + 3));
			anImage = new StImage(image);
		} finally {
			gc.dispose();
			image.flush();
		}
		return anImage;
	}
	/**
	 * Answer average color image.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage imageAverageColor() {
		StImage image = new StImage(this.iconImage().width(), this.iconImage().height());
		IsrImageProcessor.Fill_rectangle_color_(image, image.bounds(), Color.white);
		IsrImageProcessor.Fill_rectangle_color_(image, this.iconArea(), this.averageColor());
		return image;
	}
	/**
	 * Answer average gray color image.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage imageAverageGray() {
		StImage image = new StImage(this.iconImage().width(), this.iconImage().height());
		IsrImageProcessor.Fill_rectangle_color_(image, image.bounds(), Color.white);
		IsrImageProcessor.Fill_rectangle_color_(image, this.iconArea(), this.averageGray());
		return image;
	}
	/**
	 * Answer center spectra image.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage imageCenterSpectra() {
		Rectangle box = new Rectangle(Math.round((float) this.centerSpectra().x() * (this.iconImage().bounds().x + this.iconImage().bounds().width)), Math.round((float) this.centerSpectra().y() * (this.iconImage().bounds().y + this.iconImage().bounds().height)), 0, 0);
		box = new Rectangle(box.x - 2, box.y - 2, box.width + 4, box.height + 4);
		StImage image = new StImage(this.iconImage().width(), this.iconImage().height());
		IsrImageProcessor.Fill_rectangle_color_(image, image.bounds(), Color.white);
		IsrImageProcessor.Fill_rectangle_color_(image, box, Color.black);
		box = new Rectangle(box.x + 1, box.y + 1, box.width - 2, box.height - 2);
		IsrImageProcessor.Fill_rectangle_color_(image, box, Color.white);
		return image;
	}
	/**
	 * Answer my image folder.
	 * 
	 * @return sra.isr.image.objects.IsrImageFolder
	 */
	public IsrImageFolder imageFolder() {
		return this.imageFolder;
	}
	/**
	 * Answer nine colors image.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage imageNineColors() {
		StImage image = new StImage(this.iconImage().width(), this.iconImage().height());
		IsrImageProcessor.Fill_rectangle_color_(image, image.bounds(), Color.white);
		for (int i = 0; i < this.nineAreas().length; i++) {
			Rectangle area = this.nineAreas()[i];
			Color color = this.nineColors()[i];
			IsrImageProcessor.Fill_rectangle_color_(image, area, color);
		}
		return image;
	}
	/**
	 * Answer trace polylines image.
	 * (Smalltalk version used assert:do:ensure method.)
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage imageTracePolylines() {
		Graphics pixmap = null;
		StImage anImage = null;
		Object[] polylines = this.tracePolylines();
		Image image = SystemResourceSupport.createImage(this.iconImage().width(), this.iconImage().height());
		try {
			pixmap = image.getGraphics();
			for (int i = 0; i < polylines.length; i++) {
				Point[] points = (Point[]) polylines[i];
				int[] xPoints = new int[points.length];
				int[] yPoints = new int[points.length];
				for (int j = 0; j < points.length; j++) {
					xPoints[j] = points[j].x;
					yPoints[j] = points[j].y;
				}
				pixmap.drawPolyline(xPoints, yPoints, xPoints.length);
			}
			anImage = new StImage(image);
		} finally {
			pixmap.dispose();
			image.flush();
		}
		return anImage;
	}
	/**
	 * Answer typical color image.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage imageTypicalColor() {
		StImage image = new StImage(this.iconImage().width(), this.iconImage().height());
		IsrImageProcessor.Fill_rectangle_color_(image, image.bounds(), Color.white);
		IsrImageProcessor.Fill_rectangle_color_(image, this.iconArea(), this.typicalColor());
		return image;
	}
	/**
	 * Answer typical gray color image.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage imageTypicalGray() {
		StImage image = new StImage(this.iconImage().width(), this.iconImage().height());
		IsrImageProcessor.Fill_rectangle_color_(image, image.bounds(), Color.white);
		IsrImageProcessor.Fill_rectangle_color_(image, this.iconArea(), this.typicalGray());
		return image;
	}
	/**
	 * Make my variables.
	 */
	public void makeCaches() {
		this.iconImage();
		this.iconArea();
		this.nineColors();
		this.averageColor();
		this.typicalColor();
		this.centerSpectra();
		this.xSpectrum();
		this.ySpectrum();
		this.renderingImage();
	}
	/**
	 * Answer my image file modified date.
	 * 
	 * @return long
	 */
	public long modifiedDate() {
		return this.modifiedDate;
	}
	/**
	 * Answer my nine area's rectangles.
	 * 
	 * @return java.awt.Rectangle[]
	 */
	public Rectangle[] nineAreas() {
		Isr2dPoint size = new Isr2dPoint(this.iconArea().width / 3.0d, this.iconArea().height / 3.0d);
		Point[] array = new Point[4];
		array[0] = new Point(Math.round((float) size.x() * 0), Math.round((float) size.y() * 0));
		array[1] = new Point(Math.round((float) size.x() * 1), Math.round((float) size.y() * 1));
		array[2] = new Point(Math.round((float) size.x() * 2), Math.round((float) size.y() * 2));
		array[3] = new Point(Math.round((float) size.x() * 3), Math.round((float) size.y() * 3));
		for (int index = 0; index < 4; index++) {
			array[index] = new Point(Math.min(array[index].x, this.iconArea().width), Math.min(array[index].y, this.iconArea().height));
		}
		Rectangle[] collection = new Rectangle[9];
		for (int j = 0; j < array.length - 1; j++) {
			for (int i = 0; i < array.length - 1; i++) {
				Point origin = new Point(array[i].x, array[j].y);
				Point corner = new Point(array[i + 1].x, array[j + 1].y);
				Rectangle box = new Rectangle(origin.x, origin.y, corner.x - origin.x, corner.y - origin.y);
				box.translate(this.iconArea().x, this.iconArea().y);
				collection[j * 3 + i] = box;
			}
		}
		return collection;
	}
	/**
	 * Answer nine colors on my image.
	 * 
	 * @return java.awt.Color[]
	 */
	public Color[] nineColors() {
		if (nineColors == null) {
			StImage figure = this.iconImage();
			Color[] collection = new Color[9];
			Rectangle[] nineAreas = this.nineAreas();
			for (int i = 0; i < nineAreas.length; i++) {
				Rectangle area = nineAreas[i];
				StImage image = new StImage(area.width, area.height);
				if ((image.bounds().width * image.bounds().height) > 0) {
					image.copy_from_in_rule_(image.bounds(), new Point(area.x, area.y), figure, StImage.Over);
				}
				Object[] histogram = IsrImageProcessor.ColorHistogram_(image);
				double max = 0.0d;
				Color color = null;
				Color key = null;
				double value;
				for (int j = 0; j < histogram.length; j++) {
					StAssociation assoc = (StAssociation) histogram[j];
					value = ((Double) assoc.value()).doubleValue();
					key = (Color) assoc.key();
					if ((value > max) && (!StColorValue._IsGray(key))) {
						color = key;
						max = value;
					}
				}
				if (color == null) {
					max = 0.0d;
					color = null;
					for (int k = 0; k < histogram.length; k++) {
						StAssociation assoc = (StAssociation) histogram[k];
						value = ((Double) assoc.value()).doubleValue();
						key = (Color) assoc.key();
						if (value > max) {
							color = key;
							max = value;
						}
					}
				}
				if (color == null) {
					color = Color.white;
				}
				collection[i] = color;
			}
			nineColors = collection;
		}
		return nineColors;
	}
	/**
	 * Answer my image from file.
	 * (Smalltalk version used assert:do:ensure method.)
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage originalImage() {
		StImage anImage = null;
		String string = this.asFilename().getAbsolutePath();
		if (string.toLowerCase().endsWith(".jpg")) {
			Frame frame = new Frame();
			Toolkit toolKit = frame.getToolkit();
			Image image = toolKit.getImage(string);
			MediaTracker mt = new MediaTracker(frame);
			mt.addImage(image, 0);
			try {
				mt.waitForAll();
			} catch (InterruptedException e) {
				System.err.println("image load interrupted.");
				return null;
			}
			int status = mt.statusAll(false);
			if ((status & MediaTracker.ABORTED) != 0 || (status & MediaTracker.ERRORED) != 0) {
				System.err.println("could not load image.");
				return null;
			}
			anImage = new StImage(image);
			image = null;
		} else {
			System.err.println("unknown image format");
			return null;
		}
		return anImage;
	}
	/**
	 * Answer rendering image.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage renderingImage() {
		if (renderingImage == null) {
			renderingImage = this.iconImage()._convertToPalette_RenderedByErrorDiffusion(IsrImageProcessor._WhiteBlackPalette());
		}
		return renderingImage;
	}
	/**
	 * Set filename and imageFolder.
	 * 
	 * @return sra.isr.image.objects.IsrImageObject
	 * @param aString java.lang.String
	 * @param anImageFolder sra.isr.image.objects.IsrImageFolder
	 */
	private IsrImageObject setFileName_imageFolder_(String aString, IsrImageFolder anImageFolder) {
		fileName = aString;
		imageFolder = anImageFolder;
		modifiedDate = this.asFilename().lastModified();
		return null;
	}
	/**
	 * Set my imageFolder.
	 * 
	 * @param anImageFolder sra.isr.image.objects.IsrImageFolder
	 */
	public void setImageFolder_(IsrImageFolder anImageFolder) {
		imageFolder = anImageFolder;
	}
	/**
	 * Answer shape image.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage shapeImage() {
		StImage image = IsrImageProcessor.Shape_(this.iconImage());
		return image;
	}
	/**
	 * Answer thin image.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage thinImage() {
		StImage image = IsrImageProcessor.MakeThin_(this.iconImage());
		return image;
	}
	/**
	 * Answer my image trace polylines.
	 * 
	 * @return java.lang.Object[]
	 */
	public Object[] tracePolylines() {
		Object[] collection = IsrImageProcessor.TraceBorder_(this.iconImage());
		Vector polylines = new Vector();
		for (int i = 0; i < collection.length; i++) {
			Point[] polyline = (Point[]) collection[i];
			if (polyline.length > 4) {
				polylines.addElement(polyline);
			}
		}
		Object[] array = new Object[polylines.size()];
		polylines.copyInto(array);
		return array;
	}
	/**
	 * Answer typical color on my image.
	 * 
	 * @return java.awt.Color
	 */
	public Color typicalColor() {
		if (typicalColor == null) {
			int red = 0;
			int green = 0;
			int blue = 0;
			Color[] nineColors = this.nineColors();
			for (int i = 0; i < nineColors.length; i++) {
				Color color = nineColors[i];
				red = red + color.getRed();
				green = green + color.getGreen();
				blue = blue + color.getBlue();
			}
			typicalColor = new Color(Math.round(((float) red) / 9), Math.round(((float) green) / 9), Math.round(((float) blue) / 9));
		}
		return typicalColor;
	}
	/**
	 * Answer typical gray color on my image.
	 * 
	 * @return java.awt.Color
	 */
	public Color typicalGray() {
		return StColorValue.Brightness_(StColorValue._GetBrightness(this.typicalColor()));
	}
	/**
	 * Answer x-spectrum on my image.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage xSpectrum() {
		if (xSpectrum == null) {
			Object[] spectrum = IsrImageProcessor.XSpectrum_(this.iconImage());
			xSpectrum = this.xSpectrum_height_(spectrum, this.iconImage().height());
		}
		return xSpectrum;
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * (Smalltalk version used assert:do:ensure method.)
	 * 
	 * @return sra.smalltalk.StImage
	 * @param spectrum (StAssociation[]) Object[]
	 * @param height int
	 */
	private StImage xSpectrum_height_(Object[] spectrum, int height) {
		Graphics gc = null;
		StImage image = null;
		Image pixmap = SystemResourceSupport.createImage(spectrum.length, height);
		try {
			gc = pixmap.getGraphics();
			for (int i = 0; i < spectrum.length; i++) {
				StAssociation assoc = (StAssociation) spectrum[i];
				int x = ((Integer) assoc.key()).intValue();
				int value = Math.round((((Double) assoc.value()).floatValue()) * height);
				gc.setColor(Color.black);
				gc.fillRect(x, 0, 1, value);
			}
			image = new StImage(pixmap);
		} finally {
			gc.dispose();
			pixmap.flush();
		};
		return image;
	}
	/**
	 * Answer y-spectrum on my image.
	 * 
	 * @return sra.smalltalk.StImage
	 */
	public StImage ySpectrum() {
		if (ySpectrum == null) {
			Object[] spectrum = IsrImageProcessor.YSpectrum_(this.iconImage());
			ySpectrum = this.ySpectrum_width_(spectrum, this.iconImage().width());
		}
		return ySpectrum;
	}
	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * (Smalltalk version used assert:do:ensure method.)
	 * 
	 * @return sra.smalltalk.StImage
	 * @param spectrum (StAssociation[]) Object[]
	 * @param height int
	 */
	private StImage ySpectrum_width_(Object[] spectrum, int width) {
		Graphics gc = null;
		StImage image = null;
		Image pixmap = SystemResourceSupport.createImage(width, spectrum.length);
		try {
			gc = pixmap.getGraphics();
			for (int i = 0; i < spectrum.length; i++) {
				StAssociation assoc = (StAssociation) spectrum[i];
				int y = ((Integer) assoc.key()).intValue();
				int value = Math.round((((Double) assoc.value()).floatValue()) * width);
				gc.setColor(Color.black);
				gc.fillRect(0, y, value, 1);
			}
			image = new StImage(pixmap);
		} finally {
			gc.dispose();
			pixmap.flush();
		}
		return image;
	}
}
