package sra.isr.geometry.basic;

import java.io.*;
import sra.smalltalk.*;

/**
 * Isr2dPoint class
 * 
 * 	@author:        nisinaka
 * 	@created:       1998/09/29 (by nisinaka)
 * 	@updated:       2000/02/16 (by Mitsuhiro Asada)
 * 	@ImageSearcher: 019
 * 	@JDK:           1.1.6 or higher
 * 	@copyright:     2000 SRA (Software Research Associates, Inc.)
 * 
 * 	$Id: Isr2dPoint.java,v 1.4 2000/04/07 04:24:15 m-asada Exp $
 */
public class Isr2dPoint extends IsrPoint implements Serializable {
	/**
	 * The <i>x</i> coordinate.
	 */
	protected double x;

	/**
	 * The <i>y</i> coordinate.
	 */
	protected double y;
	/**
	 * Create a new Isr2dPoint.
	 *
	 * @param x double
	 * @param y double
	 */
	public Isr2dPoint(double x, double y) {
		super();
		this.x = x;
		this.y = y;
	}
	/**
	 * Answer the absolute Point.
	 *
	 * @return sra.jun.geometry.basic.IsrPoint
	 */
	public IsrPoint abs() {
		return new Isr2dPoint(Math.abs(x), Math.abs(y));
	}
	/**
	 * Answer the receiver as an array of double.
	 *
	 * @return double[]
	 */
	public double[] asArray() {
		return new double[] {x, y};
	}
	/**
	 * Answer the distance from aPoint.
	 *
	 * @return double
	 * @param aPoint sra.jun.geometry.basic.Isr2dPoint
	 */
	public double distance_(Isr2dPoint aPoint) {
		return this.minus_(aPoint).rho();
	}
	/**
	 * Answer the result of dividing this point by delta.
	 * This method corresponds to the methd '/' in Smalltalk.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 * @param delta double
	 */
	public Isr2dPoint dividedBy_(double delta) {
		return new Isr2dPoint(this.x / delta, this.y / delta);
	}
	/**
	 * Answer the result of dividing this point by aPoint.
	 * This method corresponds to the methd '/' in Smalltalk.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 * @param aPoint sra.jun.geometry.basic.Isr2dPoint
	 */
	public Isr2dPoint dividedBy_(Isr2dPoint aPoint) {
		return new Isr2dPoint(this.x / aPoint.x, this.y / aPoint.y);
	}
	/**
	 * Answer the result of a dot product with aPoint.
	 *
	 * @return double
	 * @param aPoint sra.jun.geometry.basic.Isr2dPoint
	 */
	public double dotProduct_(Isr2dPoint aPoint) {
		Isr2dPoint newPoint = this.multipliedBy_(aPoint);
		return newPoint.x + newPoint.y;
	}
	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 *
	 * @return boolean
	 * @param anObject java.lang.Object
	 */
	public boolean equals(Object anObject) {
		if (anObject instanceof Isr2dPoint == false) {
			return false;
		}
		Isr2dPoint aPoint = (Isr2dPoint) anObject;
		return this.x == aPoint.x && this.y == aPoint.y;
	}
	/**
	 * Create a new instance of Isr2dPoint and initialize it
	 * with an array of double.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 * @param anArray double[]
	 */
	public static Isr2dPoint FromArray_(double[] anArray) {
		return new Isr2dPoint(anArray[0], anArray[1]);
	}
	/**
	 * Answer a new Isr2dPoint created as a copy of another point.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 */
	public static Isr2dPoint FromPoint_(Isr2dPoint aPoint) {
		return new Isr2dPoint(aPoint.x, aPoint.y);
	}
	/**
	 * Answer the number representing the ordering of the receiver in
	 * the generality hierarchy.
	 *
	 * @return int
	 */
	public int generality() {
		return 200;
	}
	// Hash Function

	/**
	 * Computes a hash code for this object.
	 *
	 * @return int
	 */
	public int hashCode() {
		long bitsX = Double.doubleToLongBits(x);
		int xHash = (int) (bitsX ^ (bitsX >> 32));
		long bitsY = Double.doubleToLongBits(y);
		int yHash = (int) (bitsY ^ (bitsY >> 32));
		return ((xHash >> 2) | yHash);
	}
	/**
	 * Answer the result of the inner product with aPoint.
	 *
	 * @return double
	 * @param aPoint sra.jun.geometry.basic.Isr2dPoint
	 */
	public double innerProduct_(Isr2dPoint aPoint) {
		return this.dotProduct_(aPoint);
	}
	/**
	 * Answer true if this is a kind of 2D geometry object, otherwise false.
	 * @return boolean
	 */
	public boolean is2d() {
		return true;
	}
	/**
	 * Answer true if the receiver is less than aPoint.
	 * This method corresponds to the method "<" in Smalltalk.
	 *
	 * @return boolean
	 * @param aPoint sra.jun.geometry.basic.Isr2dPoint
	 */
	public boolean isLessThan_(Isr2dPoint aPoint) {
		return this.x < aPoint.x && this.y < aPoint.y;
	}
	/**
	 * Answer true if the receiver is less than or equal to aPoint.
	 * This method corresponds to the method "<=" in Smalltalk.
	 *
	 * @return boolean
	 * @param aPoint sra.jun.geometry.basic.Isr2dPoint
	 */
	public boolean isLessThanOrEqualTo_(Isr2dPoint aPoint) {
		return this.x <= aPoint.x && this.y <= aPoint.y;
	}
	/**
	 * Answer true if this point is ZERO, otherwise false.
	 * @return boolean
	 */
	public boolean isZero() {
		return this.x == 0.0d && this.y == 0.0d;
	}
	/**
	 * Set the length from the origin.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 * @param aNumber double
	 */
	public Isr2dPoint length_(double aNumber) {
		double length = this.length();
		if (length <= IsrPoint.ACCURACY) {
			return this;
		}
		return this.multipliedBy_(aNumber / length);
	}
	/**
	 * Answer the maximum point.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 * @param aPoint sra.jun.geometry.basic.Isr2dPoint
	 */
	public Isr2dPoint max_(Isr2dPoint aPoint) {
		return new Isr2dPoint(Math.max(this.x, aPoint.x), Math.max(this.y, aPoint.y));
	}
	/**
	 * Answer the minimum point.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 * @param aPoint sra.jun.geometry.basic.Isr2dPoint
	 */
	public Isr2dPoint min_(Isr2dPoint aPoint) {
		return new Isr2dPoint(Math.min(this.x, aPoint.x), Math.min(this.y, aPoint.y));
	}
	/**
	 * Answer the difference between this point and delta.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 * @param delta double
	 */
	public Isr2dPoint minus_(double delta) {
		return new Isr2dPoint(this.x - delta, this.y - delta);
	}
	/**
	 * Answer the difference between this point and aPoint.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 * @param aPoint sra.jun.geometry.basic.Isr2dPoint
	 */
	public Isr2dPoint minus_(Isr2dPoint aPoint) {
		return new Isr2dPoint(this.x - aPoint.x, this.y - aPoint.y);
	}
	/**
	 * Answer the result of multiplying this point by aPoint.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 * @param delta double
	 */
	public Isr2dPoint multipliedBy_(double delta) {
		return new Isr2dPoint(this.x * delta, this.y * delta);
	}
	/**
	 * Answer the result of multiplying this point by aPoint.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 * @param aPoint sra.jun.geometry.basic.Isr2dPoint
	 */
	public Isr2dPoint multipliedBy_(Isr2dPoint aPoint) {
		return new Isr2dPoint(this.x * aPoint.x, this.y * aPoint.y);
	}
	/**
	 * Answer the negation of the receiver.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 */
	public Isr2dPoint negated() {
		return Isr2dPoint.Zero().minus_(this);
	}
	/**
	 * Answer the sum of this point and delta.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 * @param delta double
	 */
	public Isr2dPoint plus_(double delta) {
		return new Isr2dPoint(this.x + delta, this.y + delta);
	}
	/**
	 * Answer the sum of this point and aPoint.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 * @param aPoint sra.jun.geometry.basic.Isr2dPoint
	 */
	public Isr2dPoint plus_(Isr2dPoint aPoint) {
		return new Isr2dPoint(this.x + aPoint.x, this.y + aPoint.y);
	}
	/**
	 * Print my string representation on aWriter.
	 *
	 * @param aWriter java.io.Writer
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("(");
		aWriter.write(String.valueOf(this.x));
		aWriter.write(" , ");
		aWriter.write(String.valueOf(this.y));
		aWriter.write(")");
	}
	/**
	 * Answer the product of the Isr2dPoint and the receiver.
	 *
	 * @return double
	 * @param aPoint sra.jun.geometry.basic.Isr2dPoint
	 */
	public double product_(Isr2dPoint aPoint) {
		return x * aPoint.y - y * aPoint.x;
	}
	/**
	 * Answer the rho as of a polar coordinate.
	 *
	 * @return double
	 */
	public double rho() {
		return Math.sqrt(this.dotProduct_(this));
	}
	/**
	 * Create a new Isr2dPoint with rho and theta.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 * @param rho double
	 * @param theta double
	 */
	public static Isr2dPoint Rho_theta_(double rho, double theta) {
		return new Isr2dPoint(rho * Math.cos(theta), rho * Math.sin(theta));
	}
	/**
	 * Answer the rounded Isr2dPoint.
	 * @return Isr2dPoint
	 */
	public Isr2dPoint rounded() {
		return new Isr2dPoint((double) Math.round(this.x), (double) Math.round(this.y));
	}
	/**
	 * Answer the truncated Isr2dPoint.
	 * @return Isr2dPoint
	 */
	public Isr2dPoint truncated() {
		double newX, newY;
		newX = this.x >= 0 ? Math.ceil(this.x) : Math.floor(this.x);
		newY = this.y >= 0 ? Math.ceil(this.y) : Math.floor(this.y);
		return new Isr2dPoint(newX, newY);
	}
	/**
	 * Answer the unit vector derived from this point.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 */
	public Isr2dPoint unitVector() {
		return this.length_(1.0d);
	}
	/**
	 * Answer a new Isr2dPoint with unit values.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 */
	public static Isr2dPoint Unity() {
		return new Isr2dPoint(1.0d, 1.0d);
	}
	/**
	 * Answer the x coordinate.
	 *
	 * @return double
	 */
	public double x() {
		return this.x;
	}
	/**
	 * Answer the y coordinate.
	 *
	 * @return double
	 */
	public double y() {
		return this.y;
	}
	/**
	 * Answer a new Isr2dPoint with zero values.
	 *
	 * @return sra.jun.geometry.basic.Isr2dPoint
	 */
	public static Isr2dPoint Zero() {
		return new Isr2dPoint(0.0d, 0.0d);
	}
}
