import unittest
import e4py
import sys


DB_NAME = 'temp.db'


class VertexTest(unittest.TestCase):

	def setUp(self):
		self.storage = e4py.Storage(DB_NAME)
		self.storage.state = self.storage.state & ~e4py.E4_COMMITATCLOSE
		
	def testDetach(self):
		s = self.storage
		r = s.root
		self.assert_(r.VertexCount() == 0)
		r['a'] = 1
		self.assert_(r.VertexCount() == 1)
		v = r.GetVertexRef('a')
		v.Detach()
		self.assert_(r.VertexCount() == 0)
		
	def testSetNode(self):
		s = self.storage
		r = s.root
		r['a'] = 1
		v = r.GetVertexRef('a')
		n = v.SetNode()
		self.assert_(v.type == e4py.E4_VTNODE)
		
	def testCountWithName(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		r.AddVertex('a', e4py.E4_IOLAST, 0, 2)
		r.AddVertex('a', e4py.E4_IOLAST, 0, 3)
		v = r.GetVertexRef(('a', 2))
		self.assert_(v.CountWithName() == 2)
		
	def testTotalCountWithName(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		r.AddVertex('a', e4py.E4_IOLAST, 0, 2)
		r.AddVertex('a', e4py.E4_IOLAST, 0, 3)
		v = r.GetVertexRef(('a', 2))
		self.assert_(v.TotalCountWithName() == 3)

	def testCountWithType(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		r.AddVertex('b', e4py.E4_IOLAST, 0, "2")
		r.AddVertex('c', e4py.E4_IOLAST, 0, 3.)
		r.AddVertex('d', e4py.E4_IOLAST, 0, 4)
		v = r.GetVertexRef('d')
		self.assert_(v.CountWithType() == 2)
		
	def testTotalCountWithType(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		r.AddVertex('b', e4py.E4_IOLAST, 0, "2")
		r.AddVertex('c', e4py.E4_IOLAST, 0, 3.)
		r.AddVertex('d', e4py.E4_IOLAST, 0, 4)
		v = r.GetVertexRef('a')
		self.assert_(v.TotalCountWithType() == 2)
		
	def testGetNode(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		v = r.GetVertexRef('a')
		self.assert_(v.GetNode() == r)
		
	def testMoveVertex(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		r.AddVertex('b', e4py.E4_IOLAST, 0, "2")
		r.AddVertex('c', e4py.E4_IOLAST, 0, 3.)
		v = r.GetVertexRef('a')
		self.assert_(v.rank == 1)
		v.MoveVertex(r.GetVertexRef('c'), e4py.E4_IOAT, 0)
		self.assert_(v.rank == 2)
		
	def testNext(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		r.AddVertex('b', e4py.E4_IOLAST, 0, "2")
		v = r.GetVertexRef('a')
		self.assert_(v.Next(1) == r.GetVertexRef('b'))

	def testPrev(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		r.AddVertex('b', e4py.E4_IOLAST, 0, "2")
		v = r.GetVertexRef('b')
		self.assert_(v.Prev(1) == r.GetVertexRef('a'))

	def testisDetached(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		v = r.GetVertexRef('a')
		self.assert_(not v.isDetached)
		
	def testvalue(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		v = r.GetVertexRef('a')
		self.assert_(v.value == 1)
		
	def testname(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		v = r.GetVertexRef('a')
		self.assert_(v.name == 'a')
		
	def teststorage(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		v = r.GetVertexRef('a')
		self.assert_(v.storage == s)
		
	def testroot(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		v = r.GetVertexRef('a')
		self.assert_(v.root == r)
		
	def testuid(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		v = r.GetVertexRef('a')
		self.assert_(v.uid != 0)

	def testrank(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		r.AddVertex('b', e4py.E4_IOLAST, 0, "2")
		v = r.GetVertexRef('b')
		self.assert_(v.rank == 2)
		
	def testuserData(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		v = r.GetVertexRef('a')
		v.userData = 2
		self.assert_(v.userData == 2)
		
	def testtype(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		v = r.GetVertexRef('a')
		self.assert_(v.type == e4py.E4_VTINT)

	def testkind(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		v = r.GetVertexRef('a')
		self.assert_(v.kind == e4py.E4_RKVERTEX)
		
	def testvalid(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		v = r.GetVertexRef('a')
		self.assert_(v.valid)

	def testtempUID(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		v = r.GetVertexRef('a')
		self.assert_(v.tempUID != 0)
		
	def testtransientUserData(self):
		s = self.storage
		r = s.root
		r.AddVertex('a', e4py.E4_IOLAST, 0, 1)
		v = r.GetVertexRef('a')
		v.transientUserData = 2
		self.assert_(v.transientUserData == 2)

	def tearDown(self):
		try:		#delete storage, ignore if none exists
			self.storage.Delete()
		except:
			pass
		self.storage = None


def makeTestSuite():
	classes = (VertexTest, )
	suites = []
	for c in classes:
		suites.append(unittest.makeSuite(c,'test'))
	return unittest.TestSuite(suites)


if __name__=='__main__':
	runner = unittest.TextTestRunner(stream=sys.stdout)
	unittest.main(defaultTest='makeTestSuite', testRunner=runner)

