package  com.db4o.sql;

import  com.db4o.*;
import  com.db4o.jgen.*;
import  com.db4o.lib.*;
import  java.io.*;
import  java.sql.*;


/**
 * imports data from a datasource that is unknown to the db4o system.
 * For every table a .Java file will be created in the package folder
 * specified as parameter. The .Java files will be automatically compiled
 * if javac is found on your system. For every row in the database, an object
 * will be created within db4o.
 *
 * Should the compilation fail, in case you don't have javac in your path,
 * you can compile the generated files manually and start
 * <A HREF="SqlImportContinue.html">SqlImportContinue</A> afterwards.
 *
 * <br><br>
 * <b>Note:</b> An import run imports all data. A check for duplicates does
 * not take place yet. Be careful to call SqlImport only once.
*/
public class SqlImport {

    /**
	 * starts SqlImport
     * @param arguments
	 * <br>1.) JDBC driver<br>
	 * 2.) JDBC connection String<br>
	 * 3.) destination db4o datafile<br>
	 * 4.) the package name to be used for the created .Java files.
 	 * 5.) optional user for the JDBC connection
	 * 6.) optional password for the JDBC connection
     */
    public static void main (String[] args) {
        try {
            SqlConfiguration config = new SqlConfiguration(args, true);
            try {
                Timer.start("SqlImport completed");
                Connection con = Sql.connect(config);
                if (con != null) {
                    List4 col = createJavaFiles(config, con);
                    try {
                        compile(config, col);
                    } catch (Exception e) {
                        Sql.log("SqlImport: compilation failed.");
                        Sql.log("Compile all classes in " + config.objectPath()
                                + " manually.");
                        Sql.log("Run com.db4o.sql.SqlImportContinue to restart import.");
                        return;
                    }
                    toDb4o(config, con);
                    Timer.stop("SqlImport completed");
                    con.close();
                }
            } catch (Exception e) {
                Sql.exception("SQLImport raises Exception:");
                Sql.exception(e.getMessage());
            }
        } catch (Exception e) {}
    }

    /**
     * @param a_config
     */
    static void continueImport (SqlConfiguration a_config) {
        Timer.start("SqlImport completed");

        Connection con = Sql.connect(a_config);
        toDb4o(a_config, con);
        Timer.stop("SqlImport completed");
    }

    /**
     * @param a_Config
     * @param a_connection
     * @return
     * @exception Exception
     */
    static List4 createJavaFiles (SqlConfiguration a_Config, Connection a_connection) throws Exception {
        JPackage l_package = new JPackage(a_Config.getPackage());
        List4 tables = Sql.getTables(a_connection, true);
        List4 classes = new List4();
        StringList listGenerator = new StringList();
        Iterator4 i = tables.iterator();
        while (i.hasNext()) {
            SqlTable table = (SqlTable)i.next();
			classes.add(table.createJavaFile(l_package));
            listGenerator.add(table.getName());
        }
        // Configuration File Tables.TXT
        String l_fileName = l_package.getPath() + Sql.TABLE_FILE_NAME;
        new java.io.File(l_fileName).delete();
        new RandomAccessFile(l_fileName, "rw").writeBytes(listGenerator.getSeparatedList(","));
        return classes;

		/*  use the following code to define links

         l_rs = m.getImportedKeys(null,null,"ADR");
         while ( l_rs.next()){
         System.out.println(	l_rs.getString(3));
         System.out.println(	l_rs.getString(4));
         }

         l_rs = m.getExportedKeys(null,null,"ADR");
         while ( l_rs.next()){
         System.out.println(	l_rs.getString(7));
         System.out.println(	l_rs.getString(8));
         }
         */

    }

    /**
     * @param a_Config
     * @param a_Classes
     * @exception Exception
     */
    static void compile (SqlConfiguration a_Config, List4 a_Classes) throws Exception {
        Iterator4 i = a_Classes.iterator();
        JClass l_Class;
        while (i.hasNext()) {
            l_Class = (JClass)i.next();
            Sql.log("Compiling: " + l_Class.getName() + ".Java");
            l_Class.compile();
        }
    }

    /**
     * @param a_Config
     * @param a_connection
     */
    static void toDb4o (SqlConfiguration a_Config, Connection a_connection) {
        ObjectContainer db = Db4o.openFile(a_Config.getDbFile());
        Statement l_Statement = Sql.createStatement(a_connection);
        StringList l_Tables = new StringList();
        l_Tables.parseSeparatedList(readConfigurationFile(a_Config), ",");
        Iterator4 i = l_Tables.iterator();
        while (i.hasNext()) {
            SqlTable table = SqlTable.analyze(a_Config.getPackage() + "." +
                    (String)i.next(), false);
            Iterator4 j = table.iterateFields();
            Sql.log("Processing Table: " + table.getName());
            ResultSet rs = Sql.executeQuery(l_Statement, table.selectStatement());
            try {
                while (rs.next()) {
                    j.reset();
                    Object newObject = table.newObject();
                    while (j.hasNext()) {
                        ((SqlField)j.next()).instantiateOn(newObject, rs);
                    }
                    db.set(newObject);
                }
            } catch (Exception e) {
                throw  new RuntimeException("SqlToDb4o.go Object could not be instantiated: "
                        + e.getMessage());
            }
        }
        db.close();
    }

    /**
     * @param a_Config
     * @return
     */
    static String readConfigurationFile (SqlConfiguration a_Config) {
        try {
            return  new RandomAccessFile(a_Config.objectPath() + Sql.TABLE_FILE_NAME,
                    "r").readLine();
        } catch (Exception e) {
            throw  new RuntimeException("SqlToDb4o.readConfigurationFile failed. File:"
                    + a_Config.objectPath() + Sql.TABLE_FILE_NAME);
        }
    }
}




