package com.db4o.samples.rename;

import com.db4o.*;
import com.db4o.config.*;
import com.db4o.tools.*;

/**
 * This code is intended to be used to experiment renaming classes
 * and fields.
 * 
 * Attention! This code deletes the file "rename.yap"
 * 
 * Recommended usage:
 * Run this code and look at the output.
 * Set the createFile variable to false and start experimenting yourself,
 * by changing the fieldnames in the classes RenameMe and RenameTo and
 * by modifying the configuration code.
 * 
 * Should you rename classes and fields in production environments, 
 * please *always* backup your database files.
 * 
 * Possible pitfalls to test with this code:
 * 1.) Be aware that you have to execute renaming code the first time 
 * you use classes with a changed structure. Otherwise db4o will detect 
 * the changed fields as "new" and create them in the first run. 
 * Test example:
 *   rename class fields only without running configuration code
 * Result:
 *   the old values will remain "hidden" in your objects. Renaming the
 * fields back to their old names will regain access to them.
 * 
 * 2.) If you execute renaming configuration code, without changing class
 * files you will get a similar result. Your values will be "hidden" in new
 * fields with new names. The "old" fields will be additionally detected and
 * created by db4o.
 * 
 * 3.) db4o will not permit renaming fields to names that previously 
 * existed for this database and are currently "hidden". You may additionally
 * change the names of "hidden" fields first to free the names.
 * 
 * 4.) Use fully qualified names (including packages) for renaming classes.
 * 
 * 5.) Renaming configuration code will only be executed once. db4o keeps
 * track of renaming processes and ensures that they will only be executed
 * once for each database file. This ensures that performance stays good
 * during the startup of db4o, even though you might deploy thousands of 
 * renaming statements with your application. Watch the output of renaming 
 * messages to see what is happening.
 * 
 * If these points all sound very complicated, please experiment thoroughly
 * with the following code. Everything should work the way you would expect.
 */
public class Rename
{
	static boolean createFile = true;  // set this variable to true after the first run
	
	static String fileName = "rename.yap";
	static String classNameMe = "com.db4o.samples.rename.RenameMe";
	static String classNameTo = "com.db4o.samples.rename.RenameTo";
	
	public static void main(String args []) throws Exception{

		Db4o.configure().setOut(System.out); // logging output for renaming processes is always a good idea
		
		if(createFile){
			createFileAndLogContent();
		}
		
		// configuration code
		ObjectClass oc = Db4o.configure().objectClass(classNameMe);
		oc.rename(classNameTo);
		oc.objectField("s1").rename("str1");
		oc.objectField("s2").rename("str2");
		
		openFileAndLogContent();
	}
	
	static void createFileAndLogContent() throws Exception{
		System.out.println("Rename.java.createFile is true. " + fileName + " deleted.");
		new java.io.File(fileName).delete();
		ObjectContainer con = Db4o.openFile(fileName);
		for(int i = 0; i < 2; i++){
			con.set(( (SetValues)
				Class.forName(classNameMe).newInstance()).
					setValues(i));
		}
		con.close();
		openFileAndLogContent();
	}
	
	static void openFileAndLogContent(){
		ObjectContainer con = Db4o.openFile(fileName);
		ObjectSet set = con.get(null);
		while(set.hasNext()){
			Logger.log(con,set.next());
		}
		con.close();
	}
}

