/*
 * (c) Copyright 2002 http://www.db4o.com
 * All Rights Reserved.
 */
package com.db4o.samples.reflect;

import com.db4o.*;
import com.db4o.reflect.*;

/**
 * test for custom reflection implementations.
 * <br><br>
 * db4o internally uses java.lang.reflect.* by default. On platforms that
 * do not support this package, customized implementations may be written
 * to supply all the functionality of the interfaces in the com.db4o.reflect
 * package. The sources in this sample packages demonstrate, how db4o
 * accesses the java.lang.reflect.* functionality.
 * <br><br>
 * This TestReflect method may be used to test, if you own implementation
 * provides the functionality that db4o needs. You may call the test from
 * the command line by specifying the classname of your own class that
 * implements IReflect. Alternatively you can call the test(IReflect) method.
 */
public class TestReflect {

	private boolean errors = false;
	private boolean complete = true;

	private IReflect reflector;
	private IClass classReflector;

	public static void main(String[] args) {
		Db4o.configure().messageLevel(-1);

		if (args.length > 0) {
			String className = args[0];
			Object reflector = null;
			try {
				reflector = Class.forName(className).newInstance();
				if (reflector instanceof IReflect) {
					TestReflect tr = new TestReflect();
					tr.test((IReflect) reflector);
				} else {
					log(
						"Class '"
							+ args[0]
							+ "' does not implement IReflect interface.");
				}
			} catch (InstantiationException e) {
			} catch (IllegalAccessException e) {
			} catch (ClassNotFoundException e) {
			}
			if (reflector == null) {
				log("Class could not be instantiated:'" + args[0] + "'");
			}
		} else {
			log("Pass the classname of your IReflect implementation as a parameter.");
		}
	}

	public void test(IReflect reflector) {
		this.reflector = reflector;
		String className = reflector.getClass().getName();
		log(
			"Testing IReflect implementation supplied in class '"
				+ className
				+ "'");
		testIReflect();
		log("----------------------------------------------------------");
		logSuccess(
			"Test of IReflect implementation '" + className + "'",
			!errors);
		if (!complete) {
			log("The implementation is not 100% complete. See the above log for details.");
		}

	}

	private void testIReflect() {
		logSupport("Method calls", reflector.methodCallsSupported());
		logSupport("Constructor calls", reflector.constructorCallsSupported());
		String className = TestReflectClass.class.getName();
		try {
			classReflector = reflector.forName(className);
		} catch (ClassNotFoundException e) {
			log("Class not found: '" + className + "'");
			e.printStackTrace();
		}
		logSuccess("Generation of IClass", classReflector != null);
		if (classReflector != null) {
			testIClass();
		}

		IArray array = reflector.array();
		logSupport("Arrays", (array != null));
		if (array != null) {
			testIArray();
		}
	}

	private void testIClass() {

		if (reflector.constructorCallsSupported()) {
			testIConstructor();
		}

		IField[] fields = classReflector.getDeclaredFields();
		_assert(
			fields.length == TestReflectClass.FIELD_COUNT,
			"getDeclaredFields");
		for (int i = 0; i < fields.length; i++) {
			_assert(fields != null, "getDeclaredFields[" + i + "] is valid");
			String fieldName = fields[i].getName();
			IField fieldReflector = classReflector.getDeclaredField(fieldName);
			_assert(
				fieldReflector != null,
				"getDeclaredField('" + fieldName + "') is valid");
		}

		testIField();

		try {
			IClass abstractReflector =
				reflector.forName(TestReflectAbstractClass.class.getName());
			_assert(abstractReflector.isAbstract(), "isAbstract");
			IClass interfaceReflector =
				reflector.forName(TestReflectInterface.class.getName());
			_assert(interfaceReflector.isInterface(), "isInterface");
			Object instance = classReflector.newInstance();
			_assert(instance != null, "newInstance");
		} catch (Exception e) {
			log("Unhandled Exception in testIClass");
			errors = true;
			e.printStackTrace();
		}

		if (reflector.methodCallsSupported()) {
			testIMethod();
		}
	}

	private void testIConstructor() {
		Object instance = null;
		try {
			String className = TestReflectConstructor.class.getName();
			IClass classReflector = reflector.forName(className);
			IConstructor[] constructors =
				classReflector.getDeclaredConstructors();
			if (constructors.length
				== TestReflectConstructor.CONSTRUCTOR_COUNT) {
				boolean stringIntConstructorFound = false;
				for (int i = 0; i < constructors.length; i++) {
					Class[] parametertypes =
						constructors[i].getParameterTypes();
					for (int j = 0; j < parametertypes.length; j++) {
						if (parametertypes.length == 2
							&& parametertypes[0].equals(String.class)
							&& parametertypes[1].equals(int.class)) {
							stringIntConstructorFound = true;
							String nullString = null;
							Object[] parameters =
								new Object[] { nullString, new Integer(0)};
							instance = constructors[i].newInstance(parameters);
							break;
						}
					}
					if (stringIntConstructorFound) {
						break;
					}
				}
				if (stringIntConstructorFound == false) {
					log(
						className
							+ " does not contain the expected (String, int) constructor.");
				}

			} else {
				log(
					"Expected number of constructors in '"
						+ className
						+ "':"
						+ TestReflectConstructor.CONSTRUCTOR_COUNT);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		logSuccess("Constructor test", instance != null);
	}

	private void testIMethod() {
		IMethod methodReflector =
			classReflector.getMethod(
				"foo",
				new Class[] { TestReflectClass.class });
		_assert(methodReflector != null, "getMethod");
		TestReflectClass invokeOn = new TestReflectClass();
		TestReflectClass parameter = new TestReflectClass();
		Object res =
			methodReflector.invoke(invokeOn, new Object[] { parameter });
		_assert("OK".equals(res), "IMethod.invoke() return value");
	}

	private void testIField() {
		testIField1("myString", "HiBabe", String.class);
		testIField1("myInt", new Integer(10), int.class);
		testIField1("myTyped", new TestReflectClass(), TestReflectClass.class);
		testIField1("myUntyped", "Foooo", Object.class);
		testIField1("myUntyped", new TestReflectClass(), Object.class);
		_assert(
			classReflector.getDeclaredField("myStatic").isStatic(),
			"IField.isStatic()");
		_assert(
			classReflector.getDeclaredField("myTransient").isTransient(),
			"IField.isTransient()");
	}

	private void testIField1(String fieldName, Object obj, Class clazz) {
		String fieldMessage =
			TestReflectClass.class.getName() + ":" + fieldName;
		TestReflectClass onObject = new TestReflectClass();
		IField fieldReflector = classReflector.getDeclaredField(fieldName);
		if (fieldReflector != null) {
			fieldReflector.set(onObject, obj);
			Object got = fieldReflector.get(onObject);
			_assert(got != null, fieldMessage + " IField.get returns NULL");
			_assert(
				obj.equals(got),
				fieldMessage + " IField.get returns strange Object");
			_assert(
				fieldReflector.getName().equals(fieldName),
				"IField.getName()");
			_assert(fieldReflector.isPublic(), "IField.isPublic()");
			_assert(!fieldReflector.isStatic(), "IField.isStatic()");
			_assert(!fieldReflector.isTransient(), "IField.isTransient()");
			_assert(fieldReflector.getType().equals(clazz), "IField.getType()");
		} else {
			errors = true;
			log("Field does not exist:" + fieldMessage);
		}
	}

	private void testIArray() {
		testIArray1(new Object[] {"", "hi", "Cool"});
		testIArray1(new Object[] {new Object(), new TestReflectClass(), "Woooa", new Integer(3)});
		testIArray1(new Object[] {new TestReflectClass(),new TestReflectClass()});
		testIArray2(new int[] {1,2,3});
		testIArray2(new long[] {1L,2L,3L});
	}
	
	private void testIArray1(Object[] elements){
		IArray array = reflector.array();
		Class clazz = elements[0].getClass();
		Object obj = array.newInstance(clazz,0);
		_assert(obj != null, "Creation of zero length array");
		_assert(array.getLength(obj) == 0, "Zero length array length");
		obj = array.newInstance(clazz, elements.length);
		_assert(obj != null, "Creation of variable length array");
		_assert(array.getLength(obj) == elements.length, "Variable length array length");
		for (int i = 0; i < elements.length; i++) {
			array.set(obj, i, elements[i]);
		}
		for (int i = 0; i < elements.length; i++) {
			_assert(elements[i].equals(array.get(obj, i)), "Array element comparison");
		}
	}
	
	private void testIArray2(Object arr){
		IArray array = reflector.array();
		Object element = array.get(arr, 0);
		Class clazz = element.getClass();
		Object obj = array.newInstance(clazz,0);
		_assert(obj != null, "Creation of zero length array");
		_assert(array.getLength(obj) == 0, "Zero length array length");
		int length = array.getLength(arr);
		obj = array.newInstance(clazz, length);
		_assert(obj != null, "Creation of variable length array");
		_assert(array.getLength(obj) == length, "Variable length array length");
		for (int i = 0; i < length; i++) {
			array.set(obj, i, array.get(arr,i));
		}
		for (int i = 0; i < length; i++) {
			_assert(array.get(arr,i).equals(array.get(obj, i)), "Array element comparison");
		}
	}
	

	private void logSupport(String callName, boolean flag) {
		if (!flag) {
			complete = false;
		}
		log(callName + (flag ? " are" : " are NOT") + " supported.");
	}

	private void logSuccess(String operation, boolean flag) {
		log(operation + (flag ? " was successful." : " FAILED."));
		if (!flag) {
			errors = true;
		}
	}

	private void _assert(boolean condition, String msg) {
		if (!condition) {
			log("TEST FAILED: " + msg);
			errors = true;
		}
	}

	private static void log(String msg) {
		System.out.println(msg);
	}

}
