package com.db4o.samples.internalID;

import com.db4o.*;


/**
 * Code to demonstrate the usage of internal db4o IDs.
 * <br><br>
 * Usecases like RMI, replication or interfaces to relational databases 
 * require a very performant mechanism to retrieve objects by their internal ID.
 * The following code is intended to demonstrate how this is possible by using
 * internal ID features. It also demonstrates that object lookup by internal IDs
 * is extremely fast.
 * 
 * Looking up 10,000 objects takes less than one second on normal PCs.
 * 
 * Some example results using Athlon 1400 Mhz hardware and Sun 
 * JDK 1.4 on Windows NT:<br>
 * 
 * 10,000 objects: 150 milliseconds
 * 100,000 objects: 2000 Milliseconds
 * 500,000 objects: 19 seconds (300 MB necessary. setting: Xmx300m)
 * 
 * Depending on your hardware and JVM, your results may be different.
 * <br><br>
 */
public class InternalIDSample
{
	transient long transientID;  // note the transient definition
	public String someData;
	
	public static void main(String[] args){
		Db4o.configure().messageLevel(-1);
		
		final int OBJECTCOUNT =					// the number of objects 
			(args.length == 0) ? 10000:			// configured to store 10000 objects
			new Integer(args[0]).intValue();	// may be overridden by command line
		
		final String FILENAME = "InternalDemo.yap";
		
		// delete file for new test
		new java.io.File(FILENAME).delete();
		
		ObjectContainer db = Db4o.openFile("InternalDemo.yap");
		
		System.out.println("Storing " + OBJECTCOUNT + " objects");
		for(int i = 0 ; i < OBJECTCOUNT; i ++){
			db.set(new InternalIDSample("someData" + i));
		}
		
		System.out.println("Retrieving all " + OBJECTCOUNT + " objects and assigning internal IDs.");
		ObjectSet set = db.get(new InternalIDSample());
		InternalIDSample[] retrieved = new InternalIDSample[set.size()];
		int j = 0;
		while(set.hasNext()){
			retrieved[j] = (InternalIDSample)set.next();
			
			// getting the internal ID and storing it to the transient variable
			retrieved[j].transientID = db.ext().getID(retrieved[j]);  
			
			j++;
		}
		
		
		// The ObjectContainer is closed and opened for this demo 
		// so the following retrieval code will newly instantiate
		// the stored objects
		db.close();  
		try{
			Thread.currentThread().wait(100);
		}catch(InterruptedException e){
		}
		db = Db4o.openFile("InternalDemo.yap");
		
		
		long starttime = System.currentTimeMillis();
		
		for (int i = 0; i < OBJECTCOUNT; i ++){
			
			// Extremely fast retrieval by internal ID
			InternalIDSample fastfound = 
			  (InternalIDSample)db.ext().getByID(retrieved[i].transientID);

		}
		long duration = System.currentTimeMillis() - starttime;
		System.out.println("Time spent for " + OBJECTCOUNT + " lookups:" + duration + " ms");
		
		db.close();
	}
	
	public InternalIDSample(){
		// empty constructor for db instantiation
	}
	
	public InternalIDSample(String someData){
		this.someData = someData;
	}
	
	
}
