/*
 * (c) Copyright 2002 http://www.db4o.com
 * Author: Carl Rosenberger
 * All Rights Reserved.
 */

package com.db4o.lib;

/** 
 * SubnodeCountTree
 * 
 * 
 * <br><br>Principle:<br>
 * Every node knows the number of subnodes.
 * Nodes are balanced on insert and removal, if the difference
 * between the number of subsequent and preceding nodes exceeds 2.
 * 
 * 
 * <br>Advantages against other tree algorithms:<br>
 * - The number of elements in the tree is automatically maintained.<br>
 * - Every branch also is a consistent tree that knows the number of elements.<br>
 * - The tree can be nicely constructed from the leaves. This allows reading
 * the tree from a hard drive, without having to perform compare and balance.
 */
public abstract class ScTree
{
	private ScTree i_preceding;
	private ScTree i_subsequent;
	private int i_size = 1;
	
	public final ScTree add(ScTree a_new){
		int cmp = compare(a_new);
		if(cmp < 0){
			if(i_subsequent == null){
				i_subsequent = a_new;
				i_size ++;
				return this;
			}else{
				i_subsequent = i_subsequent.add(a_new);
				if(i_preceding == null){
					return rotateLeft();
				}else{
					return balance();
				}
			}
		}else if(cmp > 0 || a_new.duplicates()){
			if(i_preceding == null){
				i_preceding = a_new;
				i_size ++;
				return this;
			}else{
				i_preceding = i_preceding.add(a_new);
				if(i_subsequent == null){
					return rotateRight();
				}else{
					return balance();
				}
			}
		}else{
			return this;
		}
	}
	
	private final ScTree balance(){
		int cmp = i_subsequent.i_size - i_preceding.i_size;
		if(cmp < -2){
			return rotateRight();
		}else if(cmp > 2){
			return rotateLeft();
		}else{
		    i_size = i_preceding.i_size + i_subsequent.i_size + 1;
		    return this;
		}
	}

	
	private final void calculateSize(){
		if(i_preceding == null){
			if (i_subsequent == null){
				i_size = 1;
			}else{
				i_size = i_subsequent.i_size + 1;
			}
		}else{
			if(i_subsequent == null){
				i_size = i_preceding.i_size + 1;
			}else{
				i_size = i_preceding.i_size + i_subsequent.i_size + 1;
			}
		}
	}
	
	abstract int compare(ScTree a_to);
	
	protected boolean duplicates() {
		return false;
	}
	
	public final ScTree find(ScTree a_tree){
		int cmp = compare(a_tree);
		if (cmp < 0){
			if(i_subsequent != null){
				return i_subsequent.find(a_tree);
			}
			return null;
		}else{
			if (cmp > 0){
				if(i_preceding != null){
					return i_preceding.find(a_tree);
				}
				return null;
			}else{
				return this;
			}
		}
	}

	public ScTree remove(){
		if(i_subsequent != null && i_preceding != null){
			i_subsequent = i_subsequent.rotateSmallestUp();
			i_subsequent.i_preceding = i_preceding;
			i_subsequent.calculateSize();
			return i_subsequent;
		}
		if(i_subsequent != null){
			return i_subsequent;
		}
		return i_preceding;
	}
	
	public final ScTree removeLike(final ScTree a_find){
		int cmp = compare(a_find);
		if(cmp == 0){
			return remove();
		}
		if (cmp > 0){
			if(i_preceding != null){
				i_preceding = i_preceding.removeLike(a_find);
			}
		}else{
			if(i_subsequent != null){
				i_subsequent = i_subsequent.removeLike(a_find);
			}
		}
		calculateSize();
		return this;
	}
	
	public final ScTree removeNode(final ScTree a_tree){
		if (this == a_tree){
			return remove();
		}
		int cmp = compare(a_tree);
		if (cmp >= 0){
			if(i_preceding != null){
				i_preceding = i_preceding.removeNode(a_tree);
			}
		}
		if(cmp <= 0){
			if(i_subsequent != null){
				i_subsequent = i_subsequent.removeNode(a_tree);	
			}
		}
		calculateSize();
		return this;
	}
	
	private final ScTree rotateLeft(){
		ScTree tree = i_subsequent;
		i_subsequent = tree.i_preceding;
		calculateSize();
		tree.i_preceding = this;
		tree.calculateSize();
		return tree;
	}

	private final ScTree rotateRight(){
		ScTree tree = i_preceding;
		i_preceding = tree.i_subsequent;
		calculateSize();
		tree.i_subsequent = this;
		tree.calculateSize();
		return tree;
	}
	
	private final ScTree rotateSmallestUp(){
		if(i_preceding != null){
			i_preceding = i_preceding.rotateSmallestUp();
			return rotateRight();
		}
		return this;
	}
	
	public int size(){
		return i_size;
	}
	
	public final void traverse(Visitor4 a_visitor){
		if(i_preceding != null){
			i_preceding.traverse(a_visitor);
		}
		a_visitor.visit(this);
		if(i_subsequent != null){
			i_subsequent.traverse(a_visitor);
		}
	}
	
	
}
