/*
 * DESKeySpec.java
 *
 * This class is part of our JCE 1.2 cleanroom implementation
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package javax.crypto.spec;

import java.security.InvalidKeyException;
import java.security.spec.KeySpec;
import java.util.Arrays;

public class DESKeySpec implements KeySpec
{
	public static final int
		DES_KEY_LEN = 8;

	/*
	 * For determining odd parity for each word, we could either:
	 * 1) count all the one bits, and see if we have an odd number
	 * 2) use a 256 bit boolean lookup table
	 * 
	 * 1 is too slow, and 2 takes too much space; so we use a tradeoff:
	 *
	 * 3) use a 16 entry table; split a byte into a low part and high part
	 * for each byte return parityEven[b & 0xf] == parityEven[b >> 4];
	 */
	protected static final boolean[] parityEven = {
		true, false, false, true, false, true, true, false,
		false, true, true, false, true, false, false, true };

	/*
	 * array with all known weak keys; all keys have their parity stripped
	 */
	private static final byte[][] weak = {
		/* weak keys; 2 pairs */
		{ 0, 0, 0, 0, 0, 0, 0, 0 },
		{ -2, -2, -2, -2, -2, -2, -2, -2 },
		{ 30, 30, 30, 30, 14, 14, 14, 14 },
		{ -32, -32, -32, -32, -16, -16, -16, -16 },

		/* semi-weak keys; 6 pairs */
		{ 0, -2, 0, -2, 0, -2, 0, -2 },
		{ -2, 0, -2, 0, -2, 0, -2, 0 },
		{ 0, 30, 0, 30, 0, 14, 0, 14 },
		{ -2, -32, -2, -32, -2, -16, -2, -16 },
		{ 0, -32, 0, -32, 0, -16, 0, -16 },
		{ -2, 30, -2, 30, -2, 14, -2, 14 },
		{ 30, -32, 30, -32, 14, -16, 14, -16 },
		{ -32, 30, -32, 30, -16, 14, -16, 14 },
		{ 30, -2, 30, -2, 14, -2, 14, -2 },
		{ -32, 0, -32, 0, -16, 0, -16, 0 },
		{ -32, -2, -32, -2, -16, -2, -16, -2 },
		{ 30, 0, 30, 0, 14, 0, 14, 0 },

		/* possibly weak keys; 48 pairs */
		{ 0, 0, -2, -2, 0, 0, -2, -2 },
		{ -2, -2, 0, 0, -2, -2, 0, 0 },
		{ 0, -2, -2, 0, 0, -2, -2, 0 },
		{ -2, 0, 0, -2, -2, 0, 0, -2 },
		{ 30, 30, -32, -32, 14, 14, -16, -16 },
		{ -32, -32, 30, 30, -16, -16, 14, 14 },
		{ 30, -32, -32, 30, 14, -16, -16, 14 },
		{ -32, 30, 30, -32, -16, 14, 14, -16 },
		{ 0, 0, 30, 30, 0, 0, 14, 14 },
		{ -2, -2, -32, -32, -2, -2, -16, -16 },
		{ 0, 30, 30, 0, 0, 14, 14, 0 },
		{ -2, -32, -32, -2, -2, -16, -16, -2 },
		{ 30, 0, 0, 30, 14, 0, 0, 14 },
		{ -32, -2, -2, -32, -16, -2, -2, -16 },
		{ 30, 30, 0, 0, 14, 14, 0, 0 },
		{ -32, -32, -2, -2, -16, -15, -2, -2 },
		{ 0, 0, -32, -32, 0, 0, -16, -16 },
		{ -2, -2, 30, 30, -2, -2, 14, 14 },
		{ 0, -32, -32, 0, 0, -16, -16, 0 },
		{ -2, 30, 30, -2, -2, 14, 14, -2 },
		{ -32, 0, 0, -32, -16, 0, 0, -16 },
		{ 30, -2, -2, 30, 14, -2, -2, 14 },
		{ -32, -32, 0, 0, -16, -16, 0, 0 },
		{ 30, 30, -2, -2, 14, 14, -2, -2 },
		{ 0, 30, -32, -2, 0, 14, -16, -2 },
		{ -2, -32, 32, 0, -2, -16, 14, 0 },
		{ 0, 30, -2, -32, 0, 14, -2, -16 },
		{ -2, -32, 0, 32, -2, -16, 0, 14 },
		{ 0, -32, 30, -2, 0, -16, 14, -2 },
		{ -2, 30, -32, 0, -2, 14, -16, 0 },
		{ 0, -32, -2, 30, 0, -16, -2, 14 },
		{ -2, 30, 0, -32, -2, 14, 0, -16 },
		{ 0, -2, 30, -32, 0, -2, 14, -16 },
		{ -2, 0, -32, 30, -2, 0, -16, 14 },
		{ 0, -2, -32, 30, 0, -2, -16, 14 },
		{ -2, 0, 30, -32, -2, 0, 14, -16 },
		{ 30, 0, -32, -2, 14, 0, -16, -2 },
		{ -32, -2, 30, 0, -16, -2, 14, 0 },
		{ 30, 0, -2, -32, 14, 0, -2, -16 },
		{ -32, -2, 0, 30, -16, -2, 0, 14 },
		{ 30, -32, 0, -2, 14, -16, 0, -2 },
		{ -32, 30, -2, 0, -16, 14, -2, 0 },
		{ 30, -32, -2, 0, 14, -16, -2, 0 },
		{ -32, 30, 0, -2, -16, 14, 0, -2 },
		{ 30, -2, 0, -32, 14, -2, 0, -16 },
		{ -32, 0, -2, 30, -16, 0, -2, 14 },
		{ 30, -2, -32, 0, 14, -2, -16, 0 },
		{ -32, 0, 30, -2, -16, 0, 14, -2 }
	};

	private byte[] key;

	public DESKeySpec(byte[] key) throws InvalidKeyException
	{
		this(key, 0);
	}

	public DESKeySpec(byte[] key, int offset) throws InvalidKeyException
	{
		if (key == null)
			throw new InvalidKeyException("key is null");

		if (key.length - offset < DES_KEY_LEN)
			throw new InvalidKeyException("key too short");

		this.key = new byte[DES_KEY_LEN];
		System.arraycopy(key, offset, this.key, 0, DES_KEY_LEN);
	}

	public byte[] getKey()
	{
		return (byte[]) key.clone();
	}

	public static boolean isParityAdjusted(byte[] key, int offset) throws InvalidKeyException
	{
		if (key == null)
			throw new InvalidKeyException("key is null");

		if (key.length - offset < DES_KEY_LEN)
			throw new InvalidKeyException("key too short");

		for (int i = 0; i < DES_KEY_LEN; i++)
		{
			byte b = key[i+offset];
			if (parityEven[b & 0xf] == parityEven[b >>> 4])
				return false;
		}
		return true;
	}

	public static boolean isWeak(byte[] key, int offset) throws InvalidKeyException
	{
		if (key == null)
			throw new InvalidKeyException("key is null");

		if (key.length - offset < DES_KEY_LEN)
			throw new InvalidKeyException("key too short");

		byte[] tmp = new byte[DES_KEY_LEN];

		for (int i = 0; i < DES_KEY_LEN; i++)
			tmp[i] = (byte) (key[offset+i] & 0xFE);

		for (int i = 0; i < weak.length; i++)
		{
			if (Arrays.equals(tmp, weak[i]))
				return true;
		}

		return false;
	}
}
