/*
 * SealedObject.java
 *
 * This class is part of our JCE 1.2 cleanroom implementation
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package javax.crypto;

import java.io.*;
import java.security.*;

public class SealedObject implements Serializable
{
	private byte[] encryptedObject = null;
	private byte[] encodedParams = null;

	private String cipherAlgorithm = null;
	private String paramsAlgorithm = null;

	public SealedObject(Serializable object, Cipher c) throws IOException, IllegalBlockSizeException
	{
		if (object == null)
			throw new InvalidParameterException("object is null");

		ByteArrayOutputStream bo = new ByteArrayOutputStream();
		ObjectOutputStream oo = new ObjectOutputStream(bo);

		oo.writeObject(object);
		oo.flush();
		oo.close();

		try
		{
			encryptedObject = c.doFinal(bo.toByteArray());
			bo.close();
		}
		catch (BadPaddingException bpe)
		{
			/* for lack of better alternatives: */
			throw new IOException(bpe.getMessage());
		}

		cipherAlgorithm = c.getAlgorithm();

		AlgorithmParameters params = c.getParameters();
		
		if (params != null)
		{
			paramsAlgorithm = params.getAlgorithm();
			encodedParams = params.getEncoded();
		}
	}

	public final String getAlgorithm()
	{
		return cipherAlgorithm;
	}

	public final Object getObject(Key key) throws IOException, ClassNotFoundException, NoSuchAlgorithmException, InvalidKeyException
	{
		Cipher c;
		AlgorithmParameters params = null;

		if (encodedParams != null)
		{
			params = AlgorithmParameters.getInstance(paramsAlgorithm);
			params.init(encodedParams);
		}

		try
		{
			c = Cipher.getInstance(cipherAlgorithm);

			if (params != null)
				c.init(Cipher.DECRYPT_MODE, key, params);
			else
				c.init(Cipher.DECRYPT_MODE, key);

			return getObject(c);
		}
		catch (InvalidAlgorithmParameterException iape)
		{
			/* for lack of better alternatives: */
			throw new IOException(iape.getMessage());
		}
		catch (NoSuchPaddingException nspe)
		{
			/* for lack of better alternatives: */
			throw new IOException(nspe.getMessage());
		}
		catch (IllegalBlockSizeException ibse)
		{
			/* for lack of better alternatives: */
			throw new IOException(ibse.getMessage());
		}
		catch (BadPaddingException bpe)
		{
			/* for lack of better alternatives: */
			throw new IOException(bpe.getMessage());
		}
	}

	public final Object getObject(Cipher c) throws IOException, ClassNotFoundException, IllegalBlockSizeException, BadPaddingException
	{
		Object obj = null;
		ByteArrayInputStream bis = new ByteArrayInputStream(c.doFinal(encryptedObject));
		ObjectInputStream ois = new ObjectInputStream(bis);
		
		try
		{
			obj = ois.readObject();
		}
		catch (StreamCorruptedException sce)
		{
			/* for lack of better alternatives: */
			throw new IOException(sce.getMessage());
		}
		catch (OptionalDataException ode)
		{
			/* for lack of better alternatives: */
			throw new IOException(ode.getMessage());
		}
		catch (InvalidClassException ice)
		{
			/* for lack of better alternatives: */
			throw new ClassNotFoundException(ice.getMessage());
		}
		
		ois.close();
		bis.close();
		
		return obj;
	}

	public final Object getObject(Key key, String provider) throws IOException, ClassNotFoundException, NoSuchAlgorithmException, NoSuchProviderException, InvalidKeyException
	{
		if (provider == null)
			throw new NoSuchProviderException("provider is null");

		Cipher c;
		AlgorithmParameters params = null;

		if (encodedParams != null)
		{
			params = AlgorithmParameters.getInstance(paramsAlgorithm, provider);
			params.init(encodedParams);
		}

		try
		{
			c = Cipher.getInstance(cipherAlgorithm, provider);

			if (params != null)
				c.init(Cipher.DECRYPT_MODE, key, params);
			else
				c.init(Cipher.DECRYPT_MODE, key);

			return getObject(c);
		}
		catch (InvalidAlgorithmParameterException iape)
		{
			/* for lack of better alternatives: */
			throw new IOException(iape.getMessage());
		}
		catch (NoSuchPaddingException nspe)
		{
			/* for lack of better alternatives: */
			throw new IOException(nspe.getMessage());
		}
		catch (IllegalBlockSizeException ibse)
		{
			/* for lack of better alternatives: */
			throw new IOException(ibse.getMessage());
		}
		catch (BadPaddingException bpe)
		{
			/* for lack of better alternatives: */
			throw new IOException(bpe.getMessage());
		}
	}
}
