/*
 * Mac.java
 *
 * This class is part of our JCE 1.2 cleanroom implementation
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package javax.crypto;

import java.security.*;
import java.security.spec.*;

public class Mac implements Cloneable
{
	private MacSpi		spi;
	private Provider	provider;
	private String		algorithm;
	private boolean		initialized;

	protected Mac(MacSpi spi, Provider provider, String algorithm)
	{
		this.spi = spi;
		this.provider = provider;
		this.algorithm = algorithm;
		this.initialized = false;
	}

	public final Object clone() throws CloneNotSupportedException
	{
		Mac tmp = (Mac) super.clone();
		tmp.spi = (MacSpi) this.spi.clone();
		return tmp;
	}

	public final byte[] doFinal() throws IllegalStateException
	{
		if (initialized)
		{
			byte[] tmp = spi.engineDoFinal();
			spi.engineReset();
			return tmp;
		}
		else
			throw new IllegalStateException("Mac not initialized");
	}

	public final byte[] doFinal(byte[] input) throws IllegalStateException
	{
		if (initialized)
		{
			if (input != null)
				spi.engineUpdate(input, 0, input.length);

			return doFinal();
		}
		else
			throw new IllegalStateException("Mac not initialized");
	}

	public final void doFinal(byte[] output, int outOffset) throws ShortBufferException, IllegalStateException
	{
		if (initialized)
		{
			int macLength = spi.engineGetMacLength();

			if (output == null)
				throw new ShortBufferException("output is null");

			if (output.length - outOffset < macLength)
				throw new ShortBufferException("output buffer is too small");

			System.arraycopy(spi.engineDoFinal(), 0, output, outOffset, macLength);
		}
		else
			throw new IllegalStateException("Mac not initialized");
	}

	public final String getAlgorithm()
	{
		return algorithm;
	}

	public static final Mac getInstance(String algorithm) throws NoSuchAlgorithmException
	{
		Object[] obj = share.getImplementation("Mac", algorithm, null);

		return new Mac((MacSpi) obj[0], (Provider) obj[1], algorithm);
	}

	public static final Mac getInstance(String algorithm, String provider) throws NoSuchAlgorithmException, NoSuchProviderException
	{
		Provider p = Security.getProvider(provider);

		if (p == null)
			throw new NoSuchProviderException("Provider " + provider + " not installed");

		Object[] obj = share.getImplementation("Mac", algorithm, p);

		return new Mac((MacSpi) obj[0], p, algorithm);
	}

	public final int getMacLength()
	{
		return spi.engineGetMacLength();
	}

	public final Provider getProvider()
	{
		return provider;
	}

	public final void init(Key key) throws InvalidKeyException
	{
		try
		{
			spi.engineInit(key, null);
			initialized = true;
		}
		catch (InvalidAlgorithmParameterException iape)
		{
			/* in this case we cannot be sure of the state anymore */
			initialized = false;
		}
	}

	public final void init(Key key, AlgorithmParameterSpec params) throws InvalidKeyException, InvalidAlgorithmParameterException
	{
		spi.engineInit(key, params);
		initialized = true;
	}

	public final void reset()
	{
		spi.engineReset();
	}

	public final void update(byte input) throws IllegalStateException
	{
		if (initialized)
			spi.engineUpdate(input);
		else
			throw new IllegalStateException("MAC not initialized");
	}

	public final void update(byte[] input) throws IllegalStateException
	{
		if (initialized)
		{
			if (input != null)
				spi.engineUpdate(input, 0, input.length);
		}
		else
			throw new IllegalStateException("MAC not initialized");
	}

	public final void update(byte[] input, int offset, int len) throws IllegalStateException
	{
		if (initialized)
		{
			if (input != null)
			{
				if (offset < 0 || len < 0 || input.length - len < offset)
					throw new IllegalArgumentException("Invalid arguments");

				spi.engineUpdate(input, offset, len);
			}
		}
		else
			throw new IllegalStateException("MAC not initialized");
	}
}
