/*
 * KeyAgreement.java
 *
 * This class is part of our JCE 1.2 cleanroom implementation
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package javax.crypto;

import java.security.*;
import java.security.spec.*;

public class KeyAgreement
{
	private KeyAgreementSpi	spi;
	private Provider		provider;
	private String			algorithm;

	protected KeyAgreement(KeyAgreementSpi keyAgreeSpi, Provider provider, String algorithm)
	{
		this.spi = keyAgreeSpi;
		this.provider = provider;
		this.algorithm = algorithm;
	}

	public final Key doPhase(Key key, boolean lastPhase) throws InvalidKeyException, IllegalStateException
	{
		return spi.engineDoPhase(key, lastPhase);
	}

	public final byte[] generateSecret() throws IllegalStateException
	{
		return spi.engineGenerateSecret();
	}

	public final int generateSecret(byte[] sharedSecret, int offset) throws IllegalStateException, ShortBufferException
	{
		return spi.engineGenerateSecret(sharedSecret, offset);
	}

	public final SecretKey generateSecret(String algorithm) throws IllegalStateException, NoSuchAlgorithmException, InvalidKeyException
	{
		return spi.engineGenerateSecret(algorithm);
	}

	public final String getAlgorithm()
	{
		return algorithm;
	}

	public static final KeyAgreement getInstance(String algorithm) throws NoSuchAlgorithmException
	{
		Object[] obj = share.getImplementation("KeyAgreement", algorithm, null);

		return new KeyAgreement((KeyAgreementSpi) obj[0], (Provider) obj[1], algorithm);
	}

	public static final KeyAgreement getInstance(String algorithm, String provider) throws NoSuchAlgorithmException, NoSuchProviderException
	{
		Provider p = Security.getProvider(provider);

		if (p == null)
			throw new NoSuchProviderException("Provider " + provider + " not installed");

		Object[] obj = share.getImplementation("KeyAgreement", algorithm, p);

		return new KeyAgreement((KeyAgreementSpi) obj[0], p, algorithm);
	}

	public final Provider getProvider()
	{
		return provider;
	}

	public final void init(Key key) throws InvalidKeyException
	{
		spi.engineInit(key, new SecureRandom());
	}

	public final void init(Key key, AlgorithmParameterSpec params) throws InvalidKeyException, InvalidAlgorithmParameterException
	{
		spi.engineInit(key, params, new SecureRandom());
	}

	public final void init(Key key, AlgorithmParameterSpec params, SecureRandom random) throws InvalidKeyException, InvalidAlgorithmParameterException
	{
		spi.engineInit(key, params, random);
	}

	public final void init(Key key, SecureRandom random) throws InvalidKeyException
	{
		spi.engineInit(key, random);
	}
}
