/*
 * CipherOutputStream.java
 *
 * This class is part of our JCE 1.2 cleanroom implementation
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package javax.crypto;

import java.io.*;

public class CipherOutputStream extends FilterOutputStream
{
	private Cipher cipher;
	private byte[] decryptedBuffer;
	private byte[] encryptedBuffer;
	
	protected CipherOutputStream(OutputStream os)
	{
		this(os, new NullCipher());
	}

	public CipherOutputStream(OutputStream os, Cipher c)
	{
		super(os);
		cipher = c;
		decryptedBuffer = new byte[1];
	}

	public void close() throws IOException
	{
		try
		{
			encryptedBuffer = cipher.doFinal();
		}
		catch (IllegalBlockSizeException ibse)
		{
			encryptedBuffer = null;
		}
		catch (BadPaddingException bpe)
		{
			encryptedBuffer = null;
		}
		flush();
		super.close();
	}

	public void flush() throws IOException
	{
		if (encryptedBuffer != null)
		{
			super.write(encryptedBuffer);
			encryptedBuffer = null;
		}
		super.flush();
	}

	public void write(int b) throws IOException
	{
		decryptedBuffer[0] = (byte) b;
		encryptedBuffer = cipher.update(decryptedBuffer, 0, 1);
		if (encryptedBuffer != null)
		{
			super.write(encryptedBuffer);
			encryptedBuffer = null;
		}
	}

	public void write(byte[] b) throws IOException
	{
		write(b, 0, b.length);
	}

	public void write(byte[] b, int off, int len) throws IOException
	{
		encryptedBuffer = cipher.update(b, off, len);
		if (encryptedBuffer != null)
		{
			super.write(encryptedBuffer);
			encryptedBuffer = null;
		}
	}
}