/*
 * CipherInputStream.java
 *
 * This class is part of our JCE 1.2 cleanroom implementation
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package javax.crypto;

import java.io.*;

public class CipherInputStream extends FilterInputStream
{
	private Cipher cipher;
	private byte[] encryptedBuffer;
	private byte[] decryptedBuffer;
	private int decryptedOffset;
	private int decryptedLength;
	private boolean eos;

	protected CipherInputStream(InputStream is)
	{
		this(is, new NullCipher());
	}

	public CipherInputStream(InputStream is, Cipher c)
	{
		super(is);
		cipher = c;
		encryptedBuffer = new byte[512];
		decryptedOffset =
		decryptedLength = 0;
		eos = false;
	}

	public int available()
	{
		return decryptedLength - decryptedOffset;
	}

	public void close() throws IOException
	{
		try
		{
			cipher.doFinal();
		}
		catch (IllegalBlockSizeException ibse)
		{
		}
		catch (BadPaddingException bpe)
		{
		}
		super.close();
	}

	public boolean markSupported()
	{
		return false;
	}

	private int bufferedRead() throws IOException
	{
		if (eos)
			return -1;

		int result = super.read(encryptedBuffer);

		if (result == -1)
		{
			eos = true;
			try
			{
				cipher.doFinal();
			}
			catch (IllegalBlockSizeException ibse)
			{
				decryptedBuffer = null;
			}
			catch (BadPaddingException bpe)
			{
				decryptedBuffer = null;
			}
			if (decryptedBuffer == null)
				return -1;
			else
			{
				decryptedOffset = 0;
				decryptedLength = decryptedBuffer.length;
			}
		}
		try
		{
			decryptedBuffer = cipher.update(encryptedBuffer, 0, result);
		}
		catch (IllegalStateException ise)
		{
			decryptedBuffer = null;
		}
		decryptedOffset = 0;
		decryptedLength = (decryptedBuffer == null) ? 0 : decryptedBuffer.length;
		return decryptedLength;
	}

	public int read() throws IOException
	{
		if (decryptedOffset >= decryptedLength)
		{
			int result = 0;
			while (result == 0)
				result = bufferedRead();
			if (result == -1)
				return -1;
		}
		return decryptedBuffer[decryptedOffset++];
	}

	public int read(byte[] b) throws IOException
	{
		return read(b, 0, b.length);
	}
	
	public int read(byte[] b, int off, int len) throws IOException
	{
		if (b == null)
			throw new IllegalArgumentException("b is null");
			
		if (len < 0)
			throw new IllegalArgumentException("len < 0");
			
		if (off < 0)
			throw new IllegalArgumentException("off < 0");

		if (decryptedOffset >= decryptedLength)
		{
			int result = 0;
			while (result == 0)
				result = bufferedRead();
			if (result == -1)
				return -1;
		}

		if (len == 0)
			return 0;

		int avail = decryptedLength - decryptedOffset;
		if (len > avail)
			len = avail;

		System.arraycopy(decryptedBuffer, decryptedOffset, b, off, len);
		decryptedOffset += len;
		return len;
	}

	public long skip(long n) throws IOException
	{
		if (n < 0L)
			return 0L; 

		int avail = decryptedLength - decryptedOffset;

		if (n > (long) avail)
			n = (long) avail;
			
		decryptedOffset += (int) n;
		return n;
	}
}