/*
 * Cipher.java
 *
 * This class is part of our JCE 1.2 cleanroom implementation
 *
 * Copyright (c) 2000 Virtual Unlimited B.V.
 *
 * Author: Bob Deblier <bob@virtualunlimited.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package javax.crypto;

import java.util.*;
import java.security.*;
import java.security.spec.*;

public class Cipher
{
	public static final int
		ENCRYPT_MODE = 1,
		DECRYPT_MODE = 2;

	private CipherSpi	spi;
	private Provider	provider;
	private String		transformation;

	private boolean initialized;

	protected Cipher(CipherSpi cipherSpi, Provider provider, String transformation)
	{
		this.spi = cipherSpi;
		this.provider = provider;
		this.transformation = transformation;
		this.initialized = false;
	}

	public final byte[] doFinal() throws IllegalStateException, IllegalBlockSizeException, BadPaddingException
	{
		return spi.engineDoFinal(null, 0, 0);
	}

	public final byte[] doFinal(byte[] input) throws IllegalStateException, IllegalBlockSizeException, BadPaddingException
	{
		if (input == null)
			throw new IllegalArgumentException("input is null");

		return spi.engineDoFinal(input, 0, input.length);
	}

	public final int doFinal(byte[] output, int outputOffset) throws IllegalStateException, IllegalBlockSizeException, ShortBufferException, BadPaddingException
	{
		if (output == null)
			throw new IllegalArgumentException("output is null");

		if (outputOffset < 0)
			throw new IllegalArgumentException("output bad offset");

		return spi.engineDoFinal(null, 0, 0, output, outputOffset);
	}

	public final byte[] doFinal(byte[] input, int inputOffset, int inputLen) throws IllegalStateException, IllegalBlockSizeException, BadPaddingException
	{
		if (input == null)
			throw new IllegalArgumentException("input is null");

		if (inputOffset < 0 || inputLen < 0 || input.length - inputOffset < inputLen)
			throw new IllegalArgumentException("input bad offset/length");

		return spi.engineDoFinal(input, inputOffset, inputLen);
	}

	public final int doFinal(byte[] input, int inputOffset, int inputLen, byte[] output) throws ShortBufferException, IllegalBlockSizeException, BadPaddingException
	{
		if (input == null)
			throw new IllegalArgumentException("input is null");

		if (inputOffset < 0 || inputLen < 0 || input.length - inputOffset < inputLen)
			throw new IllegalArgumentException("input bad offset/length");

		if (output == null)
			throw new IllegalArgumentException("output is null");

		return spi.engineDoFinal(input, inputOffset, inputLen, output, 0);
	}

	public final int doFinal(byte[] input, int inputOffset, int inputLen, byte[] output, int outputOffset) throws ShortBufferException, IllegalBlockSizeException, BadPaddingException
	{
		if (input == null)
			throw new IllegalArgumentException("input is null");

		if (inputOffset < 0 || inputLen < 0 || input.length - inputOffset < inputLen)
			throw new IllegalArgumentException("input bad offset/length");

		if (output == null)
			throw new IllegalArgumentException("output is null");

		if (outputOffset < 0)
			throw new IllegalArgumentException("output bad offset");

		return spi.engineDoFinal(input, inputOffset, inputLen, output, outputOffset);
	}

	public final String getAlgorithm()
	{
		return transformation;
	}

	public final int getBlockSize()
	{
		return spi.engineGetBlockSize();
	}

	public static final Cipher getInstance(String transformation) throws NoSuchAlgorithmException, NoSuchPaddingException
	{
		Cipher c = null;

		StringTokenizer st = new StringTokenizer(transformation, "/");

		try
		{
			int tokens = st.countTokens();

			if (tokens != 1 && tokens != 3)
				throw new NoSuchAlgorithmException("illegal transformation");

			Object[] obj = share.getImplementation("Cipher", st.nextToken(), null);

			c = new Cipher((CipherSpi) obj[0], (Provider) obj[1], transformation);

			if (tokens > 1)
			{
				c.spi.engineSetMode(st.nextToken());
				c.spi.engineSetPadding(st.nextToken());
			}
		}
		catch (NoSuchElementException e)
		{
			/* will not occur */
		}

		return c;
	}

	public static final Cipher getInstance(String transformation, String provider) throws NoSuchAlgorithmException, NoSuchProviderException, NoSuchPaddingException
	{
		Provider p = Security.getProvider(provider);

		if (p == null)
			throw new NoSuchProviderException("Provider " + provider + " not installed");

		Cipher c = null;
		StringTokenizer st = new StringTokenizer(transformation, "/");

		try
		{
			int tokens = st.countTokens();

			if (tokens != 1 && tokens != 3)
				throw new NoSuchAlgorithmException("illegal transformation");

			Object[] obj = share.getImplementation("Cipher", st.nextToken(), p);

			c = new Cipher((CipherSpi) obj[0], p, transformation);

			if (tokens > 1)
			{
				c.spi.engineSetMode(st.nextToken());
				c.spi.engineSetPadding(st.nextToken());
			}
		}
		catch (NoSuchElementException e)
		{
			/* will not occur */
		}
		return c;
	}

	public final byte[] getIV()
	{
		return spi.engineGetIV();
	}

	public final int getOutputSize(int inputLen)
	{
		return spi.engineGetOutputSize(inputLen);
	}

	public final AlgorithmParameters getParameters()
	{
		return spi.engineGetParameters();
	}

	public final Provider getProvider()
	{
		return provider;
	}

	public final void init(int opmode, Key key) throws InvalidKeyException
	{
		if (opmode != ENCRYPT_MODE && opmode != DECRYPT_MODE)
			throw new InvalidParameterException("invalid opmode");

		spi.engineInit(opmode, key, new SecureRandom());
		initialized = true;
	}

	public final void init(int opmode, Key key, AlgorithmParameters params) throws InvalidKeyException, InvalidAlgorithmParameterException
	{
		if (opmode != ENCRYPT_MODE && opmode != DECRYPT_MODE)
			throw new InvalidParameterException("invalid opmode");

		spi.engineInit(opmode, key, params, new SecureRandom());
		initialized = true;
	}

	public final void init(int opmode, Key key, AlgorithmParameterSpec params) throws InvalidKeyException, InvalidAlgorithmParameterException
	{
		if (opmode != ENCRYPT_MODE && opmode != DECRYPT_MODE)
			throw new InvalidParameterException("invalid opmode");

		spi.engineInit(opmode, key, params, new SecureRandom());
		initialized = true;
	}

	public final void init(int opmode, Key key, AlgorithmParameters params, SecureRandom random) throws InvalidKeyException, InvalidAlgorithmParameterException
	{
		if (opmode != ENCRYPT_MODE && opmode != DECRYPT_MODE)
			throw new InvalidParameterException("invalid opmode");

		spi.engineInit(opmode, key, params, random);
		initialized = true;
	}

	public final void init(int opmode, Key key, AlgorithmParameterSpec params, SecureRandom random) throws InvalidKeyException, InvalidAlgorithmParameterException
	{
		if (opmode != ENCRYPT_MODE && opmode != DECRYPT_MODE)
			throw new InvalidParameterException("invalid opmode");

		spi.engineInit(opmode, key, params, random);
		initialized = true;
	}

	public final void init(int opmode, Key key, SecureRandom random) throws InvalidKeyException
	{
		if (opmode != ENCRYPT_MODE && opmode != DECRYPT_MODE)
			throw new InvalidParameterException("invalid opmode");

		spi.engineInit(opmode, key, random);
		initialized = true;
	}

	public final byte[] update(byte[] input) throws IllegalStateException
	{
		if (!initialized && !(this instanceof NullCipher))
			throw new IllegalStateException("Cipher not initialized");

		if (input == null)
			throw new IllegalArgumentException("input is null");

		return spi.engineUpdate(input, 0, input.length);
	}

	public final byte[] update(byte[] input, int inputOffset, int inputLen) throws IllegalStateException
	{
		if (!initialized && !(this instanceof NullCipher))
			throw new IllegalStateException("Cipher not initialized");

		if (input == null)
			throw new IllegalArgumentException("input is null");

		if (inputOffset < 0 || inputLen < 0 || input.length - inputOffset < inputLen)
			throw new IllegalArgumentException("input bad offset/length");

		return spi.engineUpdate(input, inputOffset, inputLen);
	}

	public final int update(byte[] input, int inputOffset, int inputLen, byte[] output) throws IllegalStateException, ShortBufferException
	{
		if (!initialized && !(this instanceof NullCipher))
			throw new IllegalStateException("Cipher not initialized");

		if (input == null)
			throw new IllegalArgumentException("input is null");

		if (inputOffset < 0 || inputLen < 0 || input.length - inputOffset < inputLen)
			throw new IllegalArgumentException("input bad offset/length");

		if (output == null)
			throw new IllegalArgumentException("output is null");

		return spi.engineUpdate(input, inputOffset, inputLen, output, 0);
	}

	public final int update(byte[] input, int inputOffset, int inputLen, byte[] output, int outputOffset) throws IllegalStateException, ShortBufferException
	{
		if (!initialized && !(this instanceof NullCipher))
			throw new IllegalStateException("Cipher not initialized");

		if (input == null)
			throw new IllegalArgumentException("input is null");

		if (inputOffset < 0 || inputLen < 0 || input.length - inputOffset < inputLen)
			throw new IllegalArgumentException("input bad offset/length");

		if (output == null)
			throw new IllegalArgumentException("output is null");

		return spi.engineUpdate(input, inputOffset, inputLen, output, outputOffset);
	}
}
