{**************************************************}
{                                                  }
{                   llTaskBar                      }
{           Version  1.51,  26.04.2002             }
{          Copyright (c) 2002 llionsoft            }
{             All rights reserved                  }
{             mailto:team@llion.net                }
{                                                  }
{**************************************************}
unit llTaskBar;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Buttons, StdCtrls, ExtCtrls{, tlhelp32};

const
  WM_TBMESSAGE = WM_USER + $333;

type
  TllTaskButton = class(TSpeedButton)
  private
    FForm: TForm;
    OldWndProc: Pointer;
    WndProcPtr: Pointer;
    procedure SetForm(Value: TForm);
    procedure WndMethod(var Msg: TMessage);
    function GetDown: Boolean;
    procedure SetDown(Value: Boolean);
    procedure CMButtonPressed(var Message: TMessage); message CM_BUTTONPRESSED;
  protected
    procedure CalculateCaption; dynamic;
    procedure Resize; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Down: Boolean read GetDown Write SetDown;
    property Form: TForm read FForm Write SetForm default nil;
  end;

  TPictureAlign = (paLeftTop, paRightBottom);
  TllTaskBarKind = (lltbHorizontal, lltbVertical);
  TResizing = set of (rXt, rYt, rXb, rYb);

  TllTaskBar = class(TCustomControl)
  private
    { Private declarations }
    FCursorChanged: Boolean;
    FOldCursor: TCursor;
    FMinButtonWidth: Integer;
    FMaxButtonWidth: Integer;
    FButtonWidth: Integer;
    FButtonHeight: Integer;
    FCountButton: Integer;
    FButtons: TList;
    FFlat: Boolean;
    FForm: TForm;
    FButtonTransparent: Boolean;
    FActiveButton: Integer;
    FFirstButton: Integer;
    FBitmap: TBitmap;
    FPicture: TBitmap;
    FPictureWidth: Integer;
    FPictureAlign: TPictureAlign;
    FPictureBevel: Boolean;
    FPictureRatio: Boolean;
    FPictureMaximize: Boolean;
    FPictureCentered: Boolean;
    FPictureRect: TRect;
    FPictureTransparent: Boolean;
    FCountVisibleLines: Integer;
    FCountLines: Integer;
    FCountButtonInLine: Integer;
    FLineWidth: Integer;
    FLineHeight: Integer;
    FScrollBarWidth: Integer;
    FScrollBar: TScrollBar;
    FHideMinimizedMDI: Boolean;
    FKind: TllTaskBarKind;
    FMouseY: Integer;
    FMouseX: Integer;
    FOldWidth: Integer;
    FOldHeight: Integer;
    FResizing: TResizing;
    FNotRePaint: Boolean;
    FInternalCorrectSize: Boolean;
    FSynchronizeIcon: Boolean;
    FSynchronizeCaption: Boolean;
    procedure SetMinButtonWidth(Value: Integer);
    procedure SetMaxButtonWidth(Value: Integer);
    procedure SetFlat(Value: Boolean);
    procedure SetButtonTransparent(Value: Boolean);
    procedure SetPicture(Value: TBitmap);
    procedure SetPictureWidth(Value: Integer);
    procedure SetPictureAlign(Value: TPictureAlign);
    procedure SetHideMinimizedMDI(Value: Boolean);
    procedure SetKind(Value: TllTaskBarKind);
    procedure SetPictureRatio(Value: Boolean);
    procedure SetPictureCentered(Value: Boolean);
    procedure SetPictureMaximize(Value: Boolean);
    procedure SetPictureBevel(Value: Boolean);
    procedure SetPictureTransparent(Value: Boolean);
    procedure SetBitmap(Value: TBitmap);
    procedure SetSynchronizeCaption(Value: Boolean);
    procedure SetSynchronizeIcon(Value: Boolean);
    function GetButton(Index: Integer): TllTaskButton;
    function GetButtonCount: Integer;
    function GetMainForm(AOwner: TComponent): TForm;
  protected
    { Protected declarations }
    function FindForm(Form: TForm): Boolean;
    function CanResize(var NewWidth, NewHeight: Integer): Boolean; override;
    procedure Resize; override;
    function FindByForm(_Form: TForm): Integer;
    function AddButton(_Form: TForm): Integer;
    procedure DeleteButton(Index: Integer);
    function AddNewButtons: boolean;
    function DeleteOldButtons: boolean;
    procedure SetActiveButton;
    procedure SynhronizeScrollBar;
    function CorrectSize(_Width, _Height: Integer): Boolean;
    function SetCorrectSize(_Width, _Height: Integer): Boolean;
    procedure CalculatePictureRect;
    procedure CalculateButtonsState;
    procedure CalculateFirstButton;
    procedure CalculateScrollBar;
    procedure CalculateMouse(X, Y: Integer);
    procedure ShowActiveButton;
    procedure ShowLines;
    procedure SetParent(AParent: TWinControl); override;
    procedure RefreshBar(_ScrollBar, _RePaint: boolean);
    procedure ScrollBarScroll(Sender: TObject; ScrollCode: TScrollCode;
                              var ScrollPos: Integer);
    procedure ButtonClick(Sender: TObject);
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    procedure WMTMMessage(var Msg: TMessage); message WM_TBMESSAGE;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure RefreshTaskBar;
    procedure Paint; override;
    property ButtonCount: Integer read GetButtonCount;
    property Buttons[Index: Integer]: TllTaskButton read GetButton;
    property Form: TForm read FForm;
  published
    { Published declarations }
    property Align default alBottom;
    property Bitmap: TBitmap read FBitmap Write SetBitmap;
    property ButtonTransparent: Boolean read FButtonTransparent write SetButtonTransparent;
    property Color;
    property Constraints;
    property Ctl3D;
    property Enabled;
    property Kind: TllTaskBarKind read FKind Write SetKind;
    property Flat: Boolean read FFlat write SetFlat;
    property Font;
    property HideMinimizedMDI: Boolean read FHideMinimizedMDI Write SetHideMinimizedMDI;
    property MinButtonWidth: Integer read FMinButtonWidth Write SetMinButtonWidth;
    property MaxButtonWidth: Integer read FMaxButtonWidth Write SetMaxButtonWidth;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property Picture: TBitmap read FPicture Write SetPicture;
    property PictureAlign: TPictureAlign read FPictureAlign Write SetPictureAlign;
    property PictureBevel: Boolean read FPictureBevel Write SetPictureBevel;
    property PictureCentered: Boolean read FPictureCentered Write SetPictureCentered;
    property PictureMaximize: Boolean read FPictureMaximize Write SetPictureMaximize;
    property PictureRatio: Boolean read FPictureRatio Write SetPictureRatio;
    property PictureTransparent: Boolean read FPictureTransparent Write SetPictureTransparent;
    property PictureWidth: Integer read FPictureWidth Write SetPictureWidth;
    property PopupMenu;
    property ShowHint;
    property SynchronizeCaption: Boolean read FSynchronizeCaption Write SetSynchronizeCaption default True;
    property SynchronizeIcon: Boolean read FSynchronizeIcon Write SetSynchronizeIcon default True;
    property TabOrder;
    property TabStop default False;
    property Visible;
    property OnCanResize;
    property OnClick;
    property OnConstrainedResize;
{$IFNDEF VER120}
    property OnContextPopup;
{$ENDIF}
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
  end;

implementation

var HookHandle: HHOOK;
    llTaskBarHandle: HWND;

function HookProc(Code: Integer; wParam: wParam; var Msg: TMsg): LRESULT stdcall;
begin
  if Code >= 0 then begin
    if (Msg.message = WM_PAINT) or (Msg.message = WM_NCLBUTTONDOWN) or (Msg.message = WM_LBUTTONDOWN) then
      SendMessage(llTaskBarHandle, WM_TBMESSAGE, Msg.message, 0);
    Result := 0;
  end else
    Result := CallNextHookEx(HookHandle, Code, wParam, Longint(@Msg));
end;

procedure DrawIconToBitmap(Bitmap: TBitmap; HIcon: HICON);
var
  Icon: TIcon;
  _Bitmap: TBitmap;
begin
  Icon := TIcon.Create;
  try
    Icon.Handle := HIcon;
    _Bitmap := TBitmap.Create;
    try
      _Bitmap.Width := 16;
      _Bitmap.Height := 16;
      with _Bitmap.Canvas do begin
        DrawIconEx(_Bitmap.Canvas.Handle, 0, 0, Icon.Handle, 16, 16, 0, _Bitmap.Canvas.Brush.Handle, DI_NORMAL);
      end;
      _Bitmap.Transparent := True;
      Bitmap.Assign(_Bitmap);
    finally
      _Bitmap.Free;
    end;
  finally
    Icon.ReleaseHandle;
    Icon.Free
  end;
end;

{ TllTaskButton }

constructor TllTaskButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FForm := nil;
  OldWndProc := nil;
  WndProcPtr := nil;
  Margin := 2;
  Hint := '';
  CalculateCaption;
end;

destructor TllTaskButton.Destroy;
  var i: Integer;
begin
  for i := 0 to TllTaskBar(Parent).Form.MDIChildCount - 1 do
    if FForm = TllTaskBar(Parent).Form.MDIChildren[i] then begin
      if FForm <> nil then begin
        PopupMenu := nil;
        SetWindowLong(FForm.Handle, GWL_WNDPROC, Longint(OldWndProc));
        {$IFDEF VER140}
        {$WARNINGS OFF}
        {$ENDIF}
        FreeObjectInstance(WndProcPtr);
        {$IFDEF VER140}
        {$WARNINGS ON}
        {$ENDIF}
        OldWndProc := nil;
        WndProcPtr := nil;
      end;
      Break;
    end;
  if WndProcPtr <> nil then begin
    {$IFDEF VER140}
    {$WARNINGS OFF}
    {$ENDIF}
    FreeObjectInstance(WndProcPtr);
    {$IFDEF VER140}
    {$WARNINGS ON}
    {$ENDIF}
    WndProcPtr := nil;
  end;
  inherited;
end;

function TllTaskButton.GetDown: Boolean;
begin
  Result := inherited Down;
end;

procedure TllTaskButton.SetDown(Value: Boolean);
  var _Down: Boolean;
begin
  _Down := inherited Down;
  inherited Down := Value;
  if Value then
    Font.Style := Font.Style + [fsBold]
  else
    Font.Style := Font.Style - [fsBold];
  if _Down <> Value then
    CalculateCaption;
end;

procedure TllTaskButton.SetForm(Value: TForm);
begin
  if FForm <> Value then begin
    if FForm <> nil then begin
      SetWindowLong(FForm.Handle, GWL_WNDPROC, Longint(OldWndProc));
      {$IFDEF VER140}
      {$WARNINGS OFF}
      {$ENDIF}
      FreeObjectInstance(WndProcPtr);
      {$IFDEF VER140}
      {$WARNINGS ON}
      {$ENDIF}
      OldWndProc := nil;
      WndProcPtr := nil;
      Hint := '';
    end;
    FForm := Value;
    if FForm <> nil then begin
      Hint := FForm.Caption;
      Glyph.Width := 16;
      Glyph.Height := 16;
      if FForm.Icon.Handle <> 0 then
        DrawIconToBitmap(Glyph, FForm.Icon.Handle)
      else
        DrawIconToBitmap(Glyph, Application.Icon.Handle);
      {$IFDEF VER140}
      {$WARNINGS OFF}
      {$ENDIF}
      WndProcPtr := MakeObjectInstance(WndMethod);
      {$IFDEF VER140}
      {$WARNINGS ON}
      {$ENDIF}
      OldWndProc := Pointer(SetWindowLong(FForm.Handle,GWL_WNDPROC, Integer(WndProcPtr)));
    end else
      Hint := '';
    CalculateCaption;
  end;
end;

procedure TllTaskButton.CMButtonPressed(var Message: TMessage);
var
  Sender: TllTaskButton;
begin
  inherited;
  if Message.WParam = GroupIndex then
  begin
    Sender := TllTaskButton(Message.LParam);
    if Sender <> Self then
    begin
      if Sender.Down and not Down then
        if fsBold in Font.Style then begin
          Font.Style := Font.Style - [fsBold];
          CalculateCaption;
        end;
    end;
  end;
end;

procedure TllTaskButton.WndMethod(var Msg: TMessage);
begin
  case Msg.Msg of
    WM_DESTROY, WM_NCDESTROY: begin
                  SetWindowLong(FForm.Handle, GWL_WNDPROC, Longint(OldWndProc));
                  {$IFDEF VER140}
                  {$WARNINGS OFF}
                  {$ENDIF}
                  FreeObjectInstance(WndProcPtr);
                  {$IFDEF VER140}
                  {$WARNINGS ON}
                  {$ENDIF}
                  OldWndProc := nil;
                  WndProcPtr := nil;
                  Exit;
                end;
    WM_SIZE: begin
               if TllTaskBar(Parent).HideMinimizedMDI then
                 case Msg.WParam of
                   SIZE_MINIMIZED: ShowWindow(FForm.Handle, SW_HIDE);
                   else if FForm.WindowState = wsMinimized then
                     ShowWindow(FForm.Handle, SW_SHOWNOACTIVATE);
                 end;
             end;
    WM_GETTEXT: if TllTaskBar(Parent).SynchronizeCaption then
                  if Hint <> FForm.Caption then begin
                    Hint := FForm.Caption;
                    CalculateCaption;
                  end;
    WM_SETICON: if TllTaskBar(Parent).SynchronizeIcon then
                  DrawIconToBitmap(Glyph, HICON(Msg.lParam));
  end;
  with Msg do
    Result := CallWindowProc(OldWndProc, FForm.Handle, Msg, WParam, LParam);
end;

procedure TllTaskButton.CalculateCaption;
  var CaptionWidth: Integer;
      tmpStr: string;
      StrWidth: Integer;
      Coeff: Extended;
      AddWidth: Integer;
begin
  if FForm = nil then begin
    ShowHint := False;
    Exit;
  end;
  tmpStr := Hint;
  Caption := tmpStr;
  if Canvas.Font.Style <> Font.Style then
    Canvas.Font.Style := Font.Style;
  StrWidth := Canvas.TextWidth(tmpStr);
  if StrWidth = 0 then begin
    ShowHint := False;
    Exit;
  end;
  CaptionWidth := Width - 6;
  if not Glyph.Empty then
    CaptionWidth := CaptionWidth - Glyph.Width - 6;
  if CaptionWidth <= 0 then begin
    ShowHint := True;
    Exit;
  end;
  if CaptionWidth >= StrWidth then begin
    ShowHint := False;
    Exit;
  end;
  ShowHint := True;
  AddWidth := Canvas.TextWidth('...');
  CaptionWidth := CaptionWidth - AddWidth;
  if CaptionWidth <= 0 then begin 
    Caption := Hint[1] + '...';
    Exit;
  end;
  Coeff := StrWidth / CaptionWidth;
  tmpStr := Copy(tmpStr, 1, Round(Length(tmpStr) / Coeff) + 1);
  StrWidth := Canvas.TextWidth(tmpStr);
  while StrWidth > CaptionWidth do begin
      SetLength(tmpStr, Length(tmpStr) - 1);
      StrWidth := Canvas.TextWidth(tmpStr);
  end;
  if tmpStr = '' then
    tmpStr := Hint[1];
  Caption := tmpStr + '...';
end;

procedure TllTaskButton.Resize;
begin
  inherited;
  CalculateCaption;
end;

{ TllTaskBar }

const IndentVertical = 2;
      IndentHorizontal = 2;

constructor TllTaskBar.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := [csCaptureMouse, csClickEvents, csOpaque, csReplicatable];

  FCountLines := 0;
  FCountVisibleLines := 1;
  FHideMinimizedMDI := True;

  FButtonWidth := 150;
  FButtonHeight := 22;
  FMinButtonWidth := FButtonHeight;
  FMaxButtonWidth := FButtonWidth;
  FButtonTransparent := False;
  FButtons := TList.Create;

  FScrollBarWidth := 16;
  FScrollBar := TScrollBar.Create(Self);
  FScrollBar.Visible := False;
  FScrollBar.Kind := sbVertical;
  FScrollBar.Width := FScrollBarWidth;
  FScrollBar.Height := FButtonHeight;
  FScrollBar.Left := FMinButtonWidth + IndentHorizontal * 2;
  FScrollBar.Top := IndentVertical;
  FScrollBar.Min := 1;
  FScrollBar.Max := 1;
  FScrollBar.Position := 1;
  FScrollBar.TabStop := False;
  FScrollBar.OnScroll := ScrollBarScroll;

  FPicture := TBitmap.Create;
  FPictureWidth := FButtonHeight;
  FPictureAlign := paRightBottom;
  FPictureCentered := True;
  FPictureRatio := True;
  FPictureMaximize := True;
  FPictureBevel := False;

  CalculatePictureRect;

  Width := FMinButtonWidth + IndentHorizontal * 3 + FScrollBarWidth;
  Height := FButtonHeight + IndentVertical * 2;
  Color := clBtnFace;
  Align := alBottom;
  FFlat := False;
  FActiveButton := -1;
  FForm := GetMainForm(AOwner);
  FKind := lltbHorizontal;
  FBitmap := TBitmap.Create;
  FResizing := [];
  FCursorChanged := False;
  FNotRePaint := False;
  FInternalCorrectSize := False;
  FSynchronizeIcon := True;
  FSynchronizeCaption := True;

  CalculateButtonsState;
  FScrollBar.Parent := Self;
  Parent := AOwner as TWinControl;

  if (FForm <> nil) and (not (csDesigning in ComponentState)) then begin
    llTaskBarHandle := Handle;
    HookHandle := SetWindowsHookEx(WH_GETMESSAGE,@HookProc,0,GetCurrentThreadId);
  end;
end;

destructor TllTaskBar.Destroy;
  var I: Integer;
begin
  for I := 0 to FButtons.Count - 1 do
    TllTaskButton(FButtons[I]).Free;
  FButtons.Free;
  FScrollBar.Free;
  FPicture.Free;
  if (FForm <> nil) and (not (csDesigning in ComponentState)) then
    UnhookWindowsHookEx(HookHandle);
  inherited Destroy;
end;

procedure TllTaskBar.SetParent(AParent: TWinControl);
begin
  if (FForm <> nil) and (not (csDesigning in ComponentState)) then
    UnhookWindowsHookEx(HookHandle);
  inherited SetParent(AParent);
  FForm := GetMainForm(AParent);
  if (FForm <> nil) and (not (csDesigning in ComponentState)) then begin
    llTaskBarHandle := Handle;
    HookHandle := SetWindowsHookEx(WH_GETMESSAGE,@HookProc,0,GetCurrentThreadId);
  end;
end;

function TllTaskBar.GetMainForm(AOwner: TComponent): TForm;
  var _Parent: TWinControl;
begin
  _Parent := AOwner as TWinControl;
  while not(_Parent is TForm) and (_Parent <> nil) do
    _Parent := _Parent.Parent;
  if _Parent is TForm then
    Result := TForm(_Parent)
  else
    Result := nil;
end;

function TllTaskBar.GetButton(Index: Integer): TllTaskButton;
begin
  Result := FButtons[Index];
end;

function TllTaskBar.GetButtonCount: Integer;
begin
  Result := FButtons.Count;
end;

procedure TllTaskBar.SetMinButtonWidth(Value: Integer);
  var i: Integer;
begin
  if (FMinButtonWidth <> Value) and (FMaxButtonWidth >= Value) then begin
    FMinButtonWidth := Value;
    if Value > FButtonWidth then begin
      FButtonWidth := Value;
      for i := 0 to FButtons.Count-1 do
        TllTaskButton(FButtons.Items[i]).Width := Value;
      SetCorrectSize(Width, Height);
      CalculateButtonsState;
    end;
  end;
end;

procedure TllTaskBar.SetMaxButtonWidth(Value: Integer);
  var i: Integer;
begin
  if (FMaxButtonWidth <> Value) and (FMinButtonWidth <= Value) then begin
    FMaxButtonWidth := Value;
    if Value < FButtonWidth then begin
      FButtonWidth := Value;
      for i := 0 to FButtons.Count-1 do
        TllTaskButton(FButtons.Items[i]).Width := Value;
      CalculateButtonsState;
    end;
  end;
end;

procedure TllTaskBar.SetPictureAlign(Value: TPictureAlign);
begin
  if FPictureAlign = Value then
    Exit;
  FPictureAlign := Value;
  RefreshBar(True, True);
end;

procedure TllTaskBar.SetPictureBevel(Value: Boolean);
begin
  if FPictureBevel = Value then
    Exit;
  FPictureBevel := Value;
  CalculatePictureRect;
  RePaint;
end;

procedure TllTaskBar.SetBitmap(Value: TBitmap);
begin
  FBitmap.Assign(Value);
  RePaint;
end;

procedure TllTaskBar.SetPicture(Value: TBitmap);
begin
  FPicture.Assign(Value);
  if not FPicture.Empty then
    if FKind = lltbHorizontal then begin
      if Width < (FMinButtonWidth + IndentHorizontal * 4 + FScrollBarWidth + FPictureWidth) then
        Width := FMinButtonWidth + IndentHorizontal * 4 + FScrollBarWidth + FPictureWidth;
    end else begin
      if Height < (FButtonHeight + IndentHorizontal * 4 + FScrollBarWidth + FPictureWidth) then
        Height := FButtonHeight + IndentHorizontal * 4 + FScrollBarWidth + FPictureWidth;
    end;
  FPicture.Transparent := FPictureTransparent;
  SetCorrectSize(Width, Height);
  RefreshBar(True, True);
end;

procedure TllTaskBar.SetPictureWidth(Value: Integer);
begin
  if FPictureWidth = Value then
    Exit;
  FPictureWidth := Value;
  SetCorrectSize(Width, Height);
  RefreshBar(True, True);
end;

procedure TllTaskBar.SetPictureCentered(Value: Boolean);
begin
  if FPictureCentered = Value then
    Exit;
  FPictureCentered := Value;
  CalculatePictureRect;
  RePaint;
end;

procedure TllTaskBar.SetPictureRatio(Value: Boolean);
begin
  if FPictureRatio = Value then
    Exit;
  FPictureRatio := Value;
  CalculatePictureRect;
  RePaint;
end;

procedure TllTaskBar.SetPictureTransparent(Value: Boolean);
begin
  if FPictureTransparent = Value then
    Exit;
  FPictureTransparent := Value;
  if not FPicture.Empty then begin
    FPicture.Transparent := FPictureTransparent;
    RePaint;
  end;
end;

procedure TllTaskBar.SetPictureMaximize(Value: Boolean);
begin
  if FPictureMaximize = Value then
    Exit;
  FPictureMaximize := Value;
  CalculatePictureRect;
  RePaint;
end;

procedure TllTaskBar.SetFlat(Value: Boolean);
  var i: Integer;
begin
  if FFlat = Value then
    Exit;
  FFlat := Value;
  for i := 0 to FButtons.Count-1 do
    TllTaskButton(FButtons.Items[i]).Flat := FFlat;
end;

procedure TllTaskBar.SetButtonTransparent(Value: Boolean);
  var i: Integer;
begin
  if FButtonTransparent = Value then
    Exit;
  FButtonTransparent := Value;
  for i := 0 to FButtons.Count-1 do
    TllTaskButton(FButtons.Items[i]).Transparent := FButtonTransparent;
end;

procedure TllTaskBar.SetKind(Value: TllTaskBarKind);
begin
  if FKind = Value then
    Exit;
  FKind := Value;
  SetCorrectSize(Width, Height);
  RefreshBar(True, True);
end;

procedure TllTaskBar.SetHideMinimizedMDI(Value: Boolean);
  var i: Integer;
begin
  if FHideMinimizedMDI <> Value then
    FHideMinimizedMDI := Value;
  for i := 0 to FButtons.Count - 1 do
    if TllTaskButton(FButtons.Items[i]).Form.WindowState = wsMinimized then
      if Value then
        ShowWindow(TllTaskButton(FButtons.Items[i]).Form.Handle, SW_HIDE)
      else
        ShowWindow(TllTaskButton(FButtons.Items[i]).Form.Handle, SW_SHOWMINNOACTIVE);
end;

procedure TllTaskBar.SetSynchronizeCaption(Value: Boolean);
  var i: Integer;
begin
  if FSynchronizeCaption = Value then
    Exit;
  FSynchronizeCaption := Value;
  if Value then
    for i := 0 to FButtons.Count - 1 do begin
      TllTaskButton(FButtons.Items[i]).Hint := TllTaskButton(FButtons.Items[i]).Form.Caption;
      TllTaskButton(FButtons.Items[i]).CalculateCaption;
    end;
end;

procedure TllTaskBar.SetSynchronizeIcon(Value: Boolean);
  var i: Integer;
begin
  if FSynchronizeIcon = Value then
    Exit;
  FSynchronizeIcon := Value;
  if Value then
    for i := 0 to FButtons.Count - 1 do
      DrawIconToBitmap(TllTaskButton(FButtons.Items[i]).Glyph, TllTaskButton(FButtons.Items[i]).Form.Icon.Handle);
end;

procedure TllTaskBar.Paint;
  var _Rect: TRect;
      x, y: Integer;
begin
  if FNotRePaint then
    Exit;
  with Canvas do begin
    Brush.Color := Color;
    FillRect(GetClientRect);
    if not FBitmap.Empty then begin
      x := 0;
      while x < Width do begin
        y := 0;
        while y < Height do begin
          Canvas.Draw(x, y, FBitmap);
          y := y + FBitmap.Height;
        end;
        x := x + FBitmap.Width;
      end;
    end;
  end;
  FScrollBar.Repaint;
  if (FPictureWidth > 0) and (not FPicture.Empty) and (FPicture.Width > 0 ) and (FPicture.Height > 0) then begin
    if (FPictureRect.Left = 0) and (FPictureRect.Right = 0) and
       (FPictureRect.Top = 0) and (FPictureRect.Bottom = 0) then begin
      CalculateScrollBar;
      CalculatePictureRect;
    end;
    if FPictureBevel then begin
      _Rect := Rect(FPictureRect.Left - 1, FPictureRect.Top - 1, FPictureRect.Right + 1, FPictureRect.Bottom + 1);
      Frame3D(Canvas, _Rect, clBtnShadow, clBtnHighlight, 1);
    end;
    Canvas.StretchDraw(FPictureRect, FPicture);
  end;
end;

function TllTaskBar.FindForm(Form: TForm): Boolean;
  var i: Integer;
begin
  Result := False;
  for i := 0 to TForm(FForm).MDIChildCount - 1 do
    if TForm(FForm).MDIChildren[i] = Form then begin
      Result := True;
      Break;
    end;
end;

function TllTaskBar.FindByForm(_Form: TForm): Integer;
  var Res, i: Integer;
begin
  Res := -1;
  if Assigned(_Form) then
    for i := 0 to FButtons.Count-1 do
      if TllTaskButton(FButtons[i]).FForm = _Form then begin
        Res := i;
        Break;
      end;
  Result := Res;
end;

function TllTaskBar.AddButton(_Form: TForm): Integer;
  var Index: Integer;
begin
  Result := -1;
  if FindByForm(_Form) >= 0 then
    Exit;
  Index := FButtons.Add(TllTaskButton.Create(Self));
  with TllTaskButton(FButtons.Items[Index]) do
    begin
      Inc(FCountButton);
      Parent := Self;
      Visible := False;
      Name := Self.Name + 'llTaskButton' + IntToStr(FCountButton);
      Tag := FCountButton;
      Top := 1;
      Left := 1 + (FButtonWidth + 2) * Index;
      Height := FButtonHeight;
      Width := FButtonWidth;
      GroupIndex := 1;
      Transparent := FButtonTransparent;
      Flat := FFlat;
      Form:=_Form;
      OnClick := ButtonClick;
      if FHideMinimizedMDI and (_Form.WindowState = wsMinimized) then
        ShowWindow(_Form.Handle, SW_HIDE);
    end;
  Result := Index;
end;

procedure TllTaskBar.DeleteButton(Index: Integer);
begin
  if (Index < 0) or (Index >= FButtons.Count) then
    Exit;
  TllTaskButton(FButtons[Index]).Free;
  FButtons.Delete(Index);
end;

function TllTaskBar.AddNewButtons: Boolean;
  var i: Integer;
      prz: Boolean;
begin
  prz := False;
  for i := TForm(FForm).MDIChildCount - 1 downto 0 do
    if FindByForm(TForm(FForm).MDIChildren[i]) = -1 then begin
      AddButton(TForm(FForm).MDIChildren[i]);
      prz := True;
    end;
  Result := prz;
  if prz then
    SetActiveButton;
end;

function TllTaskBar.DeleteOldButtons: Boolean;
  var i: Integer;
      prz: Boolean;
begin
  i := 0;
  prz := False;
  while i <= (FButtons.Count - 1) do
    if not FindForm(TllTaskButton(FButtons[i]).Form) then begin
      DeleteButton(i);
      prz := True;
    end else
      Inc(i);
  Result := prz;
  if prz then
    SetActiveButton;
end;

procedure TllTaskBar.SetActiveButton;
  var Index: Integer;
      ButtonHasChanged : Boolean;
begin
  Index := FindByForm(FForm.ActiveMDIChild);
  ButtonHasChanged := (FActiveButton <> Index);

  if ButtonHasChanged then
    FActiveButton := Index;

  If (ButtonHasChanged) and (Index>=0) then begin
    TllTaskButton(FButtons.Items[Index]).Down := True;
    TllTaskButton(FButtons.Items[Index]).CalculateCaption;
  end;
end;

procedure TllTaskBar.CalculatePictureRect;
  var pWidth, pHeight, fpWidth, fpHeight, IndentX, IndentY: Integer;
begin
  if FPicture.Empty or (FPicture.Width < 2) or (FPicture.Height < 2) or (FPictureWidth < 3) then begin
    FPictureRect := Rect(0,0,0,0);
    Exit;
  end;
  case FPictureAlign of
    paRightBottom:begin
                    FPictureRect.Right := Width - IndentHorizontal - 1;
                    FPictureRect.Bottom := Height - IndentVertical - 1;
                    if FKind = lltbHorizontal then begin
                      FPictureRect.Left := FPictureRect.Right - FPictureWidth + 1;
                      FPictureRect.Top := IndentVertical;
                    end else begin
                      FPictureRect.Left := IndentHorizontal;
                      FPictureRect.Top := FPictureRect.Bottom - FPictureWidth + 1;
                    end;
                  end;
    paLeftTop:begin
                FPictureRect.Left := IndentHorizontal;
                FPictureRect.Top := IndentVertical;
                if FKind = lltbHorizontal then begin
                  FPictureRect.Right := FPictureRect.Left + FPictureWidth - 1;
                  FPictureRect.Bottom := Height - IndentHorizontal - 1;
                end else begin
                  FPictureRect.Right := Width - IndentHorizontal - 1;
                  FPictureRect.Bottom := FPictureRect.Top + FPictureWidth - 1;
                end;
              end;
  end;
  if FPictureBevel then begin
    FPictureRect.Left := FPictureRect.Left + 1;
    FPictureRect.Right := FPictureRect.Right - 1;
    FPictureRect.Top := FPictureRect.Top + 1;
    FPictureRect.Bottom := FPictureRect.Bottom - 1;
  end;
  if FPictureRatio then begin
    pWidth := FPicture.Width;
    pHeight := FPicture.Height;
    fpWidth := FPictureRect.Right - FPictureRect.Left + 1;
    fpHeight := FPictureRect.Bottom - FPictureRect.Top + 1;
    IndentX := 0;
    IndentY := 0;
    if fpWidth < pWidth then begin
      pHeight := round(pHeight * fpWidth / pWidth);
      pWidth := fpWidth;
    end else if FPictureMaximize and (fpWidth > pWidth) then begin
      pHeight := round(pHeight * fpWidth / pWidth);
      pWidth := fpWidth;
    end;
    if fpHeight < pHeight then begin
      pWidth := round(pWidth * fpHeight / pHeight);
      pHeight := fpHeight;
    end;
    if FPictureCentered then begin
      IndentY := (fpHeight - pHeight) div 2;
      IndentX := (fpWidth - pWidth) div 2;
    end;
    FPictureRect.Left := FPictureRect.Left + IndentX;
    FPictureRect.Right := FPictureRect.Left + pWidth - 1;
    FPictureRect.Top := FPictureRect.Top + IndentY;
    FPictureRect.Bottom := FPictureRect.Top + pHeight - 1;
  end;
end;

function TllTaskBar.SetCorrectSize(_Width, _Height: Integer): Boolean;
  var prz: Boolean;
      CountLines, CountButtons: Integer;
      nWidth, nHeight: Integer;
begin
  prz := False;
  nWidth := _Width;
  if FKind = lltbHorizontal then begin
    CountLines := Round((_Height - IndentVertical)/(FButtonHeight + IndentVertical));
    if CountLines < 1 then
      CountLines := 1;
    nHeight := (FButtonHeight + IndentVertical) * CountLines + IndentVertical;
    if FPicture.Empty then begin
      if nWidth < (FMinButtonWidth + IndentHorizontal * 3 + FScrollBarWidth) then
        nWidth := FMinButtonWidth + IndentHorizontal * 3 + FScrollBarWidth;
    end else
      if nWidth < (FMinButtonWidth + IndentHorizontal * 4 + FScrollBarWidth + FPictureWidth) then
        nWidth := FMinButtonWidth + IndentHorizontal * 4 + FScrollBarWidth + FPictureWidth;
  end else begin
    if nWidth < (FMinButtonWidth + IndentHorizontal * 2) then
      nWidth := FMinButtonWidth + IndentHorizontal * 2;
    if FPicture.Empty then begin
      CountButtons := Round((_Height - IndentVertical * 2 - FScrollBarWidth)/(FButtonHeight + IndentVertical));
      if CountButtons < 1 then
        CountButtons := 1;
      nHeight := (FButtonHeight + IndentVertical) * CountButtons + IndentVertical * 2 + FScrollBarWidth;
    end else begin
      CountButtons := Round((_Height - IndentVertical * 3 - FScrollBarWidth - FPictureWidth)/(FButtonHeight + IndentVertical));
      if CountButtons < 1 then
        CountButtons := 1;
      nHeight := (FButtonHeight + IndentVertical) * CountButtons + IndentVertical * 3 + FScrollBarWidth + FPictureWidth;
    end;
  end;
  FInternalCorrectSize := True;
  if nWidth <> Width then begin
    Width := nWidth;
    prz := True;
  end;
  if nHeight <> Height then begin
    Height := nHeight;
    prz := True;
  end;
  FInternalCorrectSize := False;
  Result := prz;
end;

function TllTaskBar.CorrectSize(_Width, _Height: Integer): boolean;
  var prz: Boolean;
      CountLines, RestLines, CountButtons, RestButtons: Integer;
begin
  if (_Height <= 0) or (_Width <= 0) then
    prz := False
  else
    prz := True;

  CountLines := 1;
  RestLines := 0;
  CountButtons := 1;
  RestButtons := 0;

  if FKind = lltbHorizontal then begin
    if (_Height <> Height) and (_Height > 0) then begin
      CountLines := (_Height - IndentVertical) div (FButtonHeight + IndentVertical);
      RestLines := (_Height - IndentVertical) mod (FButtonHeight + IndentVertical);
    end;
    if (_Width <> Width) and (_Width > 0) then begin
      if FPicture.Empty then
        CountButtons := (_Width - IndentHorizontal * 2 - FScrollBarWidth) div (FMinButtonWidth + IndentHorizontal)
      else
        CountButtons := (_Width - IndentHorizontal * 3 - FScrollBarWidth - FPictureWidth) div (FMinButtonWidth + IndentHorizontal);
      RestButtons := 0;
    end;
  end else begin
    if (_Width <> Width) and (_Width > 0) then begin
      if _Width < (FMinButtonWidth + IndentHorizontal * 2) then
        CountLines := 0
      else
        CountLines := 1;
      RestLines := 0;
    end;
    if (_Height <> Height) and (_Height > 0) then begin
      if FPicture.Empty then begin
        CountButtons := (_Height - IndentVertical * 2 - FScrollBarWidth) div (FButtonHeight + IndentVertical);
        RestButtons := (_Height - IndentVertical * 2 - FScrollBarWidth) mod (FButtonHeight + IndentVertical);
      end else begin
        CountButtons := (_Height - IndentVertical * 3 - FScrollBarWidth - FPictureWidth) div (FButtonHeight + IndentVertical);
        RestButtons := (_Height - IndentVertical * 3 - FScrollBarWidth - FPictureWidth) mod (FButtonHeight + IndentVertical);
      end;
    end;
  end;

  if (CountLines < 1) or (RestLines > 0) or (CountButtons < 1) or (RestButtons > 0) then
    prz := False;

  Result := prz;
end;

procedure TllTaskBar.CalculateScrollBar;
begin
  if FKind = lltbHorizontal then begin
    if FScrollBar.Kind <> sbVertical then
      FScrollBar.Kind := sbVertical;
    FScrollBar.Width := FScrollBarWidth;
    FScrollBar.Height := Height - IndentVertical * 2;
    FScrollBar.Left := Width - IndentHorizontal - FScrollBarWidth;
    if (not FPicture.Empty) and (FPictureAlign = paRightBottom) then
      FScrollBar.Left := FScrollBar.Left - IndentHorizontal - FPictureWidth;
    FScrollBar.Top := IndentVertical;
  end else begin
    if FScrollBar.Kind <> sbHorizontal then
      FScrollBar.Kind := sbHorizontal;
    FScrollBar.Width := Width - IndentVertical * 2;
    FScrollBar.Height := FScrollBarWidth;
    FScrollBar.Left := IndentHorizontal;
    FScrollBar.Top := Height - IndentVertical - FScrollBarWidth;
    if (not FPicture.Empty) and (FPictureAlign = paRightBottom) then
      FScrollBar.Top := FScrollBar.Top - IndentVertical - FPictureWidth;
  end;
end;

procedure TllTaskBar.CalculateButtonsState;
  var _Width: Integer;
begin
  if FKind = lltbHorizontal then begin
    FLineHeight := FButtonHeight + IndentVertical;
    FCountVisibleLines := (Height - IndentVertical) div FLineHeight;
    FLineWidth := Width - IndentHorizontal;                          
    if not FPicture.Empty then
      FLineWidth := FLineWidth - IndentHorizontal - FPictureWidth;   
    FCountButtonInLine := FButtons.Count div FCountVisibleLines;     
    if (FButtons.Count mod FCountVisibleLines) > 0 then
      Inc(FCountButtonInLine);
    if FCountButtonInLine = 0 then
      FCountButtonInLine := 1;
    _Width := FLineWidth div FCountButtonInLine - IndentHorizontal;  

    if _Width < FMinButtonWidth then begin
      FScrollBar.Visible := True;                                    
      FLineWidth := FLineWidth - IndentHorizontal - FScrollBar.Width;
      FCountButtonInLine := FButtons.Count div FCountVisibleLines;   
      if (FButtons.Count mod FCountVisibleLines) > 0 then
        Inc(FCountButtonInLine);
      if FCountButtonInLine = 0 then
        FCountButtonInLine := 1;
      _Width := FLineWidth div FCountButtonInLine - IndentHorizontal;
    end else
      FScrollBar.Visible := False;                                   

    if _Width < FMinButtonWidth then                                 
      _Width := FMinButtonWidth
    else if _Width > FMaxButtonWidth then
      _Width := FMaxButtonWidth;
    FButtonWidth := _Width;                                          
    FCountButtonInLine := FLineWidth div (FButtonWidth + IndentHorizontal);
    if FCountButtonInLine = 0 then
      FCountButtonInLine := 1;
    FCountLines := FButtons.Count div FCountButtonInLine;          
    if (FButtons.Count mod FCountButtonInLine) > 0 then
      Inc(FCountLines);
  end else begin
    FCountVisibleLines := 1;
    FLineHeight := Height - IndentVertical;
    FLineWidth := (Width - IndentHorizontal) div FCountVisibleLines - IndentHorizontal; 
    if not FPicture.Empty then
      FLineHeight := FLineHeight - IndentVertical - FPictureWidth;
    FCountButtonInLine := FLineHeight div (FButtonHeight + IndentVertical);            
    if FCountButtonInLine = 0 then
      FCountButtonInLine := 1;
    FCountLines := FButtons.Count div FCountButtonInLine;
    if (FButtons.Count mod FCountButtonInLine) > 0 then
      Inc(FCountLines);

    if FCountLines > FCountVisibleLines then begin 
      FScrollBar.Visible := True;                                   
      FLineHeight := FLineHeight - IndentVertical - FScrollBar.Height;
      FCountButtonInLine := FLineHeight div (FButtonHeight + IndentVertical); 
      if FCountButtonInLine = 0 then
        FCountButtonInLine := 1;
      FCountLines := FButtons.Count div FCountButtonInLine;
      if (FButtons.Count mod FCountButtonInLine) > 0 then
        Inc(FCountLines);
    end else
      FScrollBar.Visible := False;                                 

    FButtonWidth := FLineWidth;                                    
  end;
  CalculateFirstButton;
end;

procedure TllTaskBar.CalculateFirstButton;
begin
  FFirstButton := FActiveButton + 1 - FCountButtonInLine * FCountVisibleLines;
  if FFirstButton < 0 then
    FFirstButton := 0;
  SynhronizeScrollBar;
  ShowLines;
end;

procedure TllTaskBar.SynhronizeScrollBar;
  var a,b,Max,Pos:Integer;
begin
  a := FFirstButton div (FCountButtonInLine * FCountVisibleLines);
  if (FFirstButton mod (FCountButtonInLine * FCountVisibleLines)) > 0 then
    Inc(a);
  b := (FButtons.Count - FFirstButton) div (FCountButtonInLine * FCountVisibleLines);
  if ((FButtons.Count - FFirstButton) mod (FCountButtonInLine * FCountVisibleLines)) > 0 then
    Inc(b);
  Max := a + b;
  if Max = 0 then
    Max := FScrollBar.Min;
  FScrollBar.Max := Max;
  Pos := a + 1;
  if Pos < FScrollBar.Min then
    Pos := FScrollBar.Min
  else if Pos > FScrollBar.Max then
    Pos := FScrollBar.Max;
  FScrollBar.Position := Pos;
end;

procedure TllTaskBar.ShowActiveButton;
  var Index: Integer;
begin
  Index := FindByForm(FForm.ActiveMDIChild);
  if Index = -1 then
    Exit;
  if not TllTaskButton(FButtons.Items[Index]).Visible then
    CalculateFirstButton;
end;

procedure TllTaskBar.ShowLines;
  var i, j, _FirstButton, _LastButton: Integer;
      bDel: array of integer;
  procedure SetStateButton(Index: Integer);
    var i, Count: Integer;
        prz: Boolean;
  begin
    i := 0;
    prz := True;
    Count := Length(bDel);
    while (i < Count) and prz do begin
      if bDel[i] = Index then begin
        bDel[i] := bDel[Count - 1];
        SetLength(bDel, Count - 1);
        prz := False;
      end;
      Inc(i);
    end;
  end;
  procedure ShowButtons(_First,_Last,_Line: Integer);
    var i: Integer;
        IndentForPicture: Integer;
  begin
    if FKind = lltbHorizontal then begin
      if (Picture.Empty) or (FPictureAlign = paRightBottom) then
        IndentForPicture := 0
      else
        IndentForPicture := FPictureWidth + IndentHorizontal * 2;
      for i := _First to _Last do
        with TllTaskButton(FButtons[i]) do begin
          Width := FButtonWidth;
          Height := FButtonHeight;
          Left := IndentForPicture + IndentHorizontal + (FButtonWidth + IndentHorizontal) * (i - _First);
          Top := IndentVertical + FLineHeight * (_Line - 1);
          Visible := True;
        end;
    end else begin
      if Picture.Empty or (FPictureAlign = paRightBottom) then
        IndentForPicture := 0
      else
        IndentForPicture := FPictureWidth + IndentVertical * 2;
      for i := _First to _Last do
        with TllTaskButton(FButtons[i]) do begin
          Width := FButtonWidth;
          Height := FButtonHeight;
          Left := IndentHorizontal;
          Top := IndentForPicture + IndentVertical + (FButtonHeight + IndentVertical) * (i - _First);
          Visible := True;
        end;
    end;
  end;
begin
  if csDesigning in ComponentState then
    Exit;

  SetLength(bDel, 0);
  j := 0;
  for i := 0 to FButtons.Count - 1 do
    if TllTaskButton(FButtons.Items[i]).Visible then begin
      Inc(j);
      SetLength(bDel, j);
      bDel[j-1] := i;
    end;

  if FFirstButton >= 0 then begin
    _FirstButton := FFirstButton;
    _LastButton := _FirstButton + FCountButtonInLine * FCountVisibleLines - 1;
    if _LastButton > (FButtons.Count - 1) then
      _LastButton := FButtons.Count - 1;
    for i := _FirstButton to _LastButton do
      SetStateButton(i);
  end;

  FNotRePaint := True;

  for i := 0 to Length(bDel) - 1 do
    TllTaskButton(FButtons.Items[bDel[i]]).Visible := False;

  if FFirstButton >= 0 then begin
    _FirstButton := FFirstButton;
    for i:=1 to FCountVisibleLines do begin
      _LastButton := _FirstButton + FCountButtonInLine - 1;
      if _LastButton > (FButtons.Count - 1) then
        _LastButton := FButtons.Count - 1;
      ShowButtons(_FirstButton, _LastButton, i);
      _FirstButton := _LastButton + 1;
    end;
  end;

  FNotRePaint := False;
  RePaint;
end;

procedure TllTaskBar.RefreshTaskBar;
begin
  RefreshBar(True, True);
end;

procedure TllTaskBar.RefreshBar(_ScrollBar, _RePaint: Boolean);
begin
  if _RePaint then begin
    CalculatePictureRect;
    RePaint;
  end;
  if _ScrollBar then
    CalculateScrollBar;
  CalculateButtonsState;
end;

procedure TllTaskBar.ScrollBarScroll(Sender: TObject; ScrollCode: TScrollCode;
  var ScrollPos: Integer);
  var _FFirstButton: Integer;
begin
  _FFirstButton := FFirstButton;
  if ScrollCode = scLineDown then begin
    _FFirstButton := _FFirstButton + FCountButtonInLine * FCountVisibleLines;
    if (_FFirstButton + FCountButtonInLine * FCountVisibleLines) >= FButtons.Count then begin
      _FFirstButton := FButtons.Count - FCountButtonInLine * FCountVisibleLines;
      if _FFirstButton < 0 then
        _FFirstButton := 0;
    end;
  end else if ScrollCode = scLineUp then begin
    _FFirstButton := _FFirstButton - FCountButtonInLine * FCountVisibleLines;
    if _FFirstButton < 0 then
      _FFirstButton := 0;
  end;
  if FFirstButton <> _FFirstButton then begin
    FFirstButton := _FFirstButton;
    ShowLines;
  end;
end;

procedure TllTaskBar.ButtonClick(Sender: TObject);
  var Form: TForm;
      Index: Integer;
begin
  Form := (Sender as TllTaskButton).Form;
  Index := FindByForm(Form);
  if Index = FActiveButton then
    if Form.WindowState = wsMinimized then
      Form.WindowState := wsNormal
    else
      Form.WindowState := wsMinimized
  else begin
    if Form.WindowState = wsMinimized then
       Form.WindowState := wsNormal;
    SetWindowPos(Form.Handle, HWND_TOP, 0, 0, 0, 0, SWP_NOSIZE+SWP_NOMOVE);
    FActiveButton := FindByForm(Form);
  end;
end;

function TllTaskBar.CanResize(var NewWidth, NewHeight: Integer): Boolean;
  var prz: Boolean;
begin
  if ((Width = NewWidth) and (Height = NewHeight)) then begin
    Result := inherited CanResize(NewWidth, NewHeight);
    Exit;
  end;
  if not FInternalCorrectSize then
    SetCorrectSize(NewWidth, NewHeight);
  prz := CorrectSize(NewWidth, NewHeight);
  if prz then
    prz := inherited CanResize(NewWidth, NewHeight);
  Result := prz;
end;

procedure TllTaskBar.Resize;
begin
  inherited;
  RefreshBar(True, True);
end;

procedure TllTaskBar.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
  var prz: TResizing;
begin
  if Shift = [ssRight] then begin
  end;
  if (Shift = [ssLeft]) and (Align <> alClient) then begin
    prz := [];
    if (not (Align in [alTop, alLeft, alRight])) and (Y < 2) then
      prz := prz + [rYt]
    else if (not (Align in [alBottom, alLeft, alRight])) and (Y > (Height - 3)) then
      prz := prz + [rYb];
    if (not (Align in [alLeft, alTop, alBottom])) and (X < 2) then
      prz := prz + [rXt]
    else if (not (Align in [alRight, alTop, alBottom])) and (X > (Width - 3)) then
      prz := prz + [rXb];
    if prz <> [] then begin
      FMouseX := X;
      FMouseY := Y;
      FOldWidth := Width;
      FOldHeight := Height;
    end;
    FResizing := prz;
  end;
  inherited MouseDown(Button, Shift, X, Y);
end;

procedure TllTaskBar.CalculateMouse(X, Y: Integer);
  var prz: Integer;
begin
  prz := 0;
  if (not (Align in [alTop, alLeft, alRight])) and (Y < 2) then
    prz := 1
  else if (not (Align in [alBottom, alLeft, alRight])) and (Y > (Height - 3)) then
    prz := 2;
  if (not (Align in [alLeft, alTop, alBottom])) and (X < 2) then
    case prz of
      0: prz := 3;
      1: prz := 4;
      2: prz := 5;
    end
  else if (not (Align in [alRight, alTop, alBottom])) and (X > (Width - 3)) then
    case prz of
      0: prz := 3;
      1: prz := 5;
      2: prz := 4;
    end;
  if prz = 0 then begin
    if FCursorChanged then begin
      Screen.Cursor := FOldCursor;
      FCursorChanged := False;
    end;
  end else begin
    if not FCursorChanged then
      FOldCursor := Screen.Cursor;
    case prz of
      1,2:Screen.Cursor := crSizeNS;
      3:  Screen.Cursor := crSizeWE;
      4:  Screen.Cursor := crSizeNWSE;
      5:  Screen.Cursor := crSizeNESW;
    end;
    FCursorChanged := True;
  end;
end;

procedure TllTaskBar.MouseMove(Shift: TShiftState; X, Y: Integer);
  var dLeft, dTop: Integer;
begin
  if (FResizing = []) and (Align <> alClient) then begin
    CalculateMouse(X, Y);
  end else if FResizing <> [] then begin
    dLeft := X - FMouseX;
    dTop := Y - FMouseY;
    if rXt in FResizing then begin
      if SetCorrectSize(FOldWidth - dLeft, Height) then begin
        Left := Left + dLeft;
        FOldWidth := Width;
      end;
    end else if rXb in FResizing then begin
      if SetCorrectSize(FOldWidth + dLeft, Height) then begin
        FOldWidth := Width;
        FMouseX := Width - 1;
      end;
    end;
    if rYt in FResizing then begin
      if SetCorrectSize(Width, FOldHeight - dTop) then begin
        FOldHeight := Height;
        if dTop < 0 then
          Top := Top - FButtonHeight - IndentVertical
        else
          Top := Top + FButtonHeight + IndentVertical;
      end;
    end else if rYb in FResizing then begin
      if SetCorrectSize(Width, FOldHeight + dTop) then begin
        Height := FOldHeight + dTop;
        FMouseY := Height - 1;
        FOldHeight := Height;
      end;
    end;
  end;
  inherited MouseMove(Shift, X, Y);
end;

procedure TllTaskBar.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  inherited MouseUp(Button, Shift, X, Y);
  if FResizing <> [] then begin
    FResizing := [];
    if (not MouseCapture) and FCursorChanged then begin
      Screen.Cursor := FOldCursor;
      FCursorChanged := False;
    end;
  end;
end;

procedure TllTaskBar.CMMouseLeave(var Message: TMessage);
begin
  if (FResizing = []) and FCursorChanged then begin
    Screen.Cursor := FOldCursor;
    FCursorChanged := False;
  end;
  inherited;
end;

procedure TllTaskBar.WMTMMessage(var Msg: TMessage);
  var prz: Boolean;
begin
  prz := AddNewButtons;
  if DeleteOldButtons or prz then
    RefreshBar(False, False)
  else
    SetActiveButton;
end;

//{$i llshareware.inc}

end.
