import java.lang.Thread;

class Game extends Thread {

    private boolean threadSuspended;
    private boolean threadStopped;

    private Tetris tetris;

    private Playfield playfield;
    private Tetramino currentTetramino;
    private Tetramino nextTetramino;
    
    private boolean dropTetramino;
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
// CONSTRUCTORS                                                                             //
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//

    Game() {
    }

    Game(Tetris tetrisApplet) {

        tetris = tetrisApplet;

        threadSuspended = false;
        threadStopped = false;

        playfield = new Playfield(tetris);
        nextTetramino = new Tetramino(randomTetramino(),tetris);
        currentTetramino = nextTetramino;
    }


//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
// METHODES to ACCESS VARIABLES                                                             //
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//

    public Playfield getPlayfield() {
        return playfield;
    }
    public Tetramino getCurrentTetramino() {
        return currentTetramino;
    }
    public Tetramino getNextTetramino() {
        return nextTetramino;
    }
    public boolean getThreadSuspended() {
        return threadSuspended;
    }


//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
// PUBLIC METHODES for KEY EVENTS                                                           //
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//

    public synchronized void tryRotateTetraminoCW() {

        currentTetramino.rotateCW();
        
        if (playfield.tetraminoFits(currentTetramino)) {
            // System.out.println("Game.tryRotateTetraminoCW() - possible");
            currentTetramino.updateTetraminoBuffer();
            tetris.repaintPlayfield();
        }
        else {
            // System.out.println("Game.tryRotateTetraminoCW() - not possible");
            currentTetramino.rotateCCW();
        }
            // System.out.println("Game.tryRotateTetraminoCW() - ended");
    }

    public synchronized void tryRotateTetraminoCCW() {

        currentTetramino.rotateCCW();
        
        if (playfield.tetraminoFits(currentTetramino)) {
            // System.out.println("Game.tryRotateTetraminoCCW() - possible");
            currentTetramino.updateTetraminoBuffer();
            tetris.repaintPlayfield();
        }
        else {
            // System.out.println("Game.tryRotateTetraminoCCW() - not possible");
            currentTetramino.rotateCW();
        }
            // System.out.println("Game.tryRotateTetraminoCCW() - ended");
    }

    public synchronized void tryMoveTetraminoLeft() {

        currentTetramino.moveLeft();

        if (playfield.tetraminoFits(currentTetramino)) {
            tetris.repaintPlayfield();
        }
        else {
            currentTetramino.moveRight();
        }
    }

    public synchronized void tryMoveTetraminoRight() {

        currentTetramino.moveRight();

        if (playfield.tetraminoFits(currentTetramino)) {
            tetris.repaintPlayfield();
        }
        else {
            currentTetramino.moveLeft();
        }
    }

    public synchronized void tryMoveTetraminoDown() {

        currentTetramino.moveDown();

        if (playfield.tetraminoFits(currentTetramino)) {
            tetris.repaintPlayfield();
        }
        else {
            currentTetramino.moveUp();
        }
    }

    public synchronized void dropTetraminoToBottom() {

        int dropSleepTime = tetris.getStatistics().getDropSleepTime();
        currentTetramino.setSleepTime(dropSleepTime);
    }

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
// MODIFY THREAT STATUS                                                                     //
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//

	public void togglePause() {

        if (threadSuspended) {
            try { 
                synchronized(this) {
                    notify();
                }
            } 
            catch (IllegalMonitorStateException e){
            }
        }
        threadSuspended = ! threadSuspended;
    }

    public void stopGameThread() {
        threadStopped = true;
    }

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
// PRIVATE GAME METHODES                                                                    //
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//

    private synchronized boolean reachedBottom() {

        boolean tetraminoFits;

        currentTetramino.moveDown();
        tetraminoFits = playfield.tetraminoFits(currentTetramino);
        currentTetramino.moveUp();

        // System.out.println("Game.reachedBottom() - "+ (!tetraminoFits));
        
        return ! tetraminoFits;
    }


    private void playCurrentTetramino(Tetramino t) {

        tetris.repaintEntireGameScreen();

        while (! reachedBottom()) {

            // System.out.println("Game.playCurrentTetramino() - about to move down");
            currentTetramino.moveDown();
            tetris.repaintPlayfield();

            // System.out.println("Game.playCurrentTetramino() - moved down + repainted");

            sleepSomeTime( currentTetramino.getSleepTime() );

            if (threadSuspended) {
                waitWhileGamePaused();
            }
            if (threadStopped) {
                return;
            }
        }

        // System.out.println("Game.playCurrentTetramino() - about to placeTetramino()");
        playfield.placeTetramino(currentTetramino);
        int rowsFilled = playfield.removeCompleteRows();
        playfield.updatePlayfieldBuffer();

        tetris.getStatistics().update(rowsFilled);

        if (rowsFilled > 0) {
            tetris.getSound().playOnce(Sound.ROW_REMOVED);
        }
    }

    private void sleepSomeTime(int time) {
    
        try {
            sleep( time );
        }
        catch (InterruptedException e) {
        }
    }

    private synchronized void waitWhileGamePaused() {

        try { 
            synchronized(this) {
                while (threadSuspended)
                    wait();
            }
        } 
        catch (InterruptedException e){
        }
    }
    
    private int randomTetramino() {
        return (int) (Math.random() * Tetramino.NUMBER_OF_TETRAMINOS + 0.5);
    }



//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
// MAIN PROGRAM                                                                             //
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//

    public void restart() {
    }

    public void end() {

        stopGameThread();
    }

    public void run() {

        while (! playfield.topRowReached()) {
            
            currentTetramino = nextTetramino;
            nextTetramino    = new Tetramino(randomTetramino(),tetris);
            playCurrentTetramino(currentTetramino);

            if (threadStopped) {
                return;
            }
        }

        tetris.setRunMode(Tetris.GAME_END);
        tetris.getSound().playOnce(Sound.GAME_OVER);
        tetris.repaint();
    }
}
