/***************************************************************************

 Linux SVGALib adaptation by Phillip Ezolt pe28+@andrew.cmu.edu
  
  linux.c

  OS dependant stuff (display handling, keyboard scan...)
  This is the only file which should me modified in order to port the
  emulator to a different system.

***************************************************************************/
#ifdef svgalib

#define __SVGALIB_C

#include <vga.h>
#include <vgagl.h>
#include <vgakeyboard.h>
#include <signal.h>

#include "xmame.h"
#include "osdepend.h"
#include "driver.h"
#include "linux.c"
#include "twkuser.c"

#define VGAMODE G640x480x256
#define LOWRESMODE G320x200x256

static int visual_width;
static int visual_height;
static unsigned char *video_mem;

void svgalib_mouse_init(void);

Register scr224x288[] =
{
        { 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x37},
        { 0x3d4, 0x02, 0x38},{ 0x3d4, 0x03, 0x82},{ 0x3d4, 0x04, 0x4a},
        { 0x3d4, 0x05, 0x9a},{ 0x3d4, 0x06, 0x55},{ 0x3d4, 0x07, 0xf0},
        { 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x61},{ 0x3d4, 0x10, 0x40},
        { 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0x3f},{ 0x3d4, 0x13, 0x1c},
        { 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x40},{ 0x3d4, 0x16, 0x4a},
        { 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e},
        { 0x3ce, 0x05, 0x40},{ 0x3ce, 0x06, 0x05},{ 0x3c0, 0x10, 0x41},
        { 0x3c0, 0x13, 0x0}
};

Register scr224x288scanlines[] =
{
     { 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x37},
             { 0x3d4, 0x02, 0x38},{ 0x3d4, 0x03, 0x82},{ 0x3d4, 0x04, 0x4a},
             { 0x3d4, 0x05, 0x9a},{ 0x3d4, 0x06, 0x43},{ 0x3d4, 0x07, 0x1f},
             { 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x60},{ 0x3d4, 0x10, 0x2a},
             { 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0x1f},{ 0x3d4, 0x13, 0x1c},
             { 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x27},{ 0x3d4, 0x16, 0x3a},
             { 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e},
             { 0x3ce, 0x05, 0x40},{ 0x3ce, 0x06, 0x05},{ 0x3c0, 0x10, 0x41},
             { 0x3c0, 0x13, 0x00}
};

Register scr256x256[] =
{
     { 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x3f},
             { 0x3d4, 0x02, 0x40},{ 0x3d4, 0x03, 0x82},{ 0x3d4, 0x04, 0x4A},
             { 0x3d4, 0x05, 0x9A},{ 0x3d4, 0x06, 0x23},{ 0x3d4, 0x07, 0xb2},
             { 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x61},{ 0x3d4, 0x10, 0x0a},
             { 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0xff},{ 0x3d4, 0x13, 0x20},
             { 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x07},{ 0x3d4, 0x16, 0x1a},
             { 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e},
             { 0x3ce, 0x05, 0x40},{ 0x3ce, 0x06, 0x05},{ 0x3c0, 0x10, 0x41},
             { 0x3c0, 0x13, 0x00}
};

Register scr256x256scanlines[] =
{
     { 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x3f},
             { 0x3d4, 0x02, 0x40},{ 0x3d4, 0x03, 0x82},{ 0x3d4, 0x04, 0x4a},
             { 0x3d4, 0x05, 0x9a},{ 0x3d4, 0x06, 0x23},{ 0x3d4, 0x07, 0x1d},
             { 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x60},{ 0x3d4, 0x10, 0x0a},
             { 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0xff},{ 0x3d4, 0x13, 0x20},
             { 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x07},{ 0x3d4, 0x16, 0x1a},
             { 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e},
             { 0x3ce, 0x05, 0x40},{ 0x3ce, 0x06, 0x05},{ 0x3c0, 0x10, 0x41},
             { 0x3c0, 0x13, 0x00}
};

Register scr288x224[] =
{
        { 0x3c2, 0x0, 0xe3},{ 0x3d4, 0x0, 0x5f},{ 0x3d4, 0x1, 0x47},
        { 0x3d4, 0x2, 0x50},{ 0x3d4, 0x3, 0x82},{ 0x3d4, 0x4, 0x50},
        { 0x3d4, 0x5, 0x80},{ 0x3d4, 0x6, 0xb},{ 0x3d4, 0x7, 0x3e},
        { 0x3d4, 0x8, 0x0},{ 0x3d4, 0x9, 0x41},{ 0x3d4, 0x10, 0xda},
        { 0x3d4, 0x11, 0x9c},{ 0x3d4, 0x12, 0xbf},{ 0x3d4, 0x13, 0x24},
        { 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0xc7},{ 0x3d4, 0x16, 0x4},
        { 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x1, 0x1},{ 0x3c4, 0x4, 0xe},
        { 0x3ce, 0x5, 0x40},{ 0x3ce, 0x6, 0x5},{ 0x3c0, 0x10, 0x41},
        { 0x3c0, 0x13, 0x0}
};

Register scr288x224scanlines[] =
{
        { 0x3c2, 0x0, 0xe3},{ 0x3d4, 0x0, 0x5f},{ 0x3d4, 0x1, 0x47},
        { 0x3d4, 0x2, 0x47},{ 0x3d4, 0x3, 0x82},{ 0x3d4, 0x4, 0x50},
        { 0x3d4, 0x5, 0x9a},{ 0x3d4, 0x6, 0xb},{ 0x3d4, 0x7, 0x19},
        { 0x3d4, 0x8, 0x0},{ 0x3d4, 0x9, 0x40},{ 0x3d4, 0x10, 0xf5},
        { 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0xdf},{ 0x3d4, 0x13, 0x24},
        { 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0xc7},{ 0x3d4, 0x16, 0x4},
        { 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x1, 0x1},{ 0x3c4, 0x4, 0xe},
        { 0x3ce, 0x5, 0x40},{ 0x3ce, 0x6, 0x5},{ 0x3c0, 0x10, 0x41},
        { 0x3c0, 0x13, 0x0}
};

Register scr320x204[] =
{
     { 0x3c2, 0x00, 0xe3},{ 0x3d4, 0x00, 0x5f},{ 0x3d4, 0x01, 0x4f},
             { 0x3d4, 0x02, 0x50},{ 0x3d4, 0x03, 0x82},{ 0x3d4, 0x04, 0x54},
             { 0x3d4, 0x05, 0x80},{ 0x3d4, 0x06, 0xbf},{ 0x3d4, 0x07, 0x1f},
             { 0x3d4, 0x08, 0x00},{ 0x3d4, 0x09, 0x41},{ 0x3d4, 0x10, 0x9c},
             { 0x3d4, 0x11, 0x8e},{ 0x3d4, 0x12, 0x97},{ 0x3d4, 0x13, 0x28},
             { 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x96},{ 0x3d4, 0x16, 0xb9},
             { 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x01, 0x01},{ 0x3c4, 0x04, 0x0e},
             { 0x3ce, 0x05, 0x40},{ 0x3ce, 0x06, 0x05},{ 0x3c0, 0x10, 0x41},
             { 0x3c0, 0x13, 0x00}
};

Register scr240x272[] =
{
        { 0x3c2, 0x0, 0xe3}, { 0x3d4, 0x0, 0x5f}, { 0x3d4, 0x1, 0x3b},
        { 0x3d4, 0x2, 0x38}, { 0x3d4, 0x3, 0x82}, { 0x3d4, 0x4, 0x4a},
        { 0x3d4, 0x5, 0x9a}, { 0x3d4, 0x6, 0x55}, { 0x3d4, 0x7, 0xf0},
        { 0x3d4, 0x8, 0x0},  { 0x3d4, 0x9, 0x61}, { 0x3d4, 0x10, 0x40},
        { 0x3d4, 0x11, 0xac},{ 0x3d4, 0x12, 0x20},{ 0x3d4, 0x13, 0x1e},
        { 0x3d4, 0x14, 0x40},{ 0x3d4, 0x15, 0x40},{ 0x3d4, 0x16, 0x4a},
        { 0x3d4, 0x17, 0xa3},{ 0x3c4, 0x1, 0x1},  { 0x3c4, 0x4, 0xe},
        { 0x3ce, 0x5, 0x40}, { 0x3ce, 0x6, 0x5},  { 0x3c0, 0x10, 0x41},
        { 0x3c0, 0x13, 0x0}
};

Register scr200x320[] =
{
	{ 0x3c2, 0x00, 0xe3}, { 0x3d4, 0x00, 0x5f}, { 0x3d4, 0x01, 0x31},
	{ 0x3d4, 0x02, 0x38}, { 0x3d4, 0x03, 0x82}, { 0x3d4, 0x04, 0x4a},
	{ 0x3d4, 0x05, 0x9a}, { 0x3d4, 0x06, 0x4e}, { 0x3d4, 0x07, 0x1f},
	{ 0x3d4, 0x08, 0x00}, { 0x3d4, 0x09, 0x40}, { 0x3d4, 0x10, 0x40},
	{ 0x3d4, 0x11, 0x90}, { 0x3d4, 0x12, 0x3f}, { 0x3d4, 0x13, 0x19},
	{ 0x3d4, 0x14, 0x40}, { 0x3d4, 0x15, 0x80}, { 0x3d4, 0x16, 0x40},
	{ 0x3d4, 0x17, 0xa3}, { 0x3c4, 0x01, 0x01}, { 0x3c4, 0x04, 0x0e},
	{ 0x3ce, 0x05, 0x40}, { 0x3ce, 0x06, 0x05}, { 0x3c0, 0x10, 0x41},
	{ 0x3c0, 0x13, 0x00}
};

void set_video_mode(void)
{
	Register *reg = 0;
	int reglen = 0;
	
	if (visual_width == 224 && visual_height == 288)
          {
	    if (video_mode==2)
	      {
	        reg = scr224x288;
	        reglen = sizeof(scr224x288)/sizeof(Register);
	      }
	    else
	      {
	        reg = scr224x288scanlines;
	        reglen = sizeof(scr224x288scanlines)/sizeof(Register);
	      }
	  }
	else if (visual_width == 256 && visual_height == 256)
	  {
	    if (video_mode==2)
	      {
	        reg = scr256x256;
	        reglen = sizeof(scr256x256)/sizeof(Register);
	      }
	    else
	      {
	        reg = scr256x256scanlines;
	        reglen = sizeof(scr256x256scanlines)/sizeof(Register);
	      }
	  }
	else if (visual_width == 288 && visual_height == 224)
	  {
	    if (video_mode==2)
	      {
	        reg = scr288x224;
	        reglen = sizeof(scr288x224)/sizeof(Register);
	      }
	    else
	      {
	        reg = scr288x224scanlines;
	        reglen = sizeof(scr288x224scanlines)/sizeof(Register);
	      }
	  }
	else if (visual_width == 240 && visual_height == 272)
	  {
	    reg = scr240x272;
	    reglen = sizeof(scr240x272)/sizeof(Register);
	  }
	else if (visual_width == 320 && visual_height == 204)
	  {
	    reg = scr320x204;
	    reglen = sizeof(scr320x204)/sizeof(Register);
	  }
	else if (visual_width == 200 && visual_height == 320)
	  {
	    reg = scr200x320;
	    reglen = sizeof(scr200x320)/sizeof(Register);
	  }
	outRegArray(reg,reglen);
}


static struct sigaction sig2handler;
static struct sigaction oldsig2handler;
#if (__GLIBC__ > 1)
static sigset_t sig2sigset;
#endif

void Sig2HandlerFunction(int n)
{
   keyboard_clearstate();
   (*(oldsig2handler.sa_handler))(n);
   if (video_mode) set_video_mode();
}


/* Create a display screen, or window, large enough to accomodate a bitmap */
/* of the given dimensions. I don't do any test here (224x288 will just do */
/* for now) but one could e.g. open a window of the exact dimensions */
/* provided. Return a bitmap pointer or 0 in case of error. */
struct osd_bitmap *osd_create_display(int width,int height,int totalcolors,
		const unsigned char *palette,unsigned char *pens,int attributes)
{
   int a,b;
   bitmap = osd_create_bitmap(width,height);
   if (bitmap==NULL) return NULL;
   
   if( Machine->orientation & ORIENTATION_SWAP_XY) {
	int temp=width;
	width=height;
	height=temp;
   } 
   
/*   printf("requested res. %d x %d\n",width,height); */

   visual_width=width;

   if ((width == 200 && height <= 320))
      visual_height=320;
   else if ((width == 224 && height <= 288))
      visual_height=288;
   else if ((width == 256 && height <= 256))
      visual_height=256;
   else if ((width == 288 && height <= 224))
      visual_height=224;
   else if ((width == 320 && height <= 204))
      visual_height=204;
   else if ((width == 240 && height <= 272))
      visual_height=272;
   else
      video_mode=0;
   
   if (video_mode)
     {
        if (!vga_hasmode(LOWRESMODE))
        {
            osd_free_bitmap(bitmap);
            return NULL;
        }    
	vga_setmode(LOWRESMODE);
	gl_setcontextvga(LOWRESMODE);
	video_mem =vga_getgraphmem();
	video_mem+=((visual_height-height) /2) * width;
	
#if (__GLIBC__ > 1)
	memset(&sig2sigset,0,sizeof(sigset_t));
	sig2handler.sa_mask=sig2sigset;
#else	
	sig2handler.sa_mask=0;
#endif
	sig2handler.sa_handler=Sig2HandlerFunction;
	sig2handler.sa_flags=0;
	
	sigaction(SIGUSR2, &sig2handler, &oldsig2handler);
	
	set_video_mode();
     }
   else  
     {
        if (!vga_hasmode(VGAMODE))
        {
            osd_free_bitmap(bitmap);
            return NULL;
        }    
	vga_setmode(VGAMODE);
	gl_setcontextvga(VGAMODE);
     }
     
   /* Set the palette */
   if (totalcolors>256)
   {
      fprintf(stderr, "Warning: More than 256 colors (%d) are needed for this emulation\n", totalcolors);
      /* fill the remainder of the pens array with 0's to make sure */
      /* nothing strange happens                                    */
      for (a=256;a<totalcolors;a++) pens[a]=0;
      totalcolors=256;
   }
   
   if (totalcolors==256) /* better leave collor 0 alone, but we need it */
      b=0;               /* if the emulation wants all 256 colors.      */
    else
      b=1;  
   
   for (a=0; a < totalcolors; a++)
   {
      gl_setpalettecolor(a+b, palette[3*a] >> 2, palette[3*a+1] >> 2,
                         palette[3*a+2] >> 2);
      pens[a] = a+b;
   }
   
   /* init the keyboard */
   keyboard_init();
   keyboard_translatekeys (TRANSLATE_CURSORKEYS | TRANSLATE_DIAGONAL);

   /* 
      not sure if this is the best site but mouse init routine must be 
      called after video initialization...
   */   
   if(use_mouse)
   {
	vga_setmousesupport(TRUE);
	svgalib_mouse_init();
   }
   
#ifdef USE_TIMER
    /* svgalib catches the timer alarm. so we have to arm it after all
       other svgalib initialising. */
    if (play_sound)
    {
       if(start_timer()==OSD_NOT_OK)
       {
          osd_free_bitmap(bitmap);
          return NULL;
       }
    }
#endif
   
   return bitmap;
}


/* shut up the display */
void osd_close_display(void)
{
   if (video_mode) sigaction(SIGUSR2, &oldsig2handler, NULL);
   vga_setmode(TEXT);
   free(bitmap);
}

void osd_modify_pen(int pen,unsigned char red, unsigned char green, unsigned char blue)
{
   gl_setpalettecolor(pen,(red>>2),(green>>2),(blue>>2));
}

void osd_get_pen(int pen,unsigned char *red, unsigned char *green, unsigned char *blue)
{
    int r,g,b;
    gl_getpalettecolor(pen,&r,&g,&b);
    *red   = r<<2; 
    *green = r<<2; 
    *blue  = b<<2;
    return;
}

void osd_mark_dirty(int x1, int y1, int x2, int y2, int ui)
{
}

/* Update the display. */
void osd_update_display(void)
{
   if(video_mode)
   {
     memcpy(video_mem, bitmap->_private,bitmap->width*bitmap->height);
   }  
   else
     gl_putbox((640 - bitmap->width)/2, (480 - bitmap->height)/2, 
               bitmap->width, bitmap->height, bitmap->_private);
}


int sysdep_keyboard_init(void)
{
   /* don't do anything here instead init the keyboard in osd_create_display */
   /* this is done to make sure we won't exit unexpectedly (when failed to   */
   /* create the display without restoring the keyboard ) (dirty hack)       */
   return OSD_OK;   
}


void sysdep_keyboard_close(void)
{
   keyboard_close();
}


/* check if a key is pressed. The keycode is the standard PC keyboard code, as */
/* defined in osdepend.h. Return 0 if the key is not pressed, nonzero otherwise. */
int osd_key_pressed(int keycode)
{
   keyboard_update();   
   return keyboard_keypressed(keycode);
}


/*Wait for a keypress and return the key code. */
int osd_read_key(void)
{
   int keycode;
   keyboard_clearstate();
   while(1){
      keyboard_update();
      for (keycode=0;keycode<128;keycode++)
	if (keyboard_keypressed(keycode))   
	  return keycode;
   }
}


/* src/msdos/msdos.c added in mame 0.22.1 Not sure why, so.... */
int osd_read_keyrepeat() 
{
	return osd_read_key();	
}


int sysdep_mapkey(int from, int to) 
{
	return OSD_OK;
}

#endif /* ifdef svgalib */
