/*
 * An adaptation of MAME 0.2 for X11 / 0.81 for DOS.
 * Can be played. Has only been tested on Linux, but should port easily.
 */

#define __MAIN_C_
#include "xmame.h"
#include "sound.h"

#include <dirent.h>

/*  Zip subsystem requirements */
#include <string.h>
#include <fcntl.h>

#include "../zipfiles/zipfiles.h"

/* End Zip subsystem requirements */

extern int frameskip; /* defined in src/mame.c */
extern char *gamename; /* defined in src/mame.c */

#ifdef ggi
char *title="X-Mame (GGI) version 0.30.1"
#elif svgalib
char *title="X-Mame (SVGALib) version 0.30.1";
#else
char *title="X-Mame (X11) version 0.30.1";
#endif

/* 
 * show help and exit
 */
int show_usage() 
{
char message[]="%s\n\
Usage: xmame [game] [options]\n\
Options:\n\
\n\
*** Input ***\n\
    -[no]joy			Disable/enable joystick (if supported)\n\
    -[no]mouse			Disable/enable mouse (if supported)\n\
    -[no]pad			Disable/enable FM-TOWNS game pad (if supported)\n\
    -[no]swapjoyaxis		Swap X and Y joystick axis\n\
    -joyfilter  <value>  	Joystick \"noise\" filter value (defaults to 5)\n\
    -x11joyname <name>		Name of X-based joystick device (if compiled in)\n\
    -paddevname <name>		Name of pad device (defaults to /dev/pad00)\n\
    -mapkey <Xcode>,<Mamecode>  Set an specific key mapping\n\
    -[no]cheat                  Disable/enable cheat mode if game supports it\n\
\n\
*** Video ***\n\
    -heightscale<scale> 	Set Y-Scale aspect ratio\n\
    -widthscale	<scale> 	Set X-Scale aspect ratio\n\
    -scale 	<scale>		Set X-Y Scale to the same aspect ratio\n\
    -frameskip	<#frames>	Skip <#frames> in video refresh\n\
    -display	<display>	To select display\n\
    -[no]mitshm			Disable/enable MIT Shared Mem (if supported)\n\
    -[no]xsync			Select XFlush/Xsync screen refresh method\n\
    -[no]privatecmap		Disable/enable use of private color maps\n\
    -videomode  <mode>		Select svgalib video mode ( if compiled to )\n\
    -[no]forcetruecolor  	Disable/enable try to use pseudocolor X-Visuals\n\
    -ror			Rotate display 90 degrees rigth\n\
    -rol			Rotate display 90 degrees left\n\
    -flipx			Flip X axis\n\
    -flipy			Flip Y axis\n\
\n\
*** Sound ***\n\
    -usedsp                     ( Linux only ) Select /dev/dsp as sound dev \n\
    -useaudio                   ( Linux only ) Select /dev/audio as sound dev \n\
    -[no]sound			Disable/enable sound (if available)\n\
    -[no]fm			Disable/enable use of FM Synthesis\n\
    -samplefreq <samplefreq>    Set the sample frequency\n\
    -timerfreq <timerfreq>      Set the timer frequency (if timer based audio)\n\
\n\
*** General ***\n\
    -mamedir	<dir>		Tell XMame where roms base tree resides\n\
    -spooldir	<dir>		Tell XMame where to store high scores\n\
    -showconfig                 Display Running parameters in xmamerc style\n\
    -help | -h | -?		Show this help\n\
    -list			List supported games\n\
    -listfull			Game list with full description\n\
    -listroms			Show ROM names of a given game\n\
    -listsamples		Show audio samples filenames of a given game\n\
    -log			Log debug info\n\
\n\
Files: \n\
    %s	Global configuration file\n\
    ${HOME}/xmame/xmamerc	User global configuration file\n\
    ${HOME}/xmame/<game>.cnf	Game-dependent user configuration file\n\
\n\
Environment variables:\n\
\n\
    HOME			Users's home directory\n\
    MAMEDIR			Arcade Roms base directory\n\
    DISPLAY			X-Server to display in\n\
\n\
Mame is an Original Idea of Nicola Salmoria and Mirko Buffoni\n\
X-Mame port mantained by Juan Antonio Martinez\n\
";
	fprintf(stderr,message,title,MAMERC);
	return (OSD_NOT_OK); /* force exit on return */
}

/*
 * show running parameters in xmamerc style format ( using stdout )
 * if exitflag is true exit program
 *
 */
int show_config(int exitflag,FILE *f)
{
	fprintf(f,"##### %s Running parameters: #####\n",title);
	fprintf(f,"#\n");
	fprintf(f,"# Global paramaters\n");
	fprintf(f,"#\n");
	fprintf(f,"mamedir             %s\n",mamedir);
	fprintf(f,"spooldir            %s\n",spooldir);
	fprintf(f,"defaultgame         %s\n",gamename);
	fprintf(f,"#\n");
	fprintf(f,"# Video paramaters\n");
	fprintf(f,"#\n");
	fprintf(f,"display             %s\n",displayname);
	fprintf(f,"videomode           %d\n",video_mode);
	fprintf(f,"heightscale         %d\n",heightscale);
	fprintf(f,"widthscale          %d\n",widthscale);
	fprintf(f,"frameskip           %d\n",frameskip);
	fprintf(f,"use_xsync           %d\n",(use_xsync)?1:0);
	fprintf(f,"use_mitshm          %d\n",(mit_shm_avail)?1:0);
	fprintf(f,"private_colormap    %d\n",(use_private_cmap)?1:0);
	fprintf(f,"force_truecolor     %d\n",(force_truecolor)?1:0);
	fprintf(f,"#\n");
	fprintf(f,"# Audio paramaters\n");
	fprintf(f,"#\n");
	fprintf(f,"play_sound          %d\n",(play_sound)?1:0);
	fprintf(f,"audiodevice         %s\n",audiodevice);
	fprintf(f,"dontuse_fm_synth    %d\n",(No_FM)?1:0);
	fprintf(f,"samplefreq          %d\n",audio_sample_freq);
	fprintf(f,"timerfreq           %d\n",audio_timer_freq);
	fprintf(f,"#\n");
	fprintf(f,"# Mouse, Joystick and Trakball paramaters\n");
	fprintf(f,"#\n");
	fprintf(f,"use_mouse           %d\n",(use_mouse)?1:0);
	fprintf(f,"use_joystick        %d\n",(use_joystick)?1:0);
	fprintf(f,"x11joyname          %s\n",x11joyname);
	fprintf(f,"joyfilter           %d\n",joyfilter);
	fprintf(f,"swapjoyaxis         %d\n",(swapjoyaxis)?1:0);
	fprintf(f,"towns_pad_name      %s\n",towns_pad_dev);
	
	if (exitflag) exit(0);
	return OSD_OK;
}

/*
 * check for existence of ${HOME}/xmame/xmamerc
 * if found parse it, else try global MAMERC #defined file
 */
int parse_xmamerc_file()
{
	struct passwd   *pw;
	FILE            *file;
	int             lineno;
	char            buffer[2048];

	/* locate user's home directory */
	pw=getpwuid(getuid());
	if(!pw) { 
		fprintf(stderr,"Who are you? Not found in passwd database!!\n");
		return (OSD_NOT_OK);
	}
	sprintf(buffer,"%s/xmame/xmamerc",pw->pw_dir);
	/* try to open file. If so parse it */
	if ( (file=fopen(buffer,"r")) == (FILE *) NULL) {
		fprintf(stderr,"File %s not found.\n",buffer);
		/* No user config found - trying global file */
		strcpy(buffer, MAMERC );
		if ( (file=fopen(buffer,"r")) == (FILE *) NULL) {
			fprintf(stderr,"File %s not found. Skipping ...\n",buffer);
			return (OSD_OK);
		}
	}
	lineno=0;
	while(fgets( buffer,2047,file) ) {
	    char *p;
	    char *(q[5]);
	    int i,j;
	    lineno++;
	    /* skip comments */
	    if ( ( p=strchr(buffer,'#') ) ) *p='\0';
	    /* scan for words */
	    for(i=0;i<5;i++) q[i]=(char *)NULL;
	    for(i=0,j=0,p=buffer; *p; p++ ) {
		if ( isspace(*p) ) { *p='\0'; j=0; }
		else               { if(!j++) q[i++]=p; }
	    }   /* end of stripping words ( for i=0 )*/
	    /* test for wrong number of args */
	    if ( i==0 ) continue; /* empty line */
	    if ( i!=2 ) { 
		    fprintf(stderr,"Line %d: wrong number of parameters \n",lineno);
		    fclose(file);
		    return (OSD_NOT_OK);
	    }
	    /* now parse commands */
	    if (strcasecmp(q[0], "play_sound") == 0) {
		play_sound        = atoi( q[1] );
		continue;
	    } /* sound */
	    if (strcasecmp(q[0], "dontuse_fm_synth") == 0) {
		No_FM        = atoi( q[1] );
		continue;
	    } /* sound */
	    if (strcasecmp(q[0], "use_mouse") == 0) {
#ifdef USE_MOUSE
		use_mouse = atoi( q[1] ) ;
#endif
		continue; 
	    } /* mouse */
	    if (strcasecmp(q[0], "use_joystick") == 0) {
#ifdef USE_JOYSTICK
		use_joystick = atoi( q[1] ) ;
#endif
		continue; 
	    } /* joystick */
	    if (strcasecmp(q[0], "swapjoyaxis") == 0) {
#ifdef USE_JOYSTICK
		swapjoyaxis = atoi( q[1] ) ;
#endif
		continue; 
	    } /* swapjoyaxis */
	    if (strcasecmp(q[0], "frameskip") == 0) {
		frameskip = atoi( q[1] ) ;
		if (frameskip>3) frameskip=3;
		continue; 
	    } /* frameskip */
	    if (strcasecmp(q[0], "joyfilter") == 0) {
		joyfilter = atoi ( q[1] );
#ifdef USE_JOYSTICK
		if (joyfilter <= 0) {
			fprintf (stderr,"Line %d: Illegal joyfilter value (%d)\n", lineno,joyfilter);
			return (OSD_NOT_OK);
		}
#endif
		continue;
	    } /* joyfilter */
	    if (strcasecmp(q[0], "widthscale") == 0) {
		widthscale = atoi ( q[1] );
		if (widthscale <= 0) {
			fprintf (stderr,"Line %d: Illegal widthscale (%d)\n", lineno,widthscale);
			return (OSD_NOT_OK);
		}
		continue;
	    } /* widthscale */
	    if (strcasecmp(q[0], "heightscale") == 0) {
		heightscale = atoi ( q[1] );
		if (heightscale <= 0) {
			fprintf (stderr,"Line %d: Illegal heightscale (%d)\n", lineno,heightscale);
			return (OSD_NOT_OK);
		}
		continue;
	    } /* heightscale */
	    if (strcasecmp(q[0], "force_truecolor") == 0) {
		force_truecolor = atoi(q[1]);
		continue;
	    } /* Force use of XSync() to refresh screen */
	    if (strcasecmp(q[0], "use_xsync") == 0) {
		use_xsync = atoi(q[1]);
		continue;
	    } /* use_xsync */
	    if (strcasecmp(q[0], "private_colormap") == 0) {
		use_private_cmap = atoi(q[1]);
		continue;
	    } /* Force use of private color maps */
	    if (strcasecmp(q[0], "use_mitshm") == 0) {
#ifdef USE_MITSHM
		mit_shm_avail = ( atoi(q[1]) ) ? 0 : -1 ;
#endif
		continue;
	    } /* mit shared memory */
            if (strcasecmp(q[0], "towns_pad_name") == 0) {
                towns_pad_dev=(char *)malloc(1+strlen(q[1]) );
                if( ! towns_pad_dev ) {
                        fprintf(stderr,"Malloc error: line %d\n",lineno);
                        return (OSD_NOT_OK);
                }
                strcpy(towns_pad_dev,q[1]);
#ifdef LIN_FM_TOWNS
                use_joystick = TRUE ;
#endif
               continue;
            } /* FM TOWNS game pad */
	    if (strcasecmp(q[0], "defaultgame") == 0) {
		char *pt;
		pt=(char *)malloc(1+strlen(q[1]) );
		if( ! pt ) { 
			fprintf(stderr,"Malloc error: line %d\n",lineno);
			return (OSD_NOT_OK);
		}
		strcpy(pt,q[1]);
		gamename=pt;
		continue; 
	    } /* defaultgame */
	    if (strcasecmp(q[0], "x11joyname") == 0) {
		x11joyname=(char *)malloc(1+strlen(q[1]) );
		if( ! x11joyname ) { 
			fprintf(stderr,"Malloc error: line %d\n",lineno);
			return (OSD_NOT_OK);
		}
		strcpy(x11joyname,q[1]);
		continue;
	    } /* x11joyname */
	    if (strcasecmp(q[0], "mamedir") == 0) {
		mamedir=(char *)malloc(1+strlen(q[1]) );
		if( ! mamedir ) { 
			fprintf(stderr,"Malloc error: line %d\n",lineno);
			return (OSD_NOT_OK);
		}
		strcpy(mamedir,q[1]);
		continue;
	    } /* mamedir */
	    if (strcasecmp(q[0], "audiodevice") == 0) {
		audiodevice=(char *)malloc(1+strlen(q[1]) );
		if( ! audiodevice ) { 
			fprintf(stderr,"Malloc error: line %d\n",lineno);
			return (OSD_NOT_OK);
		}
		strcpy(audiodevice,q[1]);
		continue;
	    } /* audiodevice */
	    if (strcasecmp(q[0], "spooldir") == 0) {
		spooldir=(char *)malloc(1+strlen(q[1]) );
		if( ! spooldir ) { 
			fprintf(stderr,"Malloc error: line %d\n",lineno);
			return (OSD_NOT_OK);
		}
		strcpy(spooldir,q[1]);
		continue;
	    } /* spooldir */
	    if (strcasecmp(q[0], "display") == 0) {
		displayname=(char *)malloc(1+strlen(q[1]) );
		if( ! displayname ) { 
			fprintf(stderr,"Malloc error: line %d\n",lineno);
			return (OSD_NOT_OK);
		}
		strcpy(displayname,q[1]);
		continue;
	    } /* displayname */
	    if (strcasecmp(q[0],"mapkey") == 0) {
		int from,to;
#ifdef ultrix
		/* ultrix sscanf() requires explicit leading of 0x for hex numbers */
		if ( sscanf(q[1],"0x%x,0x%x",&from,&to) != 2 )
#else
		if ( sscanf(q[1],"%x,%x",&from,&to) != 2 )
#endif
		    fprintf(stderr,"Line %d: Invalid keymapping %s. Ignoring...\n",lineno,q[1]);
		else    if (sysdep_mapkey(from,to) ) {
			    fprintf(stderr,"Line %d: invalid keymapping\n",lineno);
			    return (OSD_NOT_OK);
			}
		continue;
	    } /* mapkey */
	    if (strcasecmp(q[0],"samplefreq") == 0) {
		audio_sample_freq = atoi(q[1]);
		if (audio_sample_freq < 11025 || audio_sample_freq > 45454) {
			fprintf (stderr,"Illegal audio sample frequency (%d)\n", audio_sample_freq);
			return (OSD_NOT_OK);
		}
		continue;
	    } /* sample freq */
	    if (strcasecmp(q[0],"timerfreq") == 0) {
		audio_timer_freq = atoi(q[1]);
		if (audio_timer_freq < 1 || audio_timer_freq > 100) {
			fprintf (stderr,"Illegal audio timer frequency (%d)\n", audio_timer_freq);
			return (OSD_NOT_OK);
		}
		continue;
	    } /* timer freq */
	    if (strcasecmp(q[0],"videomode") == 0) {
		video_mode = atoi(q[1]);
		if (video_mode < 0 || video_mode > 2) {
			fprintf (stderr,"illegal videomode (%d)\n", video_mode);
			return (OSD_NOT_OK);
		}
		continue;
	    } /* video mode */
	    /* if arrives here means unrecognized command line */
	    fprintf(stderr,"Line %d: unknown command %s\n",lineno,q[0]);
	    fclose(file);
	    return (OSD_NOT_OK);
	} /* while */
	fclose(file);
	return (OSD_OK);
}

/*
 * Put anything here you need to do when the program is started.
 * Return 0 if initialization was successful, nonzero otherwise.
 */

int osd_init_environment (void)
{
	char *pt;
	/* initialize some variables to defalut values */
	use_joystick            = FALSE;
	swapjoyaxis             = FALSE;
	use_mouse               = FALSE;
	use_private_cmap	= FALSE;
	video_sync		= FALSE;
	play_sound              = FALSE;
	No_FM			= TRUE; 
	joyfilter		= 5;
	widthscale              = 1;
	heightscale             = 1;
	mit_shm_avail           = 0;
	snapshot_no             = 0;
	video_mode		= 0;
	force_truecolor		= 0;
	use_xsync		= 0;
	audio_sample_freq       = AUDIO_SAMPLE_FREQ;
	audio_timer_freq        = AUDIO_TIMER_FREQ;
	mamedir                 = MAMEDIR;
	spooldir                = SPOOLDIR;
	displayname             = DISPLAY;
	audiodevice		= "/dev/dsp";
	towns_pad_dev		= "/dev/pad00";
	showconfig              = FALSE;
#ifdef X11_JOYSTICK
	x11joyname              = X11_JOYNAME;
#else
	x11joyname		= "";
#endif
	/* too: init Zip interface */	
	initZip();
	
	/* parse xmamerc file */
	if( parse_xmamerc_file() ) {
		fprintf(stderr,"Error in parsing xmamerc file\n");
		return (OSD_NOT_OK);
	}

	/* main programs need to chdir to mamedir before doing anything. */

	pt=getenv("MAMEDIR");
	if (pt) mamedir=pt;
	chdir(mamedir);	/* may fail; dont worry about it */

	return (OSD_OK);
}

int osd_init (int argc, char *argv[])
{
	int i;
	char *pt;
	
	/* get environment variables. This overrides xmamerc options */
	if ( (pt=getenv("DISPLAY")) ) displayname=pt;
	if ( (pt=getenv("MAMEDIR")) ) mamedir=pt;

	/* parse argument invocation. */
	/* These has Higher prioryty than conf file and environment */

	for (i = 1;i < argc;i++) {
		if (strcasecmp(argv[i], "-help") == 0) return show_usage();
		if (strcasecmp(argv[i], "-h") == 0) return show_usage();
		if (strcasecmp(argv[i], "-?") == 0) return show_usage();
/*
		if (strcasecmp(argv[i], "-list") == 0) return show_game_list();
		if (strcasecmp(argv[i], "-info") == 0) {
			if ( ! *argv[i+1] ) {
				fprintf(stderr,"Invalid use of -info option\n");
				return (OSD_NOT_OK);
			}
			return show_game_info(argv[++i]);;
		}
*/
		if (strcasecmp(argv[i], "-fm") == 0) {
			No_FM = FALSE;
			continue;
		}
		if (strcasecmp(argv[i], "-nofm") == 0) {
			No_FM = TRUE;
			continue;
		}
		if (strcasecmp(argv[i], "-pad") == 0) {
#ifdef LIN_FM_TOWNS
			use_joystick = TRUE; /* a joystick like device */
#else
			fprintf(stderr,"FM-TOWNS game pad support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-nopad") == 0) {
#ifdef LIN_FM_TOWNS
			use_joystick = FALSE; /* gamepad is joystick-like */
#else
			fprintf(stderr,"FM-TOWNS game pad support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-mouse") == 0) {
#ifdef USE_MOUSE
			use_mouse = TRUE;
#else
			fprintf(stderr,"Mouse support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-nomouse") == 0) {
#ifdef USE_MOUSE
			use_mouse = FALSE;
#else
			fprintf(stderr,"Mouse support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-joy") == 0) {
#ifdef USE_JOYSTICK
			use_joystick = TRUE;
#else
			fprintf(stderr,"Joystick support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-nojoy") == 0) {
#ifdef USE_JOYSTICK
			use_joystick = FALSE;
#else
			fprintf(stderr,"Joystick support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-swapjoyaxis") == 0) {
#ifdef USE_JOYSTICK
			swapjoyaxis = TRUE;
#else
			fprintf(stderr,"Joystick support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-noswapjoyaxis") == 0) {
#ifdef USE_JOYSTICK
			swapjoyaxis = FALSE;
#else
			fprintf(stderr,"Joystick support not compiled. Skipping\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-sound") == 0) {
			play_sound        = TRUE;
			audio_timer_freq  = AUDIO_TIMER_FREQ;
			continue;
		}
		if (strcasecmp(argv[i], "-usedsp") == 0) {
			audiodevice        = "/dev/dsp";
#ifndef linux
			fprintf(stderr,"Warning: \"-usedsp\" option ignored\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-useaudio") == 0) {
			audiodevice        = "/dev/audio";
#ifndef linux
			fprintf(stderr,"Warning: \"-usedsp\" option ignored\n");
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-nosound") == 0) {
			play_sound        = FALSE;
			continue;
		}
		if (strcasecmp(argv[i], "-forcetruecolor") == 0) {
			force_truecolor = TRUE;
			continue;
		}
		if (strcasecmp(argv[i], "-noforcetruecolor") == 0) {
			force_truecolor = FALSE;
			continue;
		}
		if (strcasecmp(argv[i], "-xsync") == 0) {
			use_xsync = TRUE;
			continue;
		}
		if (strcasecmp(argv[i], "-noxsync") == 0) {
			use_xsync = FALSE;
			continue;
		}
		if (strcasecmp(argv[i], "-privatecmap") == 0) {
			use_private_cmap = TRUE;
			continue;
		}
		if (strcasecmp(argv[i], "-noprivatecmap") == 0) {
			use_private_cmap = FALSE;
			continue;
		}
		if (strcasecmp(argv[i], "-joyfilter") == 0) {
			joyfilter = atoi (argv[++i]);
#ifdef USE_JOYSTICK
			if (joyfilter <= 0) {
				fprintf (stderr,"illegal joyfilter value (%d)\n", joyfilter);
				return (OSD_NOT_OK);
			}
#endif
			continue;
		}
		if (strcasecmp(argv[i], "-widthscale") == 0) {
			widthscale = atoi (argv[++i]);
			if (widthscale <= 0) {
				fprintf (stderr,"illegal widthscale (%d)\n", widthscale);
				return (OSD_NOT_OK);
			}
			continue;
		}
		if (strcasecmp(argv[i], "-heightscale") == 0) {
			heightscale = atoi (argv[++i]);
			if (heightscale <= 0) {
				fprintf (stderr,"illegal heightscale (%d)\n", heightscale);
				return (OSD_NOT_OK);
			}
			continue;
		}
		if (strcasecmp(argv[i], "-frameskip") == 0) {
			frameskip = atoi (argv[++i]);
			if (frameskip <= 0) {
				fprintf (stderr,"illegal frameskip value (%d)\n", frameskip);
				return (OSD_NOT_OK);
			}
			if (frameskip >3) frameskip=3;
			continue;
		}
		if (strcasecmp(argv[i], "-scale") == 0) {
			heightscale = widthscale = atoi (argv[++i]);
			if (heightscale <= 0) {
				fprintf (stderr,"illegal scale factor (%d)\n", heightscale);
				return (OSD_NOT_OK);
			}
			continue;
		}
		if (strcasecmp(argv[i], "-mitshm") == 0) {
			fprintf(stderr,"MIT Shared Memory use enabled\n");
			mit_shm_avail=1;
			continue;
		}
		if (strcasecmp(argv[i], "-nomitshm") == 0) {
			fprintf(stderr,"MIT Shared Memory use disabled\n");
			mit_shm_avail=-1;
			continue;
		}
		if (strcasecmp(argv[i], "-showconfig") == 0) {
			showconfig=TRUE;
			continue;
		}
		if (strcasecmp(argv[i], "-mamedir") == 0) {
			if ( ! *argv[i+1] ) {
				fprintf(stderr,"Invalid use of -mamedir option\n");
				return (OSD_NOT_OK);
			}
			mamedir=argv[++i];
			continue;
		}
		if (strcasecmp(argv[i], "-spooldir") == 0) {
			if ( ! *argv[i+1] ) {
				fprintf(stderr,"Invalid use of -spooldir option\n");
				return (OSD_NOT_OK);
			}
			spooldir=argv[++i];
			continue;
		}
		if (strcasecmp(argv[i], "-paddevname") == 0) {
			if ( ! *argv[i+1] ) {
				fprintf(stderr,"Invalid use of -paddevname option\n");
				return (OSD_NOT_OK);
			}
			towns_pad_dev=argv[++i];
			continue;
		}
		if (strcasecmp(argv[i], "-x11joyname") == 0) {
			if ( ! *argv[i+1] ) {
				fprintf(stderr,"Invalid use of -x11joyname option\n");
				return (OSD_NOT_OK);
			}
			x11joyname=argv[++i];
			continue;
		}
		if (strcasecmp(argv[i], "-display") == 0) {
			if ( ! *argv[i+1] ) {
				fprintf(stderr,"Invalid use of -display option\n");
				return (OSD_NOT_OK);
			}
			displayname=argv[++i];
			continue;
		}
		if (strcasecmp(argv[i],"-mapkey") == 0) {
			if ( ! *argv[i+1] ) {
				fprintf(stderr,"Invalid use of -mapkey option\n");
				return (OSD_NOT_OK);
			} else {
				int from,to;
				if ( sscanf(argv[++i],"%x,%x",&from,&to) != 2 )
					fprintf(stderr,"Invalid keymapping %s. Ignoring...\n",argv[i]);
				else    sysdep_mapkey(from,to);
			}
			continue;
		}
		if (strcasecmp(argv[i], "-samplefreq") == 0) {
			audio_sample_freq = atoi (argv[++i]);
			if (audio_sample_freq < 11025 || audio_sample_freq > 45454) {
				fprintf (stderr,"illegal audio sample frequention (%d)\n", audio_sample_freq);
				return (OSD_NOT_OK);
			}
			continue;
		}
		if (strcasecmp(argv[i], "-videomode") == 0) {
			video_mode = atoi (argv[++i]);
			if (video_mode < 0 || video_mode > 2) {
				fprintf (stderr,"illegal videomode (%d)\n", video_mode);
				return (OSD_NOT_OK);
			}
			continue;
		}
		/* these options are tested in init_machine(), so just parse and ignore */
		/* I know, a very bad prog style, but no time to get it right */
		if (strcasecmp(argv[i],"-ror") || 
		    strcasecmp(argv[i],"-rol") ||
	            strcasecmp(argv[i],"-flipx") || 
		    strcasecmp(argv[i],"-flipy") ||
		    strcasecmp(argv[i],"-cheat") ) continue ;
		/* if arrives here and not at first argument, mean syntax error */
		if (i>1) {
			fprintf(stderr,"Unknown option. Try %s -help\n",argv[0]);
			return (OSD_NOT_OK);
		}

	}
	
	/* try to set current working directory to mamedir */
	if ( chdir(mamedir)<0 ) {
		fprintf(stderr,"Cannot chdir() to %s. Sorry\n",mamedir);
		return (OSD_NOT_OK);
	}

	/* call several global initialization routines */
	sysdep_joy_initvars();
	sysdep_keyboard_init();
	sysdep_mouse_init();
	/* now invoice system-dependent initialization */
	sysdep_init();
	sysdep_audio_initvars();
#ifdef USE_TIMER
#ifndef svgalib
	/* in svgalib mode must start timer AFTER open display */
	if (play_sound)
	   if (start_timer()==OSD_NOT_OK) return(OSD_NOT_OK);
#endif
#endif
	if (showconfig==TRUE) show_config(1,stdout); 
#if 0
	else show_config(0,stdout);
#endif
	return (OSD_OK);
}


/*
 * Cleanup routines to be executed when the program is terminated.
 */

void osd_exit (void)
{
	/* to allow profiling info to be saved change to home's directory */
	chdir(getenv("HOME"));
	/* actually no global options: invoice directly unix-dep routines */
	sysdep_keyboard_close();
	sysdep_mouse_close();
	sysdep_joy_close();
	sysdep_exit();
}

/* dummy functions we will probably never be able to blink the keybleds 
   under unix anyway */
void osd_led_w(int led,int on) 
{
	return;
}

#if replaced_by_too && 0

/* this is the main replacement in `main.c'
 * other changes include:
 * 1) in the beginning of this file, there is few #include lines added
 * 2) in osd_init_environment(), a call to `initZip();' is added.
 */

/* file handling routines */

/* gamename holds the driver name, filename is only used for ROMs and samples. */
/* if 'write' is not 0, the file is opened for write. Otherwise it is opened */
/* for read. */
void *osd_fopen(const char *gamename,const char *filename,int filetype,int write)
{
	char name[100];
	void *f;


	switch (filetype)
	{
		case OSD_FILETYPE_ROM:
		case OSD_FILETYPE_SAMPLE:
		    sprintf(name,"%s/%s",gamename,filename);
		    f = fopen(name,write ? "wb" : "rb");
		    if (f == 0) {
			DIR *dirp;
                        struct dirent *dp;
		        /* on real OS's try uppercase/lowercase conflicts */
                        if ( (dirp=opendir(gamename)) == (DIR *)0) {
                           fprintf(stderr,"Cannot open dir  %s\n",gamename);
                        } else {
                            /*search entry and casecompare to desired filename*/
                            for (dp=readdir(dirp); dp ; dp=readdir(dirp))
                                if (! strcasecmp(dp->d_name,filename)) break;
                            if ( dp ) sprintf(name,"%s/%s",gamename,dp->d_name);
                            else      sprintf(name,"%s/%s",gamename,filename);
                            closedir(dirp);
		            f = fopen(name,write ? "wb" : "rb");
			}
		    }
		    if (f == 0) {
		        /*try with a .zip directory (if ZipMagic installed)*/
		        sprintf(name,"%s.zip/%s",gamename,filename);
		        f = fopen(name,write ? "wb" : "rb");
		    }
		    if (f == 0) {
		        /*try with a .zif directory (if ZipFolders installed) */
		        sprintf(name,"%s.zif/%s",gamename,filename);
		        f = fopen(name,write ? "wb" : "rb");
		    }
		    return f;
		    break;
		case OSD_FILETYPE_HIGHSCORE:
			sprintf(name,"%s/%s.hi",spooldir,gamename);
			return fopen(name,write ? "wb" : "rb");
			break;
		case OSD_FILETYPE_CONFIG:
			sprintf(name,"%s/xmame/%s.cfg",getenv("HOME"),gamename);
			return fopen(name,write ? "wb" : "rb");
			break;
		default:
			return 0;
			break;
	}
}

int osd_fread(void *file,void *buffer,int length)
{
	return fread(buffer,1,length,(FILE *)file);
}

int osd_fwrite(void *file,const void *buffer,int length)
{
	return fwrite(buffer,1,length,(FILE *)file);
}

int osd_fseek(void *file,int offset,int whence)
{
	return fseek((FILE *)file,offset,whence);
}

void osd_fclose(void *file)
{
	fclose((FILE *)file);
}

#else /* not replaced by too || 1 */

/* indented using GNU indent (v 1.9.1),
 * with options '-kr -sc -i4 -bl -bli0 -nce -npcs -ncs -bs'
 * to make it easier to work with  // Oct 29 1997 too
 * (hmm, my version gives wrong line numbers when syntax errors
 *  and -ncs seems not to work)
 */

#ifndef O_BINARY
#define O_BINARY 0
#endif

#define O_RB (O_RDONLY|O_BINARY)
#define O_WB (O_WRONLY|O_CREAT|O_TRUNC|O_BINARY)

/********************

 Search file caseinsensitively

 Arguments: 
	char * path - complete pathname to the desired file. The string will
	              be modified during search (and contains the final output).
		      
	int mode - 0 -> no messages, true only if file found
		   1 -> messages, true if file scan done
\* perhaps mode message levels ... and true/false modes (or something else) *\
********************/
int filesearch(char *path, int mode)
{
    DIR *dirp;
    struct dirent *de = NULL;
    char *ep, *dp, *fp;
#if notneeded && 0
    static char secache[100] = "";	/* last filename found */
#endif

#if notneeded && 0
    /* too: check if file is found in single element cache. useful when
     * searching the same .zip file several times in a row */
    if (strcasecmp(path, secache) == 0) {
	strcpy(path, secache);
	return TRUE;
    }
#endif

    ep = strrchr(path, '/');
    if (ep) {
	*ep = '\0';	/* I guess root directory is not supported */
	dp = path;
	fp = ep + 1;
    } else {
	dp = ".";/* well, what should be the correct name for "current dir" */
	fp = path;
    }

    if (*fp == '\0') {
	if (mode) fprintf(stderr, "No filename part on path!\n");
	goto end;
    }

    /* jamc: code to perform a upper/lowercase rom search */
    /* try to open directory */
    if ((dirp = opendir(dp)) == (DIR *) 0) { 
	if (mode) fprintf(stderr, "Unable to open directory %s\n", dp);
	goto end;
    }

    /* search entry and upcasecompare to desired file name */
    for (de = readdir(dirp); de; de = readdir(dirp))
	if (!strcasecmp(de->d_name, fp)) break;
    if (de) strcpy(fp, de->d_name);
    closedir(dirp);

    /* which do you think is better, breaking with `goto' statement or using
     * do { ... } while(0); construct and break with `continue;' */
end:
    if (ep) *ep = '/';

#if notneeded && 0
    if (de) strcpy(secache, path);
#endif

    if (de || mode) return TRUE;
    return FALSE;
}

typedef struct ZipFp ZipFp;

/* definition (first struct item) matches to the definition of `struct ZipFp'. */
typedef struct FileFp {
    struct ZipArch *iszip;	/* NULL to distinquish from zip pointer */
    int fd;
} FileFp;


static FileFp *fileopen(char *name, int flags /*, int mode */ )
{
    struct FileFp *filefp = (FileFp *)malloc(sizeof *filefp);
    int fd;

    if (!filefp) fprintf(stderr, "Out Of Memory\n");
    else {
	fd = open(name, flags, 0664);
	if (fd >= 0) {
	    filefp->iszip = NULL;
	    filefp->fd = fd;
	    return filefp;
	}
	free(filefp);
    }
    return NULL;
}

/* file handling routines */

/* gamename holds the driver name,filename is only used for ROMs and samples. */
/* if 'write' is not 0, the file is opened for write. Otherwise it is opened */
/* for read. */
void *osd_fopen(const char *gamename, const char *filename,
		int filetype, int write)
{
#if 1	
    /* initZip(); call moved to osd_init(); */
    static int last_filetype = OSD_FILETYPE_ROM; /* optimize */
#else
    /* saving this code until this system get stable place in this program */
    static int last_filetype = OSD_FILETYPE_ROM + OSD_FILETYPE_SAMPLE;
    static int zipinitdone = FALSE;
#endif
    char name[100];
    void *file;

    if (last_filetype != filetype) {
#if 0      
	/* move to initializer routines later (after which initial
	 * value of `last_filetype' makes no difference) */
	if (!zipinitdone) {
	    initZip();
	    zipinitdone = TRUE;
	} else
#endif	  
	    expungeZip();	/* would work without `else' */
	last_filetype = filetype;
    }

    switch (filetype) {
    case OSD_FILETYPE_ROM:
    case OSD_FILETYPE_SAMPLE:
	/* try first zipfile */
	if (filetype == OSD_FILETYPE_ROM) sprintf(name,"roms/%s.zip",gamename);
	else sprintf(name, "samples/%s.zip", gamename);

	/* demonstrating case where not to use goto ;) */
	do {
	    /* need to keep state to avoid unnecessary searches */
	    static char cache[100] = "";
	    static int zipfilematch = FALSE;
	    struct ZipArch *ziparch;
	    if (strcmp(name, cache) == 0) {
		if (!zipfilematch) continue;	
		/* BREAK HERE (this construct would also break `switch') */
	    } else {
		strcpy(cache, name);
		(void *)filesearch(name, 0);
	    }
	    if ((ziparch = openZip(name)) != NULL) {
		zipfilematch = TRUE;
		if ((file = (void *)zipopen(filename,ziparch)) > NULL) return file;
		else zipfilematch = FALSE;
	    }
	}
	while (0);

	/* try to open ordinary rom file */
	sprintf(name, "%s/%s", gamename, filename);
	file = fileopen(name, O_RB);	/* no write for ROM & SAMPLES */
	if (!file) {
	    /* no write for ROM & SAMPLES */
	    if (filesearch(name, 0)) file = fileopen(name, O_RB);	
	}
	return file;

    case OSD_FILETYPE_HIGHSCORE:
	sprintf(name, "%s/%s.hi", spooldir, gamename);
	return fileopen(name, write ? O_WB : O_RB /*, 0644 */ );

    case OSD_FILETYPE_CONFIG:
	sprintf(name, "%s/xmame/%s.cfg", getenv("HOME"), gamename);
	return fileopen(name, write ? O_WB : O_RB /*, 0644 */ );

    default:
	break;
    }
    return NULL;
}

int osd_fread(void *file, void *buffer, int length)
{
    FileFp *f = (FileFp *) file;
    if (f->iszip == NULL) return read(f->fd, buffer, length);
    return zipread((ZipFp *) file, (char *)buffer, length);
}

int osd_fwrite(void *file, const void *buffer, int length)
{
    FileFp *f = (FileFp *) file;
    if (f->iszip == NULL) return write(f->fd, buffer, length);
    return -1;
}

int osd_fseek(void *file, int offset, int whence)
{
    FileFp *f = (FileFp *) file;
    if (f->iszip == NULL) return lseek(f->fd, offset, whence);
    return zipseek((ZipFp *) file, offset, whence);
}

void osd_fclose(void *file)
{
    FileFp *f = (FileFp *) file;
    if (f->iszip == NULL) {
	close(f->fd);
	free(file);
    } else zipclose((ZipFp *) file);
}

#endif /* replaced_by_too && 0 */

#ifdef HAVE_GETTIMEOFDAY
/* Standard UNIX clock() is based on CPU time, not real time.
   Here is a real-time drop in replacement for UNIX systems that have the
   gettimeofday() routine.  This results in much more accurate timing for
   throttled emulation.
*/
clock_t clock()
{
  static long init_sec = 0;

  struct timeval tv;
  gettimeofday(&tv, 0);
  if (init_sec == 0) init_sec = tv.tv_sec;
  return (tv.tv_sec - init_sec) * 1000000 + tv.tv_usec;
}
#endif

int osd_get_config_samplerate (int def_samplerate)
{
	return audio_sample_freq;
}

int osd_get_config_samplebits (int def_samplebits)
{
	return 8;
}

int osd_get_config_frameskip (int def_frameskip)
{
	return frameskip;
}

void osd_set_config (int def_samplerate, int def_samplebits)
{
}

void osd_save_config (int frameskip, int samplerate, int samplebits)
{
}

#define SENSITIVITY 2

int osd_analogjoy_read(int axis)
{
	static int x=0;
	static int y=0;
	
	if (osd_key_pressed(OSD_KEY_RIGHT) && x <  127) x+=SENSITIVITY;
	if (osd_key_pressed(OSD_KEY_LEFT)  && x > -128) x-=SENSITIVITY;
	if (osd_key_pressed(OSD_KEY_DOWN)  && y <  127) y+=SENSITIVITY;
	if (osd_key_pressed(OSD_KEY_UP)    && y > -128) y-=SENSITIVITY;
	 
	
	switch(axis)
	{
		case X_AXIS:
			return x;
		case Y_AXIS:
			return y;
	}
}
