/* GTK+ frame for Hyperplay/X11.
   Copyright (C) 1997, 2000 Hypercore Software Design, Ltd.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.  */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#undef const

#include <hyperplay/gtk.h>
#include <gtk/gtk.h>
#include <algorithm>
#include <stdexcept>

#ifdef HAVE_NANA_H
# include <nana.h>
# include <cstdio>
#else
# include <cassert>
# define I assert
#endif

using namespace hyperplay::gtk;
using namespace std;

const unsigned int TIMEOUT_INTERVAL = 20;

void
gtk_frame::update_area(int x, int y, int w, int h) const
{
  for (vector<GtkWidget *>::const_iterator i = widgets.begin();
       i != widgets.end();
       ++i)
    {
      I(*i != NULL);
      gtk_widget_queue_draw_area(*i, x, y, w, h);
    }
}

void
gtk_frame::draw_rgb(int x, int y,
		    unsigned char *s_buf, size_t s_row_size,
		    int w, int h)
{
  guchar *p = rgb_buf;

  if (y < 0)
    {
      s_buf -= y * s_row_size;
      h += y;
      y = 0;
    }
  if (y < height && h > 0)
    {
      p += y * width * 3;
      if (y + h > height)
	h = height - y;

      if (x < 0)
	{
	  s_buf -= x * 3;
	  w += x;
	  x = 0;
	}
      if (x < width && w > 0)
	{
	  p += x * 3;
	  if (x + w > width)
	    w = width - x;

	  unsigned char *i = s_buf;
	  while (i != s_buf + h * s_row_size)
	    {
	      unsigned char *j = i;
	      while (j != i + w * 3)
		{
		  int r = *j++;
		  int g = *j++;
		  int b = *j++;
		  *p++ = r;
		  *p++ = g;
		  *p++ = b;
		}
	      i += s_row_size;
	      p += (width - w) * 3;
	    }
	}
    }
}

void
gtk_frame::draw_rgba(int x, int y, 
		     unsigned char *s_buf, size_t s_row_size,
		     int w, int h)
{
  guchar *p = rgb_buf;

  if (y < 0)
    {
      s_buf -= y * s_row_size;
      h += y;
      y = 0;
    }
  if (y < height && h > 0)
    {
      p += y * width * 3;
      if (y + h > height)
	h = height - y;

      if (x < 0)
	{
	  s_buf -= x * 4;
	  w += x;
	  x = 0;
	}
      if (x < width && w > 0)
	{
	  p += x * 4;
	  if (x + w > width)
	    w = width - x;

	  unsigned char *i = s_buf;
	  while (i != s_buf + h * s_row_size)
	    {
	      unsigned char *j = i;
	      while (j != i + w * 4)
		{
		  int r = *j++;
		  int g = *j++;
		  int b = *j++;
		  int a = *j++;
		  if (a == 0)
		    p += 3;
		  else if (a == 255)
		    {
		      *p++ = r;
		      *p++ = g;
		      *p++ = b;
		    }
		  else
		    {
		      // XXX - tricky but handle rounding well
		      *p += ((r - *p) * a + 255 * 255 + 127) / 255 - 255;
		      ++p;
		      *p += ((g - *p) * a + 255 * 255 + 127) / 255 - 255;
		      ++p;
		      *p += ((b - *p) * a + 255 * 255 + 127) / 255 - 255;
		      ++p;
		    }
		}
	      i += s_row_size;
	      p += (width - w) * 3;
	    }
	}
    }
}

void
gtk_frame::set_size(int new_width, int new_height)
{
  if (width != new_width || height != new_height)
    {
      delete[] rgb_buf;

      width = new_width;
      height = new_height;

      rgb_buf = new guchar [height * width * 3];

      for (vector<GtkWidget *>::const_iterator i = widgets.begin();
	   i != widgets.end();
	   ++i)
	gtk_drawing_area_size(GTK_DRAWING_AREA(*i), width, height);
    }
}

bool
gtk_frame::handle_motion_notify_event(GtkWidget *w, GdkEventMotion *e)
{
  if (_view != NULL)
    {
      _view->notify_motion(e->x, e->y);
      return true;
    }

  return false;
}

bool
gtk_frame::handle_button_press_event(GtkWidget *w, GdkEventButton *e)
{
  if (_view != NULL)
    {
      switch (e->button)
	{
	case 1:
	  _view->press_button(e->x, e->y, 1);
	  return true;
	case 2:
	case 3:
	  _view->press_button(e->x, e->y, 2);
	  return true;
	default:
#ifdef L
	  L("handle_button_press_event: unhandled button %d\n", e->button);
#endif
	  break;
	}
    }

  return false;
}

bool
gtk_frame::handle_expose_event(GtkWidget *w, GdkEventExpose *e) const
{
  int x = e->area.x;
  int y = e->area.y;
  int wi = e->area.width;
  int he = e->area.height;
#ifdef L
  L("handle_expose_event: %dx%d+%d+%d on widget %p\n", wi, he, x, y, w);
#endif

  if (x < 0)
    {
      wi += x;
      x = 0;
    }
  if (wi > width - x)
    wi = width - x;

  if (y < 0)
    {
      he += y;
      y = 0;
    }
  if (he > height - y)
    he = height - y;

  guchar *p = rgb_buf + y * width * 3 + x * 3;
  GdkGC *gc = gdk_gc_new(w->window);
  gdk_draw_rgb_image(w->window, gc, x, y, wi, he,
		     GDK_RGB_DITHER_NORMAL, p, width * 3);
  gdk_gc_unref(gc);

  return true;
}

void
gtk_frame::handle_destroy(GtkWidget *w)
{
#ifdef L
  L("handle_destroy: widget = %p\n", w);
#endif

  vector<GtkWidget *>::iterator k = remove(widgets.begin(), widgets.end(), w);
  widgets.erase(k, widgets.end());
}

/* Glue functions for widget events.  */
namespace
{
  /* Glues a GTK+ motion notify event signal to the gtk_frame.  */
  gint
  handle_motion_notify_event(GtkWidget *w, GdkEventMotion *e, gpointer data)
    throw ()
  {
    gtk_frame *f = static_cast<gtk_frame *>(data);
    I(f != NULL);

    return f->handle_motion_notify_event(w, e);
  }

  /* Glues a GTK+ button press event signal to the gtk_frame.  */
  gint
  handle_button_press_event(GtkWidget *w, GdkEventButton *e, gpointer data)
    throw ()
  {
    gtk_frame *f = static_cast<gtk_frame *>(data);
    I(f != NULL);

    return f->handle_button_press_event(w, e);
  }

  /* Glues a GTK+ expose event signal to the gtk_frame.  */
  gint
  handle_expose_event(GtkWidget *w, GdkEventExpose *e, gpointer data)
    throw ()
  {
    gtk_frame *f = static_cast<gtk_frame *>(data);
    I(f != NULL);

    return f->handle_expose_event(w, e);
  }

  /* Glues a GTK+ destroy signal to the gtk_frame.  */
  void
  handle_destroy(GtkObject *o, gpointer data)
    throw ()
  {
    GtkWidget *w = GTK_WIDGET(o);
    gtk_frame *f = static_cast<gtk_frame *>(data);
    I(f != NULL);

    f->handle_destroy(w);
  }
} // (unnamed namespace)

GtkWidget *
gtk_frame::create_widget()
{
  GtkWidget *widget = gtk_drawing_area_new();
  I(GTK_IS_DRAWING_AREA(widget));

  gtk_drawing_area_size(GTK_DRAWING_AREA(widget), width, height);
  gtk_widget_add_events(widget,
			GDK_BUTTON_PRESS_MASK | GDK_POINTER_MOTION_MASK);
  gtk_signal_connect(GTK_OBJECT(widget), "destroy",
		     GTK_SIGNAL_FUNC(&::handle_destroy), this);
  gtk_signal_connect(GTK_OBJECT(widget), "expose_event",
		     GTK_SIGNAL_FUNC(&::handle_expose_event), this);
  gtk_signal_connect(GTK_OBJECT(widget), "motion_notify_event",
		     GTK_SIGNAL_FUNC(&::handle_motion_notify_event), this);
  gtk_signal_connect(GTK_OBJECT(widget), "button_press_event",
		     GTK_SIGNAL_FUNC(&::handle_button_press_event), this);

  widgets.push_back(widget);
  return widget;
}

bool
gtk_frame::handle_timeout()
{
  if (_view != NULL)
    {
      _view->tick(gdk_time_get());
      _view->advance();
    }

  return true;
}

gtk_frame::~gtk_frame()
{
  gtk_timeout_remove(timeout_handler_id);
  delete[] rgb_buf;

  for (vector<GtkWidget *>::iterator i = widgets.begin();
       i != widgets.end();
       ++i)
    gtk_signal_disconnect_by_data(GTK_OBJECT(*i), this);
}

/* Glue funtions for non-widget events.  */
namespace
{
  /* Glues GTK+ timeout to the gtk_frame.  */
  gint
  handle_timeout(gpointer data)
    throw ()
  {
    gtk_frame *f = static_cast<gtk_frame *>(data);
    I(f != NULL);

    return f->handle_timeout();
  }
} // (unnamed namespace)

gtk_frame::gtk_frame(view *p, int iwidth, int iheight)
  : _view(p),
    width (iwidth), height (iheight),
    rgb_buf(NULL)
{
  p->tick(gdk_time_get());
  rgb_buf = new guchar [height * width * 3];
  timeout_handler_id
    = gtk_timeout_add(TIMEOUT_INTERVAL, &::handle_timeout, this);
}
