/* View objects for Hyperplay
   Copyright (C) 1996, 1997, 1999 Hypercore Software Design, Ltd.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.  */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#undef const

#include <hyperplay/view.h>
#include <algorithm>

#ifdef HAVE_NANA_H
# include <nana.h>
# include <cstdio>
#else
# include <cassert>
# define I assert
#endif

using namespace hyperplay;
using namespace std;

void
view::add_modified_region(int x, int y,
			  int width, int height)
{
  if (!modified())
    {
      min_modified_x = x;
      max_modified_x = x + width;
      min_modified_y = y;
      max_modified_y = y + height;
    }
  else
    {
      if (x < min_modified_x)
	min_modified_x = x;
      if (x + width > max_modified_x)
	max_modified_x = x + width;
      if (y < min_modified_y)
	min_modified_y = y;
      if (y + height > max_modified_y)
	max_modified_y = y + height;
    }
}

void
view::refresh()
{
  for (layer *i = layers + 0;
       i != layers + NLAYERS;
       ++i)
    {
      if (i->object != NULL && i->mapped)
	i->object->draw(main_frame(), i->x, i->y, i->v);
    }

  _main_frame->update_area(min_modified_x, min_modified_y,
			   max_modified_x - min_modified_x,
			   max_modified_y - min_modified_y);
  clear_modified_region();
}

void
view::put(unsigned int l, unsigned int key,
	  int x, int y, int v, bool m)
{
  if (l >= NLAYERS)
    throw out_of_range("put");

  if (on_layer.base() != layers + 0
      && on_layer.base() - 1 == layers + l)
    {
      on_layer->event_bindings[LEAVE].eval();
      on_layer = reverse_iterator<layer *>(layers + 0);
    }

  if (layers[l].object != NULL && layers[l].mapped)
    add_modified_region(layers[l].x, layers[l].y,
			layers[l].object->width(),
			layers[l].object->height());

  layer nl = {managed_objects[key], x, y, v, m};
  layers[l] = nl;

  if (layers[l].object != NULL && layers[l].mapped)
    add_modified_region(layers[l].x, layers[l].y,
			layers[l].object->width(),
			layers[l].object->height());
}

void
view::select(unsigned int l, int v)
{
  if (v != layers[l].v)
    {
      layers[l].v = v;
      if (layers[l].object != NULL && layers[l].mapped)
	add_modified_region(layers[l].x, layers[l].y,
			    layers[l].object->width(),
			    layers[l].object->height());
    }
}

void
view::set_visible(int l, bool m)
{
  if (m != layers[l].mapped)
    {
      layers[l].mapped = m;
      if (layers[l].object != NULL)
	add_modified_region(layers[l].x, layers[l].y,
			    layers[l].object->width(),
			    layers[l].object->height());
    }
}

void
view::bind(unsigned int n, event_type type,
	   const interp::block &block)
{
  if (n >= NLAYERS)
    throw out_of_range("bind");
  if (layers[n].object == NULL)
    throw invalid_argument("bind");

  layers[n].event_bindings[type] = block;
}

void
view::notify_motion(int x, int y)
{
  reverse_iterator<layer *> i(layers + NLAYERS);
  while (i.base() != layers + 0)
    {
      if (i->object != NULL && i->mapped
	  && i->object->inside(x - i->x, y - i->y))
	break;

      ++i;
    }

  if (i != on_layer)
    {
      if (on_layer.base() != layers + 0)
	on_layer->event_bindings[LEAVE].eval();
      on_layer = i;
      if (on_layer.base() != layers + 0)
	on_layer->event_bindings[ENTER].eval();
    }
}

void
view::press_button(int x, int y, int button)
{
  notify_motion(x, y);
  if (on_layer.base() != layers + 0)
    switch (button)
      {
      case 1:
	on_layer->event_bindings[BUTTON1].eval();
	  break;
      case 2:
	on_layer->event_bindings[BUTTON2].eval();
	break;
      default:
#ifdef L
	L("press_button: Invalid button %d\n", button);
#endif
	break;
      }
}

void
view::clear_frame()
{
  if (on_layer.base() != layers + 0)
    {
      on_layer->event_bindings[LEAVE].eval();
      on_layer = reverse_iterator<layer *>(layers + 0);
    }

  static const layer initial_value = {NULL};
  fill(layers + 0, layers + NLAYERS, initial_value);
}

// Assign N to CONTENT.
void
view::assign(unsigned int key,
	     gadget *object)
{
  if (managed_objects[key] != NULL)
    delete managed_objects[key];
  managed_objects[key] = object;
}

void
view::set_shape(unsigned int key, gadget::shape *sh)
{
  managed_objects[key]->set_shape(sh);
}

void
view::set_main_frame(frame *f)
{
  _main_frame = f;
}

// Destruct.
view::~view()
{
  clear_frame();
  for (map<unsigned int, gadget *>::iterator i
	 = managed_objects.begin();
       i != managed_objects.end();
       ++i)
    {
      if (i->second != NULL)
	delete i->second;
    }
}

view::view()
  : _main_frame(NULL),
    on_layer(layers + 0),
    min_modified_x(0), max_modified_x(0),
    min_modified_y(0), max_modified_y(0)
{
  clear_frame();
}

