/* Player base objects for Hyperplay
   Copyright (C) 1996, 2000 Hypercore Software Design, Ltd.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.  */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#undef const

#include <hyperplay/player.h>
#include <hyperplay/imagefile.h>
#include <hyperplay/visual.h>
#include <hyperplay/file.h>	/* FIXME */
#include <algo.h>
#include <assert.h>

#ifdef HAVE_NANA_H
# include <nana.h>
# include <cstdio>
#else
# include <cassert>
# define I assert
#endif

using namespace hyperplay;
using namespace std;

// Save.
void
player::save(unsigned int n)
{
  hyperplay::data_file::iterator i = d.world_chunk (n);
  if (i == d.world_end ())
    throw runtime_error ("save: Index out of range");
  w.save (*i);
}

// Load.
void
player::load(unsigned int n)
{
  hyperplay::data_file::const_iterator i = d.world_chunk (n);
  if (i == d.world_end ())
    throw runtime_error ("load: Index out of range");
  w.load (*i);
}

// Set the next scene to play to SCENE.  The specified scene will be
// played later in the main loop.
void
player::next_scene(scene_name_type name)
{
  next_scene_name = name;
  continued_ = true;
}

// Change the scene.
void
player::change_scene()
{
  if (continued ())
    {
      continued_ = false;
      current_scene_name = next_scene_name;
      current_scene_ = s->find_scene(current_scene_name);
      static const hyperplay::chunk::key_type INIT ("INIT");
      interp::context *c = interp()->create_context();
      current_scene_.call(INIT, *interp(), *c);
      delete c;
    }
}

// Call a script code KEY with interpreter INTERP and context C.
void
player::scene_call(const hyperplay::chunk::key_type &key,
			const interpreter &interp,
			interp::context &c) const
{
  current_scene_.call (key, interp, c);
}

// Call a script code KEY in NAME with interpreter INTERP and context
// C.
void
player::scene_xcall(scene_name_type name,
			 const hyperplay::chunk::key_type &key,
			 const interpreter &interp,
			 interp::context &c) const
{
  hyperplay::scene scene(s->find_scene(name));
  scene.call (key, interp, c);
}

// Change scenes.
void
player::advance()
{
  if (_state == PLAYING)
    {
      while (continued())
	{
	  clear_frame();
	  reset_timer();
	  change_scene();
	}

      if (modified())
	refresh();
    }
}

// Start playing document.
void
player::play()
{
  _state = PLAYING;
  next_scene(s->initial_scene());
}

void
player::close()
{
  _state = STOPPED;
  delete s;
  s = NULL;
  d.close();
}

void 
player::open(const char *file_name)
{
  close();

  d.open(file_name);

#ifdef ENABLE_DEBUG
# ifdef L
  L("Script name: %s\n", d.script_name().c_str());
# endif
#endif
  s = new file_script(d.script_name().c_str());

  set_interp(create_interpreter()); // FIXME.
  s->configure(interp());
  main_frame()->set_size(s->frame_width(), s->frame_height());
  //main_frame()->audio ()->set_prefix (prefix.c_str ());
  for (file_script::cursor_iterator
	 ic = static_cast<file_script *>(s)->begin_cursor();
       ic != static_cast<file_script *>(s)->end_cursor(); ++ic)
    {
      //const script::cursor &c = *ic;
      //main_frame()->define_cursor (c.index, c.and_bits, c.xor_bits, c.hot);
    }
}

// Return the file name prefix for data.
string
player::data_prefix() const
{
  return string(static_cast<file_script *>(s)->data_dir()) + "/";
}

// Destruct.
player::~player()
{
}

namespace
{
  const byte_type *
  invisible_1(const byte_type *first, const byte_type *last,
	      interp::context &c, interp::command_data *data)
  {
#ifdef L
    L("invisible.1: Not implemented\n");
#endif

    return last;
  }

  const byte_type *
  image_1(const byte_type *first, const byte_type *last,
	  interp::context &c, interp::command_data *data)
  {
    player *p = dynamic_cast<player *>(data);
    I(p != NULL);

    unsigned int n = *(unsigned char *) first++;

    const byte_type *bos = first;
    while (*first != '\0')
      ++first;
    string name (bos, first++ - bos);

    int width, height;
    data_file::unpack16(first, width); // FIXME
    data_file::unpack16(first, height); // FIXME

    unsigned long t1;
    interpreter::unpack(first, t1);
    vector<visual::point> s_pos;
    s_pos.reserve(t1);
    for (int j = 0; j != (int) t1; ++j)
      {
	visual::point ipos;
	data_file::unpack16(first, ipos.x);	// FIXME
	data_file::unpack16(first, ipos.y);	// FIXME
	s_pos.push_back(ipos);
      }

    file::png_reader reader;
    string file_name
      = p->data_prefix() + name + file::png_reader::file_suffix();
    reader.open(file_name.c_str());
    visual::raw_image *master = reader.read();
    visual::image *image
      = new visual::image(width, height,
			  master,
			  s_pos.begin(), s_pos.end());
    p->assign(n, image);

    return last;
  }

  const byte_type *
  poly_1(const byte_type *first, const byte_type *last,
	 interp::context &c, interp::command_data *data)
  {
    player *p = dynamic_cast<player *>(data);
    I(p != NULL);

    int n = *first++ & 0xff;

    unsigned long np;
    interpreter::unpack(first, np);
    vector<visual::point> vp;
    vp.reserve(np);
    for (int i = 0; i != int(np); ++i)
      {
	visual::point tp;
	data_file::unpack16(first, tp.x);
	data_file::unpack16(first, tp.y);
	vp.push_back(tp);
      }

    gadget::shape *sh = new gadget::shape(vp.begin(), vp.end());
    p->set_shape(n, sh);

    return last;
  }

  const byte_type *
  place_1(const byte_type *first, const byte_type *last,
	  interp::context &c, interp::command_data *data)
  {
    player *p = dynamic_cast<player *>(data);
    I(p != NULL);

    unsigned int l = *(unsigned char *) first++;
    unsigned int o = *(unsigned char *) first++;

    int x, y;
    data_file::unpack16(first, x);
    data_file::unpack16(first, y);

    unsigned int v = *(unsigned char *) first++;
    int m = *(unsigned char *) first++;

    p->put(l, o, x, y, v, m == 1);

    return last;
  }

  const byte_type *
  variation_1(const byte_type *first, const byte_type *last,
	      interp::context &c, interp::command_data *data)
  {
    player *p = dynamic_cast<player *>(data);
    I(p != NULL);

    int l = *first++ & 0xff;
    int v = *first++ & 0xff;

    p->select(l, v);

    return last;
  }

  const byte_type *
  map_1(const byte_type *first, const byte_type *last,
	interp::context &c, interp::command_data *data)
  {
    player *p = dynamic_cast<player *>(data);
    I(p != NULL);

    int l = *first++ & 0xff;
    int m = *first++ & 0xff;

    p->set_visible(l, m == 1);

    return last;
  }

  const byte_type *
  cursor_1(const byte_type *first, const byte_type *last,
	   interp::context &c, interp::command_data *data)
  {
#ifdef L
    L("cursor.1: Not implemented\n");
#endif

    return last;
  }

  const byte_type *
  bind_1(const byte_type *first, const byte_type *last,
	 interp::context &c, interp::command_data *data)
  {
    player *p = dynamic_cast<player *>(data);
    I(p != NULL);

    int n = *first++ & 0xff;
    int type = *first++ & 0xff;

    // XXX egcs 1.1.2 dislikes c.interp().
    interp::context *cp = &c;
    interp::block b(first, last, cp->interp());
    switch (type)
      {
      case 0:
	p->bind(n, LEAVE, b);
	break;
      case 1:
	p->bind(n, ENTER, b);
	break;
      case 2:
	p->bind(n, BUTTON1, b);
	break;
      case 3:
	p->bind(n, BUTTON2, b);
	break;
      default:
	c.abort("bind.1: Invalid event type");
	break;
      }

    return last;
  }

  const byte_type *
  bind_event_1(const byte_type *first, const byte_type *last,
	       interp::context &c, interp::command_data *data)
  {
#ifdef L
    L("bind_event.1: Not implemented\n");
#endif

    return last;
  }

  const byte_type *
  refresh_1(const byte_type *first, const byte_type *last,
	    interp::context &c, interp::command_data *data)
  {
#ifdef L
    L("refresh.1: Not implemented\n");
#endif

    return last;
  }

  const byte_type *
  sound_1(const byte_type *first, const byte_type *last,
	  interp::context &c, interp::command_data *data)
  {
#ifdef L
    L("sound.1: Not implemented\n");
#endif

    return last;
  }

  const byte_type *
  play_sound_1(const byte_type *first, const byte_type *last,
	       interp::context &c, interp::command_data *data)
  {
#ifdef L
    L("play_sound.1: Not implemented\n");
#endif

    return last;
  }

  const byte_type *
  music_1(const byte_type *first, const byte_type *last,
	  interp::context &c, interp::command_data *data)
  {
#ifdef L
    L("music.1: Not implemented\n");
#endif

    return last;
  }

  const byte_type *
  music_repeat_1(const byte_type *first, const byte_type *last,
		 interp::context &c, interp::command_data *data)
  {
#ifdef L
    L("music_repeat.1: Not implemented\n");
#endif

    return last;
  }

  const byte_type *
  play_music_1(const byte_type *first, const byte_type *last,
	       interp::context &c, interp::command_data *data)
  {
#ifdef L
    L("play_music.1: Not implemented\n");
#endif

    return last;
  }

  const byte_type *
  stop_music_1(const byte_type *first, const byte_type *last,
	       interp::context &c, interp::command_data *data)
  {
#ifdef L
    L("stop_music.1: Not implemented\n");
#endif

    return last;
  }

  // Command to get a value of a variable.
  const byte_type *
  var_1(const byte_type *first, const byte_type *last,
	interp::context &c, interp::command_data *data)
  {
    player *p = dynamic_cast<player *>(data);
    I(p != NULL);

    unsigned long pid;
    hyperplay::data_file::unpack32(first, pid);
    unsigned long name = c.o_stack.back();
    c.n_stack.push_back(p->var(name, pid));

    return last;
  }

  // Command to set a variable to a value.
  const byte_type *
  set_var_1(const byte_type *first, const byte_type *last,
	    interp::context &c, interp::command_data *data)
  {
    player *p = dynamic_cast<player *>(data);
    I(p != NULL);

    unsigned long pid;
    hyperplay::data_file::unpack32(first, pid);
    unsigned long name = c.o_stack.back();
    if (c.n_stack.empty())
      c.abort("set_property.1: Stack underflow");
    long ex_top = c.n_stack.back();
    c.n_stack.pop_back();
    p->set_var(name, pid, ex_top);

    return last;
  }

  const byte_type *
  after_1(const byte_type *first, const byte_type *last,
	  interp::context &c, interp::command_data *data)
  {
    player *p = dynamic_cast<player *>(data);
    I(p != NULL);

    unsigned long interval;
    interpreter::unpack(first, interval);
    p->schedule(interval, interp::block(first, last, c.interp()));

    return last;
  }

  const byte_type *
  sync_time_1(const byte_type *first, const byte_type *last,
	      interp::context &c, interp::command_data *data)
  {
    player *p = dynamic_cast<player *>(data);
    I(p != NULL);

    p->sync_time();

    return last;
  }
} // (unnamed namespace)

struct hplay_commands
{
  // Command to call a scene code.
  struct call_1: interp::command
  {
    player &p;
    call_1(player &pbase): p (pbase) {}
    const byte_type *operator() (const byte_type *first,
				 const byte_type *last,
				 interp::context &c,
				 const interpreter &) const
    {
      p.scene_call(first, *c.interp(), c);
      return last;
    }
  };				// struct call_1

  // Command to call a scene code of another scene.
  struct xcall_1: interp::command
  {
    player *p;
    xcall_1(player *player): p (player) {}
    const byte_type *operator() (const byte_type *first,
				 const byte_type *last,
				 interp::context &c,
				 const interpreter &) const
    {
      unsigned long scene;
      hyperplay::data_file::unpack32 (first, scene);
      assert (p != NULL);
      p->scene_xcall(scene, first, *c.interp(), c);
      return last;
    }
  };				// xcall_1

  // Save the current world.
  struct save_1: interp::command
  {
    player *p;
    save_1(player *player): p (player) {}
    const byte_type *operator() (const byte_type *,
				 const byte_type *last,
				 interp::context &c,
				 const interpreter &) const
    {
      if (c.n_stack.empty ())
	c.abort("save.1: Stack underflow");
      int ex_top = c.n_stack.back ();
      c.n_stack.pop_back ();
      assert (p != NULL);
      p->save (ex_top);
      return last;
    }
  };				// save_1

  // Load a world.
  struct load_1: interp::command
  {
    player *p;
    load_1(player *player): p (player) {}
    const byte_type *operator() (const byte_type *,
				 const byte_type *last,
				 interp::context &c,
				 const interpreter &) const
    {
      if (c.n_stack.empty ())
	c.abort("load.1: Stack underflow");
      int ex_top = c.n_stack.back ();
      c.n_stack.pop_back ();
      assert (p != NULL);
      p->load (ex_top);
      return last;
    }
  };				// load_1

  // Push the current scene on the value stack.
  struct current_scene_1: interp::command
  {
    player *p;
    current_scene_1(player *player): p (player) {}
    const byte_type *operator() (const byte_type *,
				 const byte_type *last,
				 interp::context &c,
				 const interpreter &) const
    {
      assert (p != NULL);
      c.n_stack.push_back (p->current_scene ());
      return last;
    }
  };				// current_scene_1

  // Specify the next scene from the value stack.
  struct set_next_scene_1: interp::command
  {
    player *p;
    set_next_scene_1(player *player): p (player) {}
    const byte_type *operator() (const byte_type *,
				 const byte_type *last,
				 interp::context &c,
				 const interpreter &) const
    {
      if (c.n_stack.empty ())
	c.abort("load.1: Stack underflow");
      int ex_top = c.n_stack.back ();
      c.n_stack.pop_back ();
      p->next_scene (ex_top);
      return last;
    }
  };				// set_next_scene_1

  // Specify the next scene.
  struct next_scene_1: interp::command
  {
    player &p;
    next_scene_1(player &player): p (player) {}
    const byte_type *operator() (const byte_type *first,
				 const byte_type *last,
				 interp::context &,
				 const interpreter &) const
    {
      unsigned long s;
      hyperplay::data_file::unpack32 (first, s);
      p.next_scene (s);
      return last;
    }
  };
};

namespace
{
  void
  add_commands(interpreter &i, interp::command_data *data)
  {
    i.add_command("invisible.1", &invisible_1, data);
    i.add_command("image.1", &image_1, data);
    i.add_command("poly.1", &poly_1, data);
    i.add_command("place.1", &place_1, data);
    i.add_command("variation.1", &variation_1, data);
    i.add_command("map.1", &map_1, data);
    i.add_command("cursor.1", &cursor_1, data);
    i.add_command("bind.1", &bind_1, data);
    i.add_command("bind_event.1", &bind_event_1, data);
    i.add_command("refresh.1", &refresh_1, data);
    i.add_command("sound.1", &sound_1, data);
    i.add_command("play_sound.1", &play_sound_1, data);
    i.add_command("music.1", &music_1, data);
    i.add_command("music_repeat.1", &music_repeat_1, data);
    i.add_command("play_music.1", &play_music_1, data);
    i.add_command("stop_music.1", &stop_music_1, data);

    // Variable manipulation commands
    i.add_command("var.1", &var_1, data);
    i.add_command("property.1", &var_1, data); // backward compatibility
    i.add_command("set_var.1", &set_var_1, data);
    i.add_command("set_property.1", &set_var_1, data); // backward compatibility

    i.add_command("call.1", new hplay_commands::call_1(dynamic_cast<player &>(*data)));
    i.add_command("xcall.1", new hplay_commands::xcall_1(dynamic_cast<player *>(data)));
    i.add_command("save.1", new hplay_commands::save_1(dynamic_cast<player *>(data)));
    i.add_command("load.1", new hplay_commands::load_1(dynamic_cast<player *>(data)));

    // Timer commands
    i.add_command("after.1", &after_1, data);
    i.add_command("sync_time.1", &sync_time_1, data);
    i.add_command("reschedule.1", &sync_time_1, data); // backward compatibility

    i.add_command("current_scene.1", new hplay_commands::current_scene_1(dynamic_cast<player *>(data)));
    i.add_command("set_next_scene.1", new hplay_commands::set_next_scene_1(dynamic_cast<player *>(data)));
    i.add_command("next_scene.1", new hplay_commands::next_scene_1(dynamic_cast<player &>(*data)));
  }
} // (unnamed namespace)

void
player::set_interp(interpreter *i)
{
  _interp = i;
}

interpreter *
player::create_interpreter()
{
  interpreter *i = new interpreter;
  add_commands(*i, this);
  return i;
}

player::player()
  : s(NULL),
    _interp(NULL),
    _state(STOPPED),
    continued_(false)
{
}
