/* Interpreter objects for Hyperplay
   Copyright (C) 1996, 1997, 1999 Hypercore Software Design, Ltd.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.  */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#undef const

#include <hyperplay/interp.h>
#include <hyperplay/file.h>	/* FIXME */

#include <algorithm>
#include <stdexcept>

#ifdef HAVE_NANA_H
# include <nana.h>
# include <cstdio>
#else
# include <cassert>
# define I assert
#endif

using namespace hyperplay::interp;
using namespace std;
using hyperplay::byte_type;

/* Aborts the current execution by throwing an error.  */
void
context::abort(const string &reason)
{
  throw interpreter::error(reason);
}

// Undefine for code I.
void
interpreter::undefine (int i)
{
  commands[i].first = NULL;
  commands[i].second = NULL;
}

// Get a variable length integer.
void
interpreter::unpack (const byte_type *&i, unsigned long &r)
{
  r = 0;
  size_t b = (unsigned char) *i++;
  while (b >= 128)
    {
      r = r + (b - 128) << 7;
      b = (unsigned char) *i++;
    }
  r += b;
}

// Execute code.
const byte_type *
context::execute (const byte_type *first,
		  const byte_type *last)
{
  while (first != last)
    {
      unsigned long len;
      interpreter::unpack (first, len);
      if (len == 0)
	return first;
      int cmd = (unsigned char) *first++;
      --len;
      first = _interp->dispatch(cmd, first, first + len, *this);
    }
  return last;
}

context *
interpreter::create_context() const
{
  return new context(this);
}

const byte_type *
interpreter::dispatch(int i, const byte_type *first, const byte_type *last,
		      context &c) const
{
  if (commands[i].first == NULL)
    throw runtime_error("No command bound");
  return (*commands[i].first)(first, last, c, commands[i].second);
}

void
interpreter::bind(int i, const string &name)
{
  if (i < 0 || i > 255)
    throw out_of_range("bind");

  map<string, pair<command_handler, command_data *> >::iterator k
    = named_commands.find(name);
  if (k == named_commands.end())
    throw runtime_error(name + ": Missing command");

  commands[i] = k->second;
}

void
interpreter::configure(const byte_type *first, const byte_type *last)
{
  if (first != last)
    {
      int i = (unsigned char) *first++;
      string name(first, last);
      bind(i, name);
    }
}

void
interpreter::add_command(const string &name,
			 command_handler handler, command_data *data)
{
  named_commands.insert(make_pair(name, make_pair(handler, data)));
}

void
interpreter::add_command(const string &name,
			 command *c)
{
  command_objects.push_back(c);
  add_command(name, &glue_command, c);
}

// Destruct.
interpreter::~interpreter ()
{
  while (!command_objects.empty())
    {
      delete command_objects.back();
      command_objects.pop_back();
    }
}

namespace
{
  const byte_type *
  immediate_1(const byte_type *first, const byte_type *last,
	      context &c, command_data *data)
  {
    unsigned long n;
    hyperplay::data_file::unpack32(first, n); // FIXME
    c.n_stack.push_back(n);

    return last;
  }

  const byte_type *
  pop_1(const byte_type *first, const byte_type *last,
	context &c, command_data *data)
  {
    if (c.n_stack.empty())
      c.abort("pop.1: Stack underflow");
    c.n_stack.pop_back();

    return last;
  }

  const byte_type *
  duplicate_1(const byte_type *first, const byte_type *last,
	      context &c, command_data *data)
  {
    if (c.n_stack.empty())
      c.abort("duplicate.1: Stack underflow");
    unsigned long n = c.n_stack.back();
    c.n_stack.push_back(n);

    return last;
  }

  const byte_type *
  dyadic_1(const byte_type *first, const byte_type *last,
	   context &c, command_data *data)
  {
    unsigned int op = *(unsigned char *) first++;
    int ex_top = c.n_stack.back();
    c.n_stack.pop_back();

    switch (op)
      {
      case 0:
	c.n_stack.back() += ex_top;
	break;
      case 1:
	c.n_stack.back() -= ex_top;
	break;
      case 2:
	c.n_stack.back() *= ex_top;
	break;
      case 3:
	c.n_stack.back() /= ex_top;
	break;
      case 4:
	c.n_stack.back() %= ex_top;
	break;
      default:
	c.abort("binary.1: Invalid sub-operation code");
	break;
      }

    return last;
  }

  const byte_type *
  begin_domain_1(const byte_type *first, const byte_type *last,
		 context &c, command_data *data)
  {
    unsigned long name;
    hyperplay::data_file::unpack32(first, name); // FIXME
    c.o_stack.push_back(name);

    return last;
  }

  const byte_type *
  end_domain_1(const byte_type *first, const byte_type *last,
	       context &c, command_data *data)
  {
    if (c.o_stack.empty())
      c.abort("end_object.1: Stack underflow");
    c.o_stack.pop_back();

    return last;
  }

  const byte_type *
  if_1(const byte_type *first, const byte_type *last,
       context &c, command_data *data)
  {
    // Check for the operand.
    if (first == last)
      c.abort("if.1: Missing operand");
    unsigned int op = *first++ & 0xff;
    // Check for the context.
    if (c.n_stack.size () < 2)
      c.abort("if.1: Stack underflow");
    // Do the real work now.
    int ex_top = c.n_stack.back ();
    c.n_stack.pop_back ();

    switch (op)
      {
      case 0:
	if (c.n_stack.back () == ex_top)
	  last = first;
	break;
      case 1:
	if (c.n_stack.back () != ex_top)
	  last = first;
	break;
      default:
	c.abort("if.1: Invalid condition code");
	break;
      }

    return last;
  }

  const byte_type *
  do_1(const byte_type *first, const byte_type *last,
       context &c, command_data *data)
  {
    c.execute(first, last);

    return last;
  }
} // (unnamed namespace)

// Construct.
interpreter::interpreter ()
{
  fill(commands + 0, commands + 256,
       pair<command_handler, command_data *>(0, 0));

  add_command("immediate.1", &immediate_1, NULL);
  add_command("pop.1", &pop_1, NULL);
  add_command("duplicate.1", &duplicate_1, NULL);
  add_command("dyadic.1", &dyadic_1, NULL);
  add_command("binary.1", &dyadic_1, NULL); // XXX: backward compatibility
  add_command("begin_domain.1", &begin_domain_1, NULL);
  add_command("object.1", &begin_domain_1, NULL); // XXX backward compatibility
  add_command("end_domain.1", &end_domain_1, NULL);
  add_command("end_object.1", &end_domain_1, NULL); // XXX backward compatibility
  add_command("if.1", &if_1, NULL);
  add_command("if_else.1", &if_1, NULL); // XXX backward compatibility
  add_command("do.1", &do_1, NULL);
  add_command("execute.1", &do_1, NULL); // XXX backward compatibility
}

/* Glues a command call to a command object.  */
const byte_type *
interpreter::glue_command(const byte_type *first,
			  const byte_type *last,
			  class context &context,
			  command_data *data)
{
  command *c = dynamic_cast<command *>(data);
  I(c != NULL);
  return (*c)(first, last, context, *context.interp());
}
		  
