/* PNG file reader object for Hyperplay.
   Copyright (C) 1996, 1997, 1999 Hypercore Software Design, Ltd.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.  */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#undef const

#include <hyperplay/imagefile.h>

#include <stdexcept>
#include <cerrno>

using namespace hyperplay::file;
using namespace hyperplay;
using namespace std;

visual::raw_image *
png_reader::read() const
{
  png_bytep *rows = NULL;
  &rows;			// XXX rows must not be on a register.
  if (setjmp(png_ptr->jmpbuf) != 0)
    {
      delete [] rows;
      throw runtime_error("libpng error");
    }

  int width = png_get_image_width(png_ptr, info_ptr);
  int height = png_get_image_height(png_ptr, info_ptr);
  int color_type = png_get_color_type(png_ptr, info_ptr);
  visual::raw_image *image
    = new visual::raw_image(width, height,
			    color_type == PNG_COLOR_TYPE_RGB_ALPHA);

  rows = new png_bytep [height];
  for (int y = 0; y != height; ++y)
    rows[y] = image->address(0, y);

  png_read_image(png_ptr, rows);
  png_read_end(png_ptr, NULL);

  delete [] rows;
  return image;
}

void
png_reader::open(const char *name)
{
  if (fp != NULL)
    fclose(fp);

  fp = fopen(name, "rb");
  if (fp == NULL)
    throw runtime_error(string(name) + ": " + strerror(errno));

  if (png_ptr == NULL)
    {
      png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING,
				       NULL, NULL, NULL);
      if (png_ptr == NULL)
	throw runtime_error("png_create_read_struct failed");

      info_ptr = png_create_info_struct(png_ptr);
      if (info_ptr == NULL)
	{
	  png_destroy_read_struct(&png_ptr, NULL, NULL);
	  png_ptr = NULL;
	  throw runtime_error("png_create_info_struct failed");
	}
    }

  if (setjmp(png_ptr->jmpbuf) != 0)
    throw runtime_error("libpng error");

  png_init_io(png_ptr, fp);
  png_read_info(png_ptr, info_ptr);

  int color_type = png_get_color_type(png_ptr, info_ptr);
  if (color_type == PNG_COLOR_TYPE_PALETTE
      || png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS))
    png_set_expand(png_ptr);

  int bit_depth = png_get_bit_depth(png_ptr, info_ptr);
  if (bit_depth == 16)
    png_set_strip_16(png_ptr);
  else if (bit_depth < 8)
    png_set_packing(png_ptr);

  if (color_type == PNG_COLOR_TYPE_GRAY
      || color_type == PNG_COLOR_TYPE_GRAY_ALPHA)
    png_set_gray_to_rgb(png_ptr);

  const char *env = getenv("SCREEN_GAMMA");
  if (env != NULL)
    {
      double screen_gamma = atof(env);
      double file_gamma;
      if (!png_get_gAMA(png_ptr, info_ptr, &file_gamma))
	file_gamma = 0.45455;
      png_set_gamma(png_ptr, screen_gamma, file_gamma);
    }

  png_read_update_info(png_ptr, info_ptr);
}

png_reader::~png_reader()
{
  if (setjmp(png_ptr->jmpbuf) != 0)
    throw runtime_error("libpng error");

  if (png_ptr != NULL)
    png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
  if (fp != NULL)
    fclose(fp);
}

png_reader::png_reader()
  : fp(NULL),
    png_ptr(NULL),
    info_ptr(NULL)
{
}

#if 0

#include <algo.h>
#include <function.h>
#include <stdexcept>

using namespace std;

// Return the size of the image in pixels.
hplay_png_reader::size_type
hplay_png_reader::size () const
{
  return size_type (png.info_ptr->width, png.info_ptr->height);
}

// Test if the color at I is transparent.
bool
hplay_png_reader::transparent (color_iterator i) const
{
  ptrdiff_t d = 0;
  distance (color_begin (), i, d);
  if (d >= png.info_ptr->num_trans)
    return false;
  else
    return png.info_ptr->trans[d] == 0;
}

// Read all the rows of the image into the array from FIRST.
hplay_png_reader &
hplay_png_reader::read_image (char *const *first)
{
  png_read_image (png.png_ptr, (png_bytep *) first);
  return *this;
}

// Empty destructor to avoid generation in client side.
hplay_png_reader::~hplay_png_reader ()
{
}

hplay_png_reader::hplay_png_reader (const char *n)
  : stream (n, ios::in | ios::binary),
    png ()
{
  png_set_read_fn (png.png_ptr, this, &read_data);
  png_read_info (png.png_ptr, png.info_ptr);
  if (png.info_ptr->color_type != PNG_COLOR_TYPE_PALETTE)
    throw runtime_error ("Non-paletted image is not supported");
  if (png.info_ptr->bit_depth < 8)
    png_set_packing (png.png_ptr);
  png_read_update_info (png.png_ptr, png.info_ptr);
}

void
hplay_png_reader::read_data (png_structp png_ptr,
			     png_bytep buf,
			     png_size_t n)
{
  hplay_png_reader *that = (hplay_png_reader *) png_get_io_ptr (png_ptr);
  if (!that->stream.read ((char *) buf, n))
    png_error (png_ptr, "read failed");
}

#endif

