/* Script class for Hyperplay
   Copyright (C) 1996, 2000 Hypercore Software Design, Ltd.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.  */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#undef const

#include <hyperplay/file.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#include <algorithm>
#include <cstdlib>

#ifndef SCRIPT_PATH_ENV_NAME
# define SCRIPT_PATH_ENV_NAME "HYPERPLAY_SCRIPT_PATH"
#endif

using namespace hyperplay;
using namespace std;

const char *const DEFAULT_SCRIPTS_DIR = PKGDATADIR "/scripts";
const char *const SCRIPT_FILE_SUFFIX = ".hps";

script::cursor
file_script::cursor_iterator::operator*() const
{
  vector <char> d;
  assert(i != 0);
  (*i).load(d);
  vector<char>::const_iterator j = d.begin();
  cursor c;
  c.index = *(unsigned char *) j++;
  data_file::unpack16(j, c.hot.x);
  data_file::unpack16(j, c.hot.y);
  vector <char>::const_iterator k = j + 32 * 4;
  copy(j, k, c.and_bits);
  j = k;
  k += 32 * 4;
  copy(j, k, c.xor_bits);
  return c;
}

const chunk::key_type CURS("CURS");

file_script::cursor_iterator &
file_script::cursor_iterator::operator++()
{
  assert(i != 0);
  ++i;
  i = find(i, end, CURS);
  return *this;
}

file_script::cursor_iterator::cursor_iterator(data_file::const_iterator first,
					      data_file::const_iterator last)
  : i(find(first, last, CURS)), end(last)
{
  assert(first != 0);
  assert(last != 0);
}

vector<string> file_script::search_path_a;

scene
file_script::find_scene(unsigned long n) const
{
  scene_map::const_iterator i = scenes.find(n);
  if (i == scenes.end())
    return scene();
  else
    return i->second.apply(file);
}

// Arrange INTERP using commands in TABLE for execution of codes in
// the script.  If a specified command is not provided, this
// function throws `runtime_error'.
void
file_script::configure(interpreter *interp) const
{
  for (vector <chunk>::const_iterator i = file.begin();
       i != file.end();
       ++i)
    {
      static const chunk::key_type key("CMDT");
      if (*i == key)
	{
	  chunk::data dat;
	  i->load(dat);
	  interp->configure(dat.begin(), dat.end());
	}
    }
}

void
file_script::open(const char *script_name)
{
  /* "/NAME.hps" */
  string suffix("/");
  suffix.append(script_name);
  suffix.append(SCRIPT_FILE_SUFFIX);

  string dir;
  string file_name;
  for (vector<string>::iterator i = search_path_a.begin();
       i != search_path_a.end();
       ++i)
    {
      /* First, SCRIPTSDIR/NAME.hps is tried.  */
      dir = *i;
      file_name = dir + suffix;
      if (access(file_name.c_str(), R_OK) != -1)
	break;

      /* If that fails, SCRIPTSDIR/NAME/NAME.hps is tried.  */
      dir.append("/");
      dir.append(script_name);
      file_name = dir + suffix;
      if (access(file_name.c_str(), R_OK) != -1)
	break;
    }

  data_dir_a = dir;
  file.open(file_name);
  scenes = scene_map ();
  make_index (file.begin (), file.end (), scenes);

  static const chunk::key_type FSIZ ("FSIZ");
  vector <chunk>::const_iterator i =
    find (file.begin (), file.end (), FSIZ);
  if (i == file.end ())
    {
      set_frame_size(640, 480);
    }
  else
    {
      chunk::data d;
      (*i).load (d);
      chunk::data::const_iterator j = d.begin ();

      int width, height;
      data_file::unpack16(j, width);
      data_file::unpack16(j, height);
      set_frame_size(width, height);
    }

  static const chunk::key_type TITL("TITL");
  i = find (file.begin (), file.end (), TITL);
  if (i == file.end ())
    set_title("Untitled");
  else
    {
      chunk::data d;
      (*i).load (d);
      set_title(string(d.begin (), d.size ()));
    }
}

file_script::~file_script()
{
}

file_script::file_script(const char *script_name)
{
  if (search_path_a.empty())
    {
      const char *path = getenv(SCRIPT_PATH_ENV_NAME);
      if (path != NULL)
	set_search_path(path);
      else
	set_search_path("");
    }

  open(script_name);
}

void
file_script::make_index(data_file::const_iterator first,
			data_file::const_iterator last,
			scene_map &scenes)
{
  static const chunk::key_type key ("SCNE");
  assert (first != 0);
  assert (last != 0);
  first = find (first, last, key);
  while (first != last)
    {
      chunk::data d;
      (*first++).load (d);
      const char *i = d.begin ();
      unsigned long n;
      data_file::unpack32 (i, n);
      data_file::const_iterator next = find (first, last, key);
      scenes.insert (scene_map::value_type (n, scene_token (first, next)));
      first = next;
    }
}

void
file_script::set_search_path(const char *path)
{
  vector<string> t;

  string p(path);
  p.append(":");

  string::size_type pos1 = 0;
  for (;;)
    {
      string::size_type pos2 = p.find_first_of(":", pos1);
      if (pos2 == string::npos)
	break;

      if (pos1 == pos2)
	t.push_back(DEFAULT_SCRIPTS_DIR);
      else
	t.push_back(p.substr(pos1, pos2));

      pos1 = pos2 + 1;
    }

  search_path_a.swap(t);
}
