/* Data file object for Hyperplay
   Copyright (C) 1996, 1997, 1999 Hypercore Software Design, Ltd.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.  */

#include "hyperplay/file.h"

#include <algorithm>
#include <stdexcept>

using namespace std;
using namespace hyperplay;

// `data_file::toc_size' returns the size of the table of contents.
data_file::size_type
data_file::size () const
{
  return chunks.size ();
}

//
data_file::iterator
data_file::begin ()
{
  return chunks.begin ();
}

// `data_file::toc_begin' returns the beginning of the table of
// contents.
data_file::const_iterator
data_file::begin () const
{
  return chunks.begin ();
}

//
data_file::iterator
data_file::end ()
{
  return chunks.end ();
}

// `data_file::toc_end' returns the end of the table of contents.
data_file::const_iterator
data_file::end () const
{
  return chunks.end ();
}

// `data_file::read' reads bytes from the stream.
void
data_file::read (char *buf, size_t n) const
{
  assert (buf != 0);
  assert (fs != 0);
  if (!fs->read (buf, n))
    throw runtime_error (fname + ": short file");
}

// `data_file::advance' advances the file position of the stream.
void
data_file::advance (size_t n) const
{
  assert (fs != 0);
  if (!fs->seekg (n, ios::cur))
    throw runtime_error (fname + ": short file");
}

// `check_file_signature' checks the file signature, and throws an
// exception unless it is right.
inline void
check_file_signature(const char *buf)
{
  assert(buf != 0);
  static const char sig[] = "HpCF";
  if (!equal(sig + 0, sig + 4, buf))
    throw runtime_error("file signature mismatch");
}

// `read_uint32' get an 32-bit integer from a stream.
inline unsigned long
read_uint32 (istream &stream)
{
  char buf[4];
  if (!stream.read (buf, 4))
    throw runtime_error ("short file");
  const char *p = buf;
  unsigned long n;
  data_file::unpack32 (p, n);
  return n;
}

void
data_file::extract (back_insert_iterator <vector <chunk> > i)
{
  char s[12];
  read (s, 12);
  check_file_signature (s);
  assert (fs != 0);
  unsigned long n = read_uint32 (*fs);
  while (n != 0)
    {
      if (n < 4)
	throw runtime_error (fname + ": invalid file format");
      char keybuf[4];
      read (keybuf, 4);
      n -= 4;
      streampos first = fs->tellg ();
      advance (n);
      streampos last = fs->tellg ();
      *i++ = chunk (keybuf, &*fs, first, last);
      n = read_uint32 (*fs);
    }
}

// `data_file::close' closes the file stream.
void
data_file::close ()
{
  assert (fs != 0);
  if (fs->is_open())
    {
      fs->close ();
      if (!*fs)
	throw runtime_error (fname + ": close failed");
      chunks = toc ();
      fname = "(not open)";
    }
}

// `for_reading' is the open mode for reading data files.
const ios::openmode for_reading = ios::in | ios::binary;
const ios::openmode for_read_write = ios::in | ios::out | ios::binary;

// `data_file::open' opens file NAME and make on-memory table of
// contents.
void
data_file::open (const string &name)
{
  fname = name;
  assert (fs != 0);
  fs->open (name.c_str (), for_read_write);
  if (!*fs)
    throw runtime_error (fname + ": open failed");
  chunks = toc ();
  extract (back_inserter (chunks));
}

// explicit out-of-line definition for DLL
data_file::~data_file ()
{
  delete fs;
}

// explicit out-of-line definition for DLL
data_file::data_file ()
  : fname ("(not open)"),
    fs (new fstream)
{
  assert (fs != 0);
}

// Construct and open file NAME.
data_file::data_file (const string &name)
  : fname (name),
    fs (new fstream (name.c_str (), for_read_write))
{
  assert (fs != 0);
  if (!*fs)
    throw runtime_error (fname + ": open failed");
  extract (back_inserter (chunks));
}
