/* -*-C++-*- */
/* View objects for Hyperplay
   Copyright (C) 1996, 2000 Hypercore Software Design, Ltd.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.  */

#ifndef _HYPERPLAY_VIEW_H
#define _HYPERPLAY_VIEW_H 1

#include <hyperplay/interp.h>

namespace hyperplay
{
  using namespace std;

  /* Number of layers.  */
  const size_t NLAYERS = 256;

  enum event_type {LEAVE = 0, ENTER, BUTTON1, BUTTON2, EVENT_MAX};

  /* Interface to a frame.  */
  struct frame
  {
    virtual void set_size(int width, int height) = 0;
    virtual void draw_rgb(int, int, unsigned char *, size_t, int, int) = 0;
    virtual void draw_rgba(int, int, unsigned char *, size_t, int, int) = 0;
    virtual void update_area(int x, int y, int width, int height) const = 0;
  };

  /* Visual gadget.  */
  class gadget
  {
  public:
    class point
    {
    public:
      int x, y;

    public:
      point(int u, int v)
	: x(u), y(v) {}

      template <class T>
      point(const T &w)
	: x(w.x), y(w.y) {}
    };

    /* Shape of gadgets.  */
    class shape
    {
    private:
      vector<point> vertexes;

    public:
      template <class InputIterator>
      shape(InputIterator first, InputIterator last)
	: vertexes(first, last) {}

    public:
      bool inside(int x, int y) const;
    };

  private:
    int width_a, height_a;
    shape *shape_a;

  public:
    gadget(int w, int h);
    virtual ~gadget();

  public:
    int width() const
    {return width_a;}
    int height() const
    {return height_a;}
    bool inside(int, int) const;
    void set_shape(shape *);

  public:
    virtual void draw(frame *, int, int, int) = 0;
  };

  /* Gadget manager.  */
  class gadget_manager
  {
  private:
    /* Map of gadgets managed by this object.  */
    map<unsigned int, gadget *> gadgets;

  public:
    ~gadget_manager();

  public:
    /* FIXME: This method is a design mistake as file name shoud not
       be used here.  */
    template <class InputIterator>
    void load_image(unsigned int, const char *, int w, int h,
		    InputIterator first, InputIterator last);
  };

  class sound;

  /* Sound manager.  */
  class sound_manager
  {
  private:
    map<unsigned int, sound *> sounds;

  public:
    ~sound_manager();
  };

  /* View.  */
  class view
  {
  protected:
    struct layer
    {
      gadget *object;
      int x, y;
      int v;
      bool mapped;
      interp::block event_bindings[EVENT_MAX];
    };

  private:
    frame *_main_frame;

    /* Gadget manager of this object.  */
    gadget_manager gm;

    /* Map of managed objects.  These objects must be deleted at an
       appropriate time.  */
    map<unsigned int, gadget *> managed_objects;

    /* Sequence of layers.  */
    layer layers[NLAYERS];

    reverse_iterator<layer *> on_layer;

    int min_modified_x, max_modified_x;
    int min_modified_y, max_modified_y;

  public:
    view();
    virtual ~view();

  public:
    /* Returns the current main frame.  */
    frame *main_frame() const
      {return _main_frame;}

    /* Sets the main frame to frame F.  */
    void set_main_frame(frame *f);

  public:
    /* Clears the frame.  */
    void clear_frame();

    /* Assigns KEY to OBJECT for later use.  */
    void assign(unsigned int key, gadget *object);

    /* Sets object KEY's shape to SH.  */
    void set_shape(unsigned int key, gadget::shape *sh);

  public:
    /* Puts object on layer.  */
    void put(unsigned int, unsigned int, int, int, int, bool);

    /* Sets the state of layer.  */
    void select(unsigned int, int);

    /* Sets visibility of layer.  */
    void set_visible(int, bool);

    /* Binds event handler.  */
    void bind(unsigned int, event_type, const interp::block &);

    /* Notifies mouse motion.  */
    void notify_motion(int, int);

    /* Notifies button press.  */
    void press_button(int, int, int);

  public:
    bool modified() const
      {return (min_modified_x != max_modified_x
	       || min_modified_y != max_modified_y);}
    void clear_modified_region()
      {max_modified_x = min_modified_x; max_modified_y = min_modified_y;}
    void add_modified_region(int, int, int, int);

    /* Refreshes image.  */
    void refresh();

  public:
    virtual void tick(unsigned long t) {}
    virtual void advance() {}
  };

  /* Function templates.  */

  template <class InputIterator>
  void
  gadget_manager::load_image(unsigned int key, const char *name,
			     int width, int height,
			     InputIterator first, InputIterator last)
  {
    // FIXME: Write code.
  }
} // hyperplay

#endif /* not _HYPERPLAY_VIEW_H */
