/* -*-C++-*- */
/* Player base objects for Hyperplay
   Copyright (C) 1996, 1997, 1999 Hypercore Software Design, Ltd.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.  */

#ifndef _HYPERPLAY_PLAYER_H
#define _HYPERPLAY_PLAYER_H 1

#include <hyperplay/document.h>
#include <hyperplay/script.h>
#include <hyperplay/timer.h>
#include <hyperplay/view.h>
#include <string>

namespace hyperplay
{
  using namespace std;

#ifdef HPLAY_API
  class HPLAY_API player;
  template <class Frame> struct HPLAY_API player_fns;
#endif /* HPLAY_API */

  // Common base class of players.
  class player
    : public view,
      public virtual interp::command_data
  {
  public:
    typedef unsigned long scene_name_type;

    enum player_state {STOPPED, PLAYING};

  protected:
    document d;
    script *s;

  private:
    world w;
    // Interpreter used to execute code in the script.
    interpreter *_interp;
    timer event_timer;

    player_state _state;
    scene_name_type current_scene_name;
    scene current_scene_;
    // True if the next scene is available.  Used in the main loop.
    // Must be set to false on initialization.
    bool continued_;
    scene_name_type next_scene_name;

  public:
    player();
    ~player();

  public:
    /* Sets search path for script files.  */
    void set_script_path(const char *path)
    {file_script::set_search_path(path);}

    string data_prefix () const;

    /* Opens a document file FILE_NAME.  */
    void open(const char *file_name);
    void close();

    void save (unsigned int = 0);
    void load (unsigned int = 0);
    scene_name_type current_scene() const
      {return current_scene_name;}
    void next_scene (scene_name_type);
    void scene_call (const chunk::key_type &,
		     const interpreter &,
		     interp::context &) const;
    void scene_xcall (scene_name_type,
		      const chunk::key_type &,
		      const interpreter &,
		      interp::context &) const;

    void advance();

    long var(unsigned long domain, unsigned long name) const
      {return w.var(domain, name);}
    void set_var(unsigned long domain, unsigned long name, long value)
      {w.set_var(domain, name, value);}

    void schedule(unsigned long t, const interp::block &b)
      {event_timer.schedule(t, b);}
    void sync_time()
      {event_timer.sync();}

    void play();

  public:			// Frame interface
    void tick(unsigned long t)
      {event_timer.tick(t);}

  protected:
    bool continued () const
      {return continued_;}
    void change_scene ();
    interpreter *create_interpreter();
    interpreter *interp() const
      {return _interp;}
    void set_interp(interpreter *);
    void reset_timer()
      {event_timer.reset();}
  };
} // hyperplay

#endif /* not _HYPERPLAY_PLAYER_H */
