/* -*-C++-*- */
/* Interpreter objects for Hyperplay
   Copyright (C) 1996, 1997, 1999 Hypercore Software Design, Ltd.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.  */

#ifndef _HYPERPLAY_INTERP_H
#define _HYPERPLAY_INTERP_H 1

#include <hyperplay/types.h>
#include <hyperplay/defs.h>
#include <map>
#include <vector>
#include <string>
#include <stdexcept>

#if __WATCOMC__ > 1000
# pragma pack (push, 4)
#else
# pragma pack (4)
#endif

namespace hyperplay
{
  namespace interp
  {
    using namespace std;

#ifdef HPLAY_API
    class HPLAY_API command;
    class HPLAY_API interpreter;
#endif

    struct command_data
    {
      virtual ~command_data() {}
    };

    class interpreter;

    /* Context of execution.  */
    class context
    {
      friend class interpreter;
    public:
      vector<int> n_stack;
      vector<unsigned long> o_stack;
    private:
      const interpreter *_interp;
    protected:
      context(const interpreter *i)
	: _interp(i) {o_stack.push_back(0);}
    public:
      const interpreter *interp() {return _interp;}
      const byte_type *execute(const byte_type *, const byte_type *);
      void abort(const string &);
    };

    // Base class of commands.  This is an abstract class.
    class command
      : public virtual command_data
    {
    public:
      virtual const byte_type *operator()(const byte_type *,
					  const byte_type *,
					  context &,
					  const interpreter &) const = 0;
      virtual ~command () {}
    };

    // Class of interpreters.
    class interpreter
    {
    public:
      struct error
	: std::runtime_error
      {
	error (): std::runtime_error ("interpreter error") {}
	error (const string &str): std::runtime_error (str) {}
      };

      typedef const byte_type *(*command_handler)(const byte_type *,
						  const byte_type *,
						  context &, command_data *);

    protected:
      static const byte_type *glue_command(const byte_type *,
					   const byte_type *,
					   context &, command_data *);

    private:
      pair<command_handler, command_data *> commands[256];
      map<string, pair<command_handler, command_data *> > named_commands;
      vector<command *> command_objects;

    public:
      interpreter ();
      virtual ~interpreter();

    public:
      void add_command(const string &, command_handler, command_data *);
      void add_command(const string &, command *);
      virtual void configure(const byte_type *, const byte_type *);

      const byte_type *dispatch(int, const byte_type *, const byte_type *,
				context &) const;

      context *create_context() const;

      void undefine (int);
      // template <class InputIterator, class T>
      // static void unpack (InputIterator &, T &);
      static void unpack (const byte_type *&, unsigned long &);

    protected:
      void bind(int, const string &);
    };

    /* Block of executable code.  */
    class block
    {
    protected:
      vector<byte_type> code;	// FIXME
      const interpreter *i;
    public:
      void eval() const
	{
	  context *c = i->create_context();
	  c->execute(code.begin(), code.end());
	  delete c;
	}
    public:
      block()
	: i(NULL) {}
      block(const byte_type *first, const byte_type *last,
	    const interpreter *interp)
	: code(first, last), i(interp) {}
    };

    // Trivial class that just call a method without arguments.
    template <class T, class M>
      class hplay_trivial_command: public command
    {
    public:
      T *target;
      M fun;
      hplay_trivial_command (T &t, M f): target (&t), fun (f) {}
      const byte_type *operator() (const byte_type *, const byte_type *last,
				   context &,
				   const interpreter &) const
	{
	  (target->*fun) ();
	  return last;
	}
    };

    // Make a trivial command.
    template <class T, class M>
      inline hplay_trivial_command <T, M> *
      hplay_make_command (T &t, M mf)
      {
	return new hplay_trivial_command <T, M> (t, mf);
      }
  } // interp
} // hyperplay

#if __WATCOMC__ > 1000
# pragma pack (pop)
#else
# pragma pack ()
#endif

#endif /* not _HYPERPLAY_INTERP_H */

