/* -*-C++-*- */
/* File access objects for Hyperplay
   Copyright (C) 1996, 2000 Hypercore Software Design, Ltd.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.  */

#ifndef _HYPERPLAY_FILE_H
#define _HYPERPLAY_FILE_H 1

#include <hyperplay/defs.h>
#include <hyperplay/script.h>
#include <vector>
#include <fstream>
#include <string>

namespace hyperplay
{
  using namespace std;

#ifdef HPLAY_API
  class HPLAY_API chunk;
  class HPLAY_API data_file;
#endif /* HPLAY_API */

  // Class `data_file'

  // A `data_file' is the interface to a Hyperplay data file.  A data
  // file contains a finite number of chunks.

  class data_file
  {
    data_file (const data_file &); // left undefined
    data_file &operator= (const data_file &); // left undefined

  // chunk
  protected:
    typedef vector <chunk> toc;
    toc chunks;
  public:
    typedef toc::size_type size_type;
    typedef toc::iterator iterator;
    typedef toc::const_iterator const_iterator;
    size_type size () const;
    iterator begin ();
    const_iterator begin () const;
    iterator end ();
    const_iterator end () const;

    // stream
  protected:
    string fname;
    fstream *fs;
  public:
    void open (const string &);
    void close ();
  protected:
    void read (char *, size_t) const;
    void advance (size_t) const;
    void extract (back_insert_iterator <vector <chunk> >);

  public:
    data_file ();
    data_file (const string &);
    ~data_file ();
  public:
    // template <class InputIterator, class Result)
    // static void unpack16 (InputIterator &, Result &);
    static void unpack16 (const char *&i, int &result)
      {
	result = (unsigned char) *i++;
	result = result << 8 | (unsigned char) *i++;
      }
    static void unpack16 (const char *&i, unsigned int &result)
      {
	result = (unsigned char) *i++;
	result = result << 8 | (unsigned char) *i++;
      }
    // template <class InputIterator, class Result)
    // static void unpack32 (InputIterator &, Result &);
    static void unpack32 (const char *&i, unsigned long &result)
      {
	result = (unsigned char) *i++;
	result = result << 8 | (unsigned char) *i++;
	result = result << 8 | (unsigned char) *i++;
	result = result << 8 | (unsigned char) *i++;
      }
  };				// data_file

  inline bool
    operator!= (const chunk::key_type &x,
		const chunk::key_type &y)
  {
    return !(x == y);
  }

  /* File-based script.  */
  class file_script
    : public script
  {
  protected:
    struct scene_token
    {
      data_file::const_iterator f;
      data_file::const_iterator l;

      scene_token() {}
      scene_token(data_file::const_iterator first,
		  data_file::const_iterator last)
	: f(first), l(last) {}

      scene apply(const data_file &file) const
      {return scene(f, l, &file);}
    };
    typedef map<unsigned long, scene_token, less<unsigned long> > scene_map;

    class cursor_iterator
      : public forward_iterator<cursor, ptrdiff_t>
    {
      friend class file_script;
      friend bool operator==(const cursor_iterator &a,
			     const cursor_iterator &b)
	{return a.i == b.i;}

    protected:
      data_file::const_iterator i;
      data_file::const_iterator end;

    public:
      cursor operator*() const;
      cursor_iterator &operator++();
      cursor_iterator operator++(int)
	{cursor_iterator tmp = *this; ++*this; return tmp;}

    protected:
      cursor_iterator(data_file::const_iterator,
		      data_file::const_iterator);
    };

  private:
    /* Search path for script files.  */
    static vector<string> search_path_a;

  public:
    static void set_search_path(const char *path);

  protected:
    static void make_index(data_file::const_iterator,
			   data_file::const_iterator,
			   scene_map &);

  private:
    string data_dir_a;
    mutable data_file file;
    scene_map scenes;

  public:
    file_script(const char *script_name);
    ~file_script();

  public:
    /* Returns the data directory for this script.  */
    const char *data_dir() const
    {return data_dir_a.c_str();}

  protected:
    void open(const char *script_name);

  public:
    cursor_iterator begin_cursor() const
      {return cursor_iterator(file.begin(), file.end());}
    cursor_iterator end_cursor() const
      {return cursor_iterator(file.end(), file.end());}

  public:
    void configure(interpreter *) const;
    scene find_scene(unsigned long) const;
  };
} // hyperplay

#endif /* not _HYPERPLAY_FILE_H */

