/* main.cc - sample main program for Hyperplay/X11.
   Copyright (C) 1997, 1999 Hypercore Software Design, Ltd.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.  */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#undef const

#include "getopt.h"
#include <hyperplay/gtk.h>
#include <hyperplay/player.h>
#include <gtk/gtk.h>
#include <libintl.h>
#include <string>
#include <cstdio>

#define _(MSG) gettext(MSG)

using namespace hyperplay;
using namespace std;

namespace
{
  /* Value of `--script-path' option.  */
  string opt_script_path("");

  int opt_help = false;
  int opt_version = false;

  const struct option longopts[] =
  {
    {"script-path", required_argument, NULL, 'S'},
    {"help", no_argument, &opt_help, true},
    {"version", no_argument, &opt_version, true},
    {NULL, 0, NULL, 0}
  };

  /* Parses optional program arguments.  This function returns true if
     all options are valid.  */
  bool
  parse_options(int argc, char **argv)
  {
    for (;;)
      {
	int index;
	int opt = getopt_long(argc, argv, "S:", longopts, &index);
	if (opt == -1)		// no more options
	  break;

	switch (opt)
	  {
	  case 'S':
	    opt_script_path = optarg;
	    break;

	  case '?':		// unknown option
	    return false;

	  case 0:		// long option
	    break;

	  default:
	    // logic error
	    abort();
	  }
      }

    return true;
  }

  /* Displays a help to the standard output.  ARG0 is used as the
     program name.  */
  void
  display_help(const char *arg0)
  {
    printf(_("Usage: %s [OPTION]... SAVE-FILE\n"), arg0);
    printf(_("Play a Hyperplay script.\n"));
    printf("\n");
    printf(_("  -S, --script-path=PATH search PATH for script files\n"));
    printf(_("      --help            display this help and exit\n"));
    printf(_("      --version         output version information and exit\n"));
    printf("\n");
    printf(_("Report bugs to <hyperplay@lists.hypercore.co.jp>.\n"));
  }
} // (unnamed namespace)

namespace
{
  /* Player application.  */
  class player_app
  {
  private:
    player p;
    gtk::gtk_frame f;

  public:
    player_app(const char *);

  public:
    void run();
    GtkWidget *create_window();
  };
} // (unnamed namespace)

void
player_app::run()
{
  p.play();

  gtk_main();
}

GtkWidget *
player_app::create_window()
{
  GtkAccelGroup *ag1 = gtk_accel_group_new();

  GtkWidget *main_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(main_window), "Hyperplay");
  gtk_window_add_accel_group(GTK_WINDOW(main_window), ag1);
  gtk_signal_connect(GTK_OBJECT(main_window), "delete_event",
		     GTK_SIGNAL_FUNC(gtk_main_quit), this);

  {
    GtkWidget *vbox1 = gtk_vbox_new(false, 0);
    gtk_container_add(GTK_CONTAINER(main_window), vbox1);
    gtk_widget_show(vbox1);

    {
      GtkItemFactory *if1 = gtk_item_factory_new(GTK_TYPE_MENU_BAR,
						 "<Main>", ag1);
      {
	GtkItemFactoryEntry if_entries[] =
	{
	  {_("/_File/_Open..."), NULL, NULL, 0, ""},
	  {_("/_File/_Save"), NULL, NULL, 0, ""},
	  {_("/_File/Save _As..."), NULL, NULL, 0, ""},
	  {_("/_File/"), NULL, NULL, 0, "<Separator>"},
	  {_("/_File/E_xit"), NULL,
	   reinterpret_cast<GtkItemFactoryCallback>(&gtk_main_quit), 1, ""},
	  {_("/_Help/_About..."), NULL, NULL, 0, ""}
	};
	gtk_item_factory_create_items(if1,
				      sizeof if_entries / sizeof if_entries[0],
				      if_entries, this);
	gtk_box_pack_start(GTK_BOX(vbox1), if1->widget, false, false, 0);
	gtk_widget_show(if1->widget);
      }

      //gtk_object_unref(GTK_OBJECT(if1));

      GtkWidget *pw = f.create_widget();
      gtk_box_pack_start(GTK_BOX(vbox1), pw, true, true, 0);
      gtk_widget_show(pw);
    }
  }

  gtk_accel_group_unref(ag1);

  return main_window;
}

player_app::player_app(const char *file_name)
  : f(&p, 640, 400)
{
  p.set_main_frame(&f);

  if (!opt_script_path.empty())
    p.set_script_path(opt_script_path.c_str());

  p.open(file_name);
}

/* Main function.  */
int
main(int argc, char **argv)
{
  gtk_set_locale();
  gtk_init(&argc, &argv);
  gdk_rgb_init();

#ifdef LOCALEDIR
  bindtextdomain(PACKAGE, LOCALEDIR);
#endif
  textdomain(PACKAGE);

  if (!parse_options(argc, argv))
    {
      fprintf(stderr, _("Try `%s --help' for more information.\n"), argv[0]);
      return EXIT_FAILURE;
    }

  if (opt_version)
    {
      printf("%s %s\n", PACKAGE, VERSION);
      return EXIT_SUCCESS;
    }

  if (opt_help)
    {
      display_help(argv[0]);
      return EXIT_SUCCESS;
    }

  if (argc < optind + 1)
    {
      fprintf(stderr, _("%s: missing file argument\n"), argv[0]);
      fprintf(stderr, _("Try `%s --help' for more information.\n"), argv[0]);
      return EXIT_FAILURE;
    }

  if (argc > optind + 1)
    {
      fprintf(stderr, _("%s: too many arguments\n"), argv[0]);
      fprintf(stderr, _("Try `%s --help' for more information.\n"), argv[0]);
      return EXIT_FAILURE;
    }

  try
    {
      player_app app(argv[optind]);

      GtkWidget *toplevel = app.create_window();
      gtk_widget_show(toplevel);

      app.run();
    }
  catch (exception &x)
    {
      fprintf(stderr, _("%s: unhandled exception: %s\n"), argv[0], x.what());
      abort();
    }
  catch (...)
    {
      fprintf(stderr, _("%s: unhandled exception\n"), argv[0]);
      abort();
    }

  return EXIT_SUCCESS;
}

