/*
 * glChess - A 3D chess interface
 *
 * Copyright (C) 2001  Robert Ancell <bob27@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <GL/gl.h>

#include "global.h"
#include "texture.h"
#include "image.h"

/*
 * Loads a .rgb image and makes an openGL texture out of it 
 */
GLuint load_texture(char *file_name, int is_one_colour)
{
  IMAGE *texture_image = ImageLoad(file_name);
  GLint texture_name;
  GLubyte *texture_data;
  GLenum data_type;
  int x, y, pad = 0;

  if (texture_image == NULL)
  {
    fprintf(stderr, "Error loading texture :%s\n", file_name);
    return 0;			/*
				 * note: this is 0 _not_ NULL 
				 */
  }

  if (is_one_colour)
  {
    texture_data =
	malloc(sizeof(GLubyte) * texture_image->sizeX *
	       texture_image->sizeY);

    for (x = 0; x < texture_image->sizeX; x++)
    {
      for (y = 0; y < texture_image->sizeY; y++)
      {
	/*
	 * Get the red component 
	 */
	texture_data[(x + y * texture_image->sizeX)]
	    = texture_image->data[(x + texture_image->sizeX * y)
				  * texture_image->sizeZ + pad];
      }
      pad += (texture_image->sizeX * texture_image->sizeZ) % 4;
    }
  } else
  {
    /*
     * Allocate space for the texture data 
     */
    /*
     * FIXME: should really be able to be accessed later 
     */
    texture_data = malloc(sizeof(GLubyte)
			  * texture_image->sizeX
			  * texture_image->sizeY * texture_image->sizeZ);

    /*
     * Copy the data (from the image) to a new array 
     */
    for (x = 0; x < texture_image->sizeX; x++)
    {
      for (y = 0; y < texture_image->sizeY; y++)
      {
	texture_data[(x + y * texture_image->sizeX) *
		     texture_image->sizeZ] =
	    texture_image->data[(x + texture_image->sizeX * y) *
				texture_image->sizeZ + pad];
	texture_data[(x + y * texture_image->sizeX) *
		     texture_image->sizeZ + 1] =
	    texture_image->data[(x + texture_image->sizeX * y) *
				texture_image->sizeZ + 1 + pad];
	texture_data[(x + y * texture_image->sizeX) *
		     texture_image->sizeZ + 2] =
	    texture_image->data[(x + texture_image->sizeX * y) *
				texture_image->sizeZ + 2 + pad];
	if (texture_image->sizeZ > 3)
	  texture_data[(x + y * texture_image->sizeX) *
		       texture_image->sizeZ + 3] =
	      texture_image->data[(x + texture_image->sizeX * y)
				  * texture_image->sizeZ + 3 + pad];
      }
      pad += (texture_image->sizeX * texture_image->sizeZ) % 4;
    }
  }

  /*
   * Generate the texture 
   */
  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
  glGenTextures(1, (GLuint *) & texture_name);
  glBindTexture(GL_TEXTURE_2D, texture_name);

  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

  if (is_one_colour)
  {
    data_type = GL_LUMINANCE;
    texture_image->sizeZ = 1;
  } else if (texture_image->sizeZ > 3)
    data_type = GL_RGBA;
  else
    data_type = GL_RGB;
  glTexImage2D(GL_TEXTURE_2D,
	       0,
	       texture_image->sizeZ,
	       texture_image->sizeX,
	       texture_image->sizeY,
	       0, data_type, GL_UNSIGNED_BYTE, texture_data);

  /*
   * The data is copied so remove the local version 
   */
  free(texture_data);

  /*
   * Free the image 
   */
  free(texture_image->data);
  free(texture_image);

  return texture_name;
}

/*
 * Load all the games textures 
 */
void load_textures(Game * game)
{
  load_player_textures(game->white_player);
  load_player_textures(game->black_player);
  load_number_textures(game);
  load_letter_textures(game);
}

/*
 * Unloads all the game textures 
 */
void free_textures(Game * game)
{
  free_player_textures(game->white_player);
  free_player_textures(game->black_player);
}

/*
 * Reloads all the games textures 
 */
void reload_textures(Game * game)
{
  reload_player_textures(game->white_player);
  reload_player_textures(game->black_player);
  reload_number_textures(game);
  reload_letter_textures(game);
}

/*
 * Load the textures for a player 
 */
void load_player_textures(Player * player)
{
  char file_name[100];

  snprintf(file_name, 100, "%s%s", player->texture_dir,
	   player->piece_texture_file);
  player->piece_texture = load_texture(file_name, FALSE);
  snprintf(file_name, 100, "%s%s", player->texture_dir,
	   player->selected_piece_texture_file);
  player->selected_piece_texture = load_texture(file_name, FALSE);
  snprintf(file_name, 100, "%s%s", player->texture_dir,
	   player->board_texture_file);
  player->board_texture = load_texture(file_name, FALSE);
}

/*
 * Frees the textures for a player 
 */
void free_player_textures(Player * player)
{
  glDeleteTextures(1, &player->piece_texture);
  glDeleteTextures(1, &player->selected_piece_texture);
  glDeleteTextures(1, &player->board_texture);
}

/*
 * Reloads a players textures 
 */
void reload_player_textures(Player * player)
{
  free_player_textures(player);
  load_player_textures(player);
}

/* Loads the numbers */
void load_number_textures(Game * game)
{
  char file_name[100];
  int i;

  /* FIXME: this is temporarily static */
  for (i = 0; i < 8; i++)
  {
    snprintf(file_name, 100, "%s%c.rgb", game->texture_dir, '1' + i);
    game->number_textures[i] = load_texture(file_name, TRUE);
  }
}

/*
 * Frees the number textures 
 */
void free_number_textures(Game * game)
{
  glDeleteTextures(8, game->number_textures);
}

/*
 * Reload the numbers 
 */
void reload_number_textures(Game * game)
{
  free_number_textures(game);
  load_number_textures(game);
}

/*
 * Loads the letters 
 */
void load_letter_textures(Game * game)
{
  char file_name[100];
  int i;

  /*
   * FIXME: this is temporarily static 
   */
  for (i = 'a'; i <= 'h'; i++)
  {
    snprintf(file_name, 100, "%s%c.rgb", game->texture_dir, (char) i);
    game->letter_textures[i - 'a'] = load_texture(file_name, TRUE);
  }
}

/*
 * Frees the letter textures 
 */
void free_letter_textures(Game * game)
{
  glDeleteTextures(8, game->letter_textures);
}

/*
 * Reload the letters 
 */
void reload_letter_textures(Game * game)
{
  free_letter_textures(game);
  load_letter_textures(game);
}
