/*
 * glChess - A 3D chess interface
 *
 * Copyright (C) 2001  Robert  Ancell <bob27@users.sourceforge.net>
 *                     Michael Duelli <duelli@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * splash.c - splash screen of spinning chess pieces 
 */

#include <stdio.h>
#include <gtk/gtk.h>
#include <gtkgl/gtkglarea.h>
#include <GL/gl.h>

#include "global.h"
#include "splash.h"
#include "interface.h"
#include "draw.h"

extern float cog_height[6];

#define ROT_SPEED 30
#define PIECE_DIST 20.0
#define CHANGE_TIME 5.0

static gint idle_tag = -1;
float splash_rot = 0.0;
int splash_piece, is_cycle = 1;
GTimer *timer = NULL;

/*
 * Start a timer 
 */
void init_splash(glChessWidget * glcwidget)
{
  GLfloat zero_pos[] = { 0.0, 0.0, 0.0 };

  timer = g_timer_new();

  gtk_gl_area_make_current(GTK_GL_AREA(glcwidget->glarea));
  idle_tag = gtk_idle_add(display_splash, (gpointer) glcwidget);

  /*
   * Reset the lights... 
   */
  glDisable(GL_LIGHT1);
  glLightfv(GL_LIGHT0, GL_POSITION, zero_pos);

  /*
   * Time meter not used in splash screen 
   */
  gtk_label_set_text(GTK_LABEL(glcwidget->time_label), "");
  /*
   * Player label not used in splash screen 
   */
  gtk_label_set_text(GTK_LABEL(glcwidget->player_label), "");

  /*
   * Set up the texture if set 
   */
  if (glcwidget->game->is_texture)
  {
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
    glEnable(GL_TEXTURE_2D);
  }

  g_timer_start(timer);
}

/* Tie up some loose ends */
void end_splash(GtkWidget * widget, glChessWidget * glcwidget)
{
  /* Quit if not on game screen */
  if (idle_tag == -1)
    return;

  /* Don't try to refresh anymore */
  gtk_idle_remove(idle_tag);
  idle_tag = -1;

  /* Get rid of the timer */
  g_timer_destroy(timer);
}

/*
 * Draw the splash screen 
 */
gint display_splash(gpointer widget)
{
  glChessWidget *glcwidget = (glChessWidget *) widget;
  Game *game = glcwidget->game;
  Player *player = game->current_player;
  GLfloat white_texture_colour[4] = { 1.0, 1.0, 1.0, 1.0 };
  static int nframes = 0;

  /*
   * Do animations, if returns false, not redrawing that fast so skip
   * frame, otherwise draw frame if can make area current. Also draw frame
   * if redisplay is set 
   */
  if ((!animate_splash(glcwidget, &nframes) && !game->redisplay) ||
      (!gtk_gl_area_make_current(GTK_GL_AREA(glcwidget->glarea))))
    return TRUE;

  /*
   * Clear the buffers 
   */
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  /*
   * Reset the transformation matrix 
   */
  glLoadIdentity();

  /*
   * Translate away from camera 
   */
  glTranslatef(0.0f, 0.0f, -PIECE_DIST);

  /*
   * Rotate piece 
   */
  /*
   * FIXME: this *should* be able to be tidied up to how it used to be 
   */
  glRotatef(splash_rot, 0.0f, 1.0f, 0.0f);
  glRotatef(180.0f + splash_rot, 0.0f, 0.0f, -1.0f);
  glRotatef(splash_rot, 0.0f, 1.0f, 0.0f);
  glRotatef(90.0f, 1.0f, 0.0f, 0.0f);
  glRotatef(180.0f, 0.0f, 1.0f, 0.0f);

  /*
   * Move center of rotation 
   */
  glTranslatef(0.0, -cog_height[splash_piece - 1], 0.0);

  /*
   * Set the colour/material/texture 
   */
  glColor4fv(player->piece_colour);
  if (game->is_texture)
  {
    glMaterialfv(GL_FRONT, GL_DIFFUSE, white_texture_colour);
    glBindTexture(GL_TEXTURE_2D, player->piece_texture);
  } else
  {
    /*
     * FIXME: this _shouldn't be required -- the properties dialog seems
     * to be affecting the enabled state of things... 
     */
    glDisable(GL_TEXTURE_2D);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, player->piece_colour);
  }
  /*
   * Draw the piece 
   */
  glCallList(splash_piece);

  /*
   * Swap the buffers 
   */
  gtk_gl_area_swapbuffers(GTK_GL_AREA(glcwidget->glarea));

  /*
   * Finished one frame 
   */
  nframes++;

  /*
   * Don't redraw again 
   */
  game->redisplay = FALSE;

  return TRUE;
}

/*
 * Animates the pieces in the splash screen 
 */
gint animate_splash(glChessWidget * glcwidget, int *nframes)
#define FPS_UPDATE 1
{
  static gdouble dt;
  static gdouble elapsed = 0.0;

  /*
   * Get the change in time, dt 
   */
  dt += g_timer_elapsed(timer, NULL);
  g_timer_reset(timer);

  /*
   * No timestep when paused 
   */
  if (glcwidget->game->is_paused)
    dt = -0.00000001f;

  /*
   * Wait until the minimum dt has happened until drawing frame 
   */
  if (dt < glcwidget->game->min_dt)
    return FALSE;

  /*
   * Total elapsed time 
   */
  elapsed += dt;

  /*
   * Update the fps meter every FPS_UPDATE 
   */
  if ((int) (elapsed / FPS_UPDATE) != (int) (elapsed - dt / FPS_UPDATE))
  {
    char fps_string[10];

    snprintf(fps_string, 10, "%d fps", *nframes);
    *nframes = 0;
    gtk_label_set_text(GTK_LABEL(glcwidget->fps_label), fps_string);
  }

  /*
   * Change piece every CHANGE_TIME 
   */
  if (is_cycle)
    splash_piece = 6 - (int) (elapsed / CHANGE_TIME) % 6;

  /*
   * Rotate the piece 
   */
  splash_rot += ROT_SPEED * dt;

  /*
   * Frame completed so reset dt 
   */
  dt = 0.0f;

  return TRUE;
}
