/*
 * glChess - A 3D chess interface
 *
 * Copyright (C) 2001  Robert  Ancell <bobg27@users.sourceforge.net>
 *                     Michael Duelli <duelli@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* san.c - converts SAN (Standard Algebraic Notation) to internal notation */

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "global.h"
#include "engine.h"
#include "san.h"

extern int debug;
extern int promotion;

/* If it's a castling move, it returns 1 or 2 */
int san_check_castling(char *san_str)
{
  if (san_str[1] == '-')
  {
    if (san_str[3] != '-')
      return 1;			/* short: i.e. O-O   => e1g1 */
    else
      return 2;			/* long: i.e. O-O-O => e1c1 */
  }

  return 0;			/* No castling move */
}

/* Tell me which piece is moved */
int san_extract_piece(char *san_str)
{
  switch (san_str[0])
  {
  case 'K':
    return KING;
  case 'Q':
    return QUEEN;
  case 'B':
    return BISHOP;
  case 'N':
    return KNIGHT;
  case 'R':
    return ROOK;
  default:
    return PAWN;
  }
}

/* FIXME: May be we shouldn't ignore the mate '#' char */
/* Gets rid of all the 'x', '+', '=' and '#' chars */
void san_simplify(char *san_str)
{
  char *s = san_str;
  char *t = san_str;

  while (*s != '\n')
  {
    if (*s == 'x' || *s == '+' || *s == '#' || *s == '=')
    {
      if (*s == '=')
      {
	promotion = san_extract_piece(++s);
	if (promotion == PAWN)
	  promotion = -1;
      }

      s++;
    } else
      *t++ = *s++;
  }
  *t = '\0';
}

#define TAR_X target[0]		/* Abbreviations */
#define TAR_Y target[1]
#define VALID(a) ((a >= 0) && (a <= 7))	/* Check if the coordinate is valid */

/* Convert SAN to coordinated */
char *san_to_coordinated(char *san_str, int board[8][8])
{
  int piece;
  int castling;
  int start[2];
  int target[2];
  char *move = malloc(sizeof(char) * 5);

  /* That's the easiest so let's do it first */
  if ((castling = san_check_castling(san_str)) != 0)
  {
    if (castling == 1)
    {				/* short castling */
      snprintf(move, 5, "e8g8");
    } else
    {				/* long castling */

      snprintf(move, 5, "e8c8");
    }
    return strdup(move);
  }

  san_simplify(san_str);

  TAR_X = san_str[strlen(san_str) - 2] - 97;
  TAR_Y = san_str[strlen(san_str) - 1] - 49;

  start[0] = start[1] = -1;

  /* If we already got a row or a col we don't need to search elsewhere */
  if (strlen(san_str) == 4)
  {				/* e.g. N4c3, Rad1 */
    if (isdigit(san_str[1]))
      start[1] = san_str[1] - 49;
    else if (isalpha(san_str[1]))
      start[0] = san_str[1] - 97;
  }

  piece = san_extract_piece(san_str);

  if (start[0] != -1 || start[1] != -1)
  {
    int i = 0;

    /* If we got one start coordinate we can simply find the other */
    if (start[0] != -1)
    {
      /* Check the start column */
      while (i++ <= 7)
	if (board[start[0]][i] == BLACK piece)
	  break;

      start[1] = i;
    } else
    {
      /* Check the row */
      while (i++ <= 7)
	if (board[i][start[1]] == BLACK piece)
	  break;

      start[0] = i;
    }
    snprintf(move, 5, "%c%d%c%d",
	     start[0] + 'a', start[1] + 1, TAR_X + 'a', TAR_Y + 1);
  } else
  {
    int i = 0;
    int j = 0;

    switch (piece)
    {
    case KING:
      /* King can only move 1 so check the 8 fields surrounding the target */
      if (VALID(TAR_X - 1) && VALID(TAR_Y + 1)
	  && board[TAR_X - 1][TAR_Y + 1] == BLACK KING)
      {
	start[0] = TAR_X - 1;
	start[1] = TAR_Y + 1;
      } else if (VALID(TAR_X) && VALID(TAR_Y + 1)
		 && board[TAR_X][TAR_Y + 1] == BLACK KING)
      {
	start[0] = TAR_X;
	start[1] = TAR_Y + 1;
      } else if (VALID(TAR_X + 1) && VALID(TAR_Y + 1)
		 && board[TAR_X + 1][TAR_Y + 1] == BLACK KING)
      {
	start[0] = TAR_X + 1;
	start[1] = TAR_Y + 1;
      } else if (VALID(TAR_X + 1) && VALID(TAR_Y)
		 && board[TAR_X + 1][TAR_Y] == BLACK KING)
      {
	start[0] = TAR_X + 1;
	start[1] = TAR_Y;
      } else if (VALID(TAR_X + 1) && VALID(TAR_Y - 1)
		 && board[TAR_X + 1][TAR_Y - 1] == BLACK KING)
      {
	start[0] = TAR_X + 1;
	start[1] = TAR_Y - 1;
      } else if (VALID(TAR_X) && VALID(TAR_Y - 1)
		 && board[TAR_X][TAR_Y - 1] == BLACK KING)
      {
	start[0] = TAR_X;
	start[1] = TAR_Y - 1;
      } else if (VALID(TAR_X - 1) && VALID(TAR_Y - 1)
		 && board[TAR_X - 1][TAR_Y - 1] == BLACK KING)
      {
	start[0] = TAR_X - 1;
	start[1] = TAR_Y - 1;
      } else if (VALID(TAR_X - 1) && VALID(TAR_Y)
		 && board[TAR_X - 1][TAR_Y] == BLACK KING)
      {
	start[0] = TAR_X - 1;
	start[1] = TAR_Y;
      }

      snprintf(move, 5, "%c%d%c%d",
	       start[0] + 'a', start[1] + 1, TAR_X + 'a', TAR_Y + 1);
      break;
    case QUEEN:
      /* Check column, lower half */
      for (i = TAR_Y - 1; VALID(i) && board[TAR_X][i] == -1; i--)
	;

      if (VALID(i) && board[TAR_X][i] == BLACK QUEEN)
      {
	start[0] = TAR_X;
	start[1] = i;
      } else
      {
	/* Check column, upper half */
	for (i = TAR_Y + 1; VALID(i) && board[TAR_X][i] == -1; i++)
	  ;

	if (VALID(i) && board[TAR_X][i] == BLACK QUEEN)
	{
	  start[0] = TAR_X;
	  start[1] = i;
	}
      }

      /* If no result, check row */
      if (start[0] == -1)
      {
	/* Check row, left half */
	for (i = TAR_X - 1; VALID(i) && board[i][TAR_Y] == -1; i--)
	  ;

	if (VALID(i) && board[i][TAR_Y] == BLACK QUEEN)
	{
	  start[0] = i;
	  start[1] = TAR_Y;
	} else
	{
	  /* Check row, right half */
	  for (i = TAR_X + 1; VALID(i) && board[i][TAR_Y] == -1; i++)
	    ;

	  if (VALID(i) && board[i][TAR_Y] == BLACK QUEEN)
	  {
	    start[0] = i;
	    start[1] = TAR_Y;
	  }
	}
      }

      /* If there is still no result, check the diagonals */
      /* Check diagonal a-h, lower half */
      for (i = TAR_X - 1, j = TAR_Y - 1; VALID(i) && VALID(j)
	   && board[i][j] == -1; i--, j--)
	;

      if (VALID(i) && board[i][j] == BLACK QUEEN)
      {
	start[0] = i;
	start[1] = j;
      } else
      {
	/* Check diagonal a-h, upper half */
	for (i = TAR_X + 1, j = TAR_Y + 1; VALID(i) && VALID(j)
	     && board[i][j] == -1; i++, j++)
	  ;

	if (VALID(i) && board[i][j] == BLACK QUEEN)
	{
	  start[0] = i;
	  start[1] = j;
	}
      }

      if (start[0] == -1)
      {
	/* Check diagonal h-a, upper half */
	for (i = TAR_X - 1, j = TAR_Y + 1; VALID(i) && VALID(j)
	     && board[i][j] == -1; i--, j++)
	  ;

	if (VALID(i) && board[i][j] == BLACK QUEEN)
	{
	  start[0] = i;
	  start[1] = j;
	} else
	{
	  /* Check diagonal h-a, lower half */
	  for (i = TAR_X + 1, j = TAR_Y - 1; VALID(i) && VALID(j)
	       && board[i][j] == -1; i++, j--)
	    ;

	  if (VALID(i) && board[i][j] == BLACK QUEEN)
	  {
	    start[0] = i;
	    start[1] = j;
	  }
	}
      }

      snprintf(move, 5, "%c%d%c%d",
	       start[0] + 'a', start[1] + 1, TAR_X + 'a', TAR_Y + 1);
      break;
    case BISHOP:
      /* Check diagonal a-h, lower half */
      for (i = TAR_X - 1, j = TAR_Y - 1; VALID(i) && VALID(j)
	   && board[i][j] == -1; i--, j--)
	;

      if (VALID(i) && board[i][j] == BLACK BISHOP)
      {
	start[0] = i;
	start[1] = j;
      } else
      {
	/* Check diagonal a-h, upper half */
	for (i = TAR_X + 1, j = TAR_Y + 1; VALID(i) && VALID(j)
	     && board[i][j] == -1; i++, j++)
	  ;

	if (VALID(i) && board[i][j] == BLACK BISHOP)
	{
	  start[0] = i;
	  start[1] = j;
	}
      }

      if (start[0] == -1)
      {
	/* Check diagonal h-a, upper half */
	for (i = TAR_X - 1, j = TAR_Y + 1; VALID(i) && VALID(j)
	     && board[i][j] == -1; i--, j++)
	  ;

	if (VALID(i) && board[i][j] == BLACK BISHOP)
	{
	  start[0] = i;
	  start[1] = j;
	} else
	{
	  /* Check diagonal h-a, lower half */
	  for (i = TAR_X + 1, j = TAR_Y - 1; VALID(i) && VALID(j)
	       && board[i][j] == -1; i++, j--)
	    ;

	  if (VALID(i) && board[i][j] == BLACK BISHOP)
	  {
	    start[0] = i;
	    start[1] = j;
	  }
	}
      }

      snprintf(move, 5, "%c%d%c%d",
	       start[0] + 'a', start[1] + 1, TAR_X + 'a', TAR_Y + 1);
      break;
    case KNIGHT:
      /* Check the 8 fields the knight could have come from */
      if (VALID(TAR_X - 2) && VALID(TAR_Y + 1) &&
	  board[TAR_X - 2][TAR_Y + 1] == BLACK KNIGHT)
      {
	start[0] = TAR_X - 2;
	start[1] = TAR_Y + 1;
      }
	else if (VALID(TAR_X - 1) && VALID(TAR_Y + 2) &&
		 board[TAR_X - 1][TAR_Y + 2] == BLACK KNIGHT)
      {
	start[0] = TAR_X - 1;
	start[1] = TAR_Y + 2;
      }
	else if (VALID(TAR_X + 1) && VALID(TAR_Y + 2) &&
		 board[TAR_X + 1][TAR_Y + 2] == BLACK KNIGHT)
      {
	start[0] = TAR_X + 1;
	start[1] = TAR_Y + 2;
      }
	else if (VALID(TAR_X + 2) && VALID(TAR_Y + 1) &&
		 board[TAR_X + 2][TAR_Y + 1] == BLACK KNIGHT)
      {
	start[0] = TAR_X + 2;
	start[1] = TAR_Y + 1;
      }
	else if (VALID(TAR_X + 2) && VALID(TAR_Y - 1) &&
		 board[TAR_X + 2][TAR_Y - 1] == BLACK KNIGHT)
      {
	start[0] = TAR_X + 2;
	start[1] = TAR_Y - 1;
      }
	else if (VALID(TAR_X + 1) && VALID(TAR_Y - 2) &&
		 board[TAR_X + 1][TAR_Y - 2] == BLACK KNIGHT)
      {
	start[0] = TAR_X + 1;
	start[1] = TAR_Y - 2;
      }
	else if (VALID(TAR_X - 1) && VALID(TAR_Y - 2) &&
		 board[TAR_X - 1][TAR_Y - 2] == BLACK KNIGHT)
      {
	start[0] = TAR_X - 1;
	start[1] = TAR_Y - 2;
      }
	else if (VALID(TAR_X - 2) && VALID(TAR_Y - 1) &&
		 board[TAR_X - 2][TAR_Y - 1] == BLACK KNIGHT)
      {
	start[0] = TAR_X - 2;
	start[1] = TAR_Y - 1;
      }
      snprintf(move, 5, "%c%d%c%d",
	       start[0] + 'a', start[1] + 1, TAR_X + 'a', TAR_Y + 1);
      break;
    case ROOK:
      /* Check column, lower half */
      for (i = TAR_Y - 1; VALID(i) && board[TAR_X][i] == -1; i--)
	;

      if (VALID(i) && board[TAR_X][i] == BLACK ROOK)
      {
	start[0] = TAR_X;
	start[1] = i;
      } else
      {
	/* Check column, upper half */
	for (i = TAR_Y + 1; VALID(i) && board[TAR_X][i] == -1; i++)
	  ;

	if (VALID(i) && board[TAR_X][i] == BLACK ROOK)
	{
	  start[0] = TAR_X;
	  start[1] = i;
	}
      }

      /* If no result, check row */
      if (start[0] == -1)
      {
	/* Check row, left half */
	for (i = TAR_X - 1; VALID(i) && board[i][TAR_Y] == -1; i--)
	  ;

	if (VALID(i) && board[i][TAR_Y] == BLACK ROOK)
	{
	  start[0] = i;
	  start[1] = TAR_Y;
	} else
	{
	  /* Check row, right half */
	  for (i = TAR_X + 1; VALID(i) && board[i][TAR_Y] == -1; i++)
	    ;

	  if (VALID(i) && board[i][TAR_Y] == BLACK ROOK)
	  {
	    start[0] = i;
	    start[1] = TAR_Y;
	  }
	}
      }

      snprintf(move, 5, "%c%d%c%d",
	       start[0] + 'a', start[1] + 1, TAR_X + 'a', TAR_Y + 1);
      break;
    case PAWN:
      if (strlen(san_str) == 2)
      {				/* e.g. e4, c3, ...      */
	if (board[TAR_X][TAR_Y + 1] == BLACK PAWN)
	{
	  snprintf(move, 5, "%c%d%c%d",
		   san_str[0], TAR_Y + 1 + 1, san_str[0], TAR_Y + 1);	/* one field */
	} else
	{
	  snprintf(move, 5, "%c%d%c%d",
		   san_str[0], TAR_Y + 1 + 2, san_str[0], TAR_Y + 1);	/* two fields */
	}
      } else if (strlen(san_str) == 3)
      {				/* e.g. exd5, cxd4, ... */
	snprintf(move, 5, "%c%d%c%d",
		 san_str[0], TAR_Y + 1 + 1, san_str[1], TAR_Y + 1);
      }
      break;
    default:
      if (debug)
	debug_output("SAN_TO_COORDINATED: Couldn't convert move\n");

      break;
    }
  }
  if (debug)
    debug_output("SAN_TO_COORDINATED: move %s\n", move);

  return move;
}
