/*
 * glChess - A 3D chess interface
 *
 * Copyright (C) 2001  Robert Ancell <bob27@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * This file contains all the GTK code for the preferences dialog (there's
 * quite a lot of it) 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtkgl/gtkglarea.h>
#include <GL/glu.h>

#include "global.h"
#include "preferences.h"
#include "menu.h"
#include "draw.h"
#include "texture.h"
#include "game.h"
#include "player.h"
#include "models.h"

#include "engine.h"
#include "picksquare.h"

/*
 * Adds a text entry into a table 
 */
GtkWidget *add_text_entry(GtkWidget * table,
			  gchar * label_text, gchar * entry_text, gint loc)
{
  GtkWidget *label, *entry;

  label = gtk_label_new(label_text);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, loc, loc + 1);
  entry = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(entry), entry_text);
  gtk_table_attach_defaults(GTK_TABLE(table), entry, 1, 2, loc, loc + 1);

  return entry;
}

void toggle_value(GtkWidget * check_button, int *property)
{
  *property = GTK_TOGGLE_BUTTON(check_button)->active;
}

GtkWidget *add_check_button(PreferenceDialog * pdialog,
			    GtkWidget * table, gchar * label,
			    int *property, gint loc)
{
  GtkWidget *check_button;

  check_button = gtk_check_button_new_with_label(label);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_button), *property);
  gtk_signal_connect(GTK_OBJECT(check_button), "clicked",
		     GTK_SIGNAL_FUNC(toggle_value), property);
  gtk_table_attach_defaults(GTK_TABLE(table), check_button, 0, 2, loc,
			    loc + 1);

  return check_button;
}

/*
 * Copies the entries contents to the players data (non-GTK) 
 */
void store_file_name(GtkWidget * widget, char **name_loc)
{
  gchar *fname = gtk_entry_get_text(GTK_ENTRY(widget));

  free(*name_loc);
  *name_loc = malloc(sizeof(char) * (strlen(fname) + 1));
  strcpy(*name_loc, fname);
}

/*
 * Sets the chosen filename into the entry 
 */
void set_file_name(GtkWidget * widget, FileEntry * fentry)
{
  Player *player = fentry->page->player;
  gchar *fname =
      gtk_file_selection_get_filename(GTK_FILE_SELECTION(fentry->filesel));

  /*
   * If has the directory path on the front, strip it off 
   */
  if (!strncmp(player->texture_dir, fname, strlen(player->texture_dir)))
    fname = &fname[strlen(player->texture_dir)];

  gtk_entry_set_text(GTK_ENTRY(fentry->entry), fname);

  /*
   * Update the thumbnail by simulating an 'enter' press on the entry
   * widget 
   */
  gtk_signal_emit_by_name(GTK_OBJECT(fentry->entry), "activate");
}

/*
 * Brings up a file dialog to select a file 
 */
void get_file_name(GtkWidget * widget, FileEntry * fentry)
{
  gchar full_fname[100], *fname;

  if (fentry->filesel != NULL)
    return;

  gtk_entry_get_text(GTK_ENTRY(fentry->entry));
  fentry->filesel = gtk_file_selection_new(fentry->filesel_title);

  fname = gtk_entry_get_text(GTK_ENTRY(fentry->entry));
  /*
   * If a directory 
   */
  if (fname[strlen(fname) - 1] == '/')
    gtk_file_selection_set_filename(GTK_FILE_SELECTION
				    (fentry->filesel), fname);
  else
  {
    snprintf(full_fname, 100, "%s%s",
	     fentry->page->player->texture_dir, fname);
    gtk_file_selection_set_filename(GTK_FILE_SELECTION
				    (fentry->filesel), full_fname);
  }

  gtk_signal_connect(GTK_OBJECT(fentry->filesel),
		     "unrealize",
		     GTK_SIGNAL_FUNC(exit_dialog), &fentry->filesel);
  gtk_signal_connect(GTK_OBJECT
		     (GTK_FILE_SELECTION(fentry->filesel)->ok_button),
		     "clicked", GTK_SIGNAL_FUNC(set_file_name), fentry);
  gtk_signal_connect_object(GTK_OBJECT
			    (GTK_FILE_SELECTION
			     (fentry->filesel)->ok_button), "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(fentry->filesel));
  gtk_signal_connect_object(GTK_OBJECT
			    (GTK_FILE_SELECTION
			     (fentry->filesel)->cancel_button),
			    "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(fentry->filesel));

  gtk_widget_show_all(fentry->filesel);
}

/*
 * Adds a file entry into a table 
 */
FileEntry *add_file_entry(PlayerPage * page,
			  gchar * label_text, gchar * entry_text, gint loc)
{
  FileEntry *fentry = malloc(sizeof(FileEntry));
  GtkWidget *label, *hbox;

  fentry->page = page;
  fentry->filesel_title = g_strdup(label_text);
  fentry->filesel = NULL;

  label = gtk_label_new(label_text);
  gtk_table_attach_defaults(GTK_TABLE(page->table),
			    label, 0, 1, loc, loc + 1);
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_table_attach_defaults(GTK_TABLE(page->table),
			    hbox, 1, 2, loc, loc + 1);
  fentry->entry = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(fentry->entry), entry_text);
  gtk_box_pack_start(GTK_BOX(hbox), fentry->entry, TRUE, TRUE, 1);
  /*
   * A button to bring up a file selection window 
   */
  fentry->elipse_button = gtk_button_new_with_label("...");
  gtk_signal_connect(GTK_OBJECT(fentry->elipse_button),
		     "clicked", GTK_SIGNAL_FUNC(get_file_name), fentry);
  gtk_box_pack_end(GTK_BOX(hbox), fentry->elipse_button, TRUE, TRUE, 1);

  return fentry;
}

/*
 * Reloads the textures for the preview 
 */
void reload_thumbnail_textures(GtkWidget * widget, PlayerPage * page)
{
  /*
   * Redraw the thumbnail 
   */
  /*
   * FIXME: this _should_ be a function 
   */
  if (gtk_gl_area_make_current(GTK_GL_AREA(page->thumbnail)))
  {
    reload_player_textures(page->player);

    draw_thumbnail(page);
    gtk_gl_area_swapbuffers(GTK_GL_AREA(page->thumbnail));
  }
}

/*
 * Add a file entry with the appropriate signals connected 
 */
FileEntry *add_texture_file_entry(PlayerPage * page,
				  gchar * title, gchar ** file_string,
				  gint loc)
{
  FileEntry *fentry;

  fentry = add_file_entry(page, title, *file_string, loc);
  gtk_signal_connect(GTK_OBJECT(fentry->entry),
		     "changed",
		     GTK_SIGNAL_FUNC(store_file_name), file_string);
  gtk_signal_connect(GTK_OBJECT(fentry->entry),
		     "activate",
		     GTK_SIGNAL_FUNC(reload_thumbnail_textures), page);

  return fentry;
}

/*
 * Draws the thumbnail (preview) 
 */
void draw_thumbnail(PlayerPage * page)
{
  Game *game = page->pdialog->game_copy;
  Player *player = page->player;
  extern float cog_height[6];
  GLfloat white_texture_colour[4] = { 1.0, 1.0, 1.0, 1.0 };
  int piece = KNIGHT;

  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  glTranslatef(0.0, -cog_height[piece - 1], -20.0);
  if (piece == KNIGHT)
    glRotatef(-90.0, 0.0, 1.0, 0.0);

  glPushAttrib(GL_LIGHTING | GL_TEXTURE_BIT | GL_ENABLE_BIT | GL_FOG_BIT);

  if (game->is_fog)
    glEnable(GL_FOG);
  else
    glDisable(GL_FOG);

  if (game->is_light)
    glEnable(GL_LIGHTING);
  else
    glDisable(GL_LIGHTING);

  if (game->is_smooth)
    glShadeModel(GL_SMOOTH);
  else
    glShadeModel(GL_FLAT);

  glColor4fv(player->piece_colour);
  if (game->is_texture)
  {
    glColor4f(1.0, 1.0, 1.0, 0.0);
    glEnable(GL_TEXTURE_2D);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
    glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, white_texture_colour);
    glBindTexture(GL_TEXTURE_2D, player->piece_texture);
  } else
  {
    glDisable(GL_TEXTURE_2D);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, player->piece_colour);
  }

  switch (piece)
  {
  case KING:
    draw_king();
    break;
  case QUEEN:
    draw_queen();
    break;
  case BISHOP:
    draw_bishop();
    break;
  case KNIGHT:
    draw_knight();
    break;
  case ROOK:
    draw_rook();
    break;
  case PAWN:
    draw_pawn();
    break;
  }

  glPopAttrib();
}

/*
 * Reshape the thumbnail view 
 */
void reshape_thumbnail(int w, int h)
{
  glViewport(0, 0, (GLsizei) w, (GLsizei) h);

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();

  gluPerspective(60.0, (GLfloat) w / (GLfloat) h, 0.5, 10000.0);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
}

gint
thumbnail_expose_view(GtkWidget * widget,
		      GdkEventExpose * event, PlayerPage * page)
{
  /*
   * Draw only last expose 
   */
  if (event->count < 0)
    return TRUE;

  /*
   * Redraw the view 
   */
  if (gtk_gl_area_make_current(GTK_GL_AREA(widget)))
  {
    draw_thumbnail(page);
    gtk_gl_area_swapbuffers(GTK_GL_AREA(widget));
  }

  return TRUE;
}

gint
thumbnail_reshape_view(GtkWidget * widget,
		       GdkEventConfigure * event, PlayerPage * page)
{
  if (gtk_gl_area_make_current(GTK_GL_AREA(widget)))
    reshape_thumbnail(widget->allocation.width, widget->allocation.height);
  else
    return FALSE;

  return TRUE;
}

gint thumbnail_init_view(GtkWidget * widget, PlayerPage * page)
{
  GLfloat light_pos[3] = { -30.0, 30.0, 30.0 };
  GLfloat light_ambient[4] = { 0.125, 0.125, 0.125, 0.0 };

  /*
   * openGL functions can only be called if make_current) returns TRUE 
   */
  if (gtk_gl_area_make_current(GTK_GL_AREA(widget)))
  {
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_CULL_FACE);
    glEnable(GL_LIGHT0);
    glLightfv(GL_LIGHT0, GL_POSITION, light_pos);
    glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient);
    /*
     * FIXME: to be implemented 
     */
    /*
     * glLightfv(GL_LIGHT0, GL_AMBIENT,
     * page->pdialog->game_copy->light_ambient_colour); 
     */
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

    load_textures(page->pdialog->game_copy);
  } else
    return FALSE;

  return TRUE;
}

/*
 * Add a small GLarea to preview the changes to the pieces properties 
 */
void add_piece_thumbnail(PlayerPage * page, guint loc)
{
  /*
   * FIXME: don't need doublebuffer... how to remove? 
   */
  /*
   * int attrlist[] = { GDK_GL_RGBA, GDK_GL_RED_SIZE,1,
   * GDK_GL_GREEN_SIZE,1, GDK_GL_BLUE_SIZE,1, GDK_GL_DEPTH_SIZE,1,
   * GDK_GL_NONE };
   */
  int attrlist[] = {
    GDK_GL_RGBA,
    GDK_GL_RED_SIZE, 1,
    GDK_GL_GREEN_SIZE, 1,
    GDK_GL_BLUE_SIZE, 1,
    GDK_GL_DOUBLEBUFFER,
    GDK_GL_DEPTH_SIZE, 1,
    GDK_GL_STENCIL_SIZE, 1,
    GDK_GL_NONE
  };

  page->thumbnail = gtk_gl_area_new(attrlist);
  gtk_widget_set_usize(GTK_WIDGET(page->thumbnail), 80, 0);
  /*
   * Signals 
   */
  gtk_widget_set_events(page->thumbnail, GDK_EXPOSURE_MASK);
  gtk_signal_connect(GTK_OBJECT(page->thumbnail),
		     "expose_event",
		     GTK_SIGNAL_FUNC(thumbnail_expose_view), page);
  gtk_signal_connect(GTK_OBJECT(page->thumbnail),
		     "configure_event",
		     GTK_SIGNAL_FUNC(thumbnail_reshape_view), page);
  gtk_signal_connect(GTK_OBJECT(page->thumbnail),
		     "realize", GTK_SIGNAL_FUNC(thumbnail_init_view),
		     page);

  gtk_table_attach_defaults(GTK_TABLE(page->table),
			    page->thumbnail, 2, 3, 0, loc);

}

/*
 * Changes the colour of the colour panel 
 */
void set_cpanel_colour(GtkWidget * wdiget, ColourEntry * centry)
{
  GtkWidget *cpanel = centry->cpanel;
  GLfloat *colours = *centry->colours;
  GdkColor gdk_colour;
  GdkColormap *colormap;

  /*
   * Set the initial colour 
   */
  gdk_colour.red = (guint16) (colours[0] * 65535.0);
  gdk_colour.green = (guint16) (colours[1] * 65535.0);
  gdk_colour.blue = (guint16) (colours[2] * 65535.0);
  colormap = gdk_window_get_colormap((cpanel->window));
  gdk_color_alloc(colormap, &gdk_colour);
  gdk_window_set_background(cpanel->window, &gdk_colour);
  gdk_window_clear(cpanel->window);
}

/*
 * Stores the new colour 
 */
void set_colour(GtkWidget * widget, ColourEntry ** centry_ptr)
{
  ColourEntry *centry = *centry_ptr;
  gdouble gcolours[4];
  int i;

  gtk_color_selection_get_color(GTK_COLOR_SELECTION
				(centry->colour_selection), gcolours);
  for (i = 0; i < 4; i++)
    (*centry->colours)[i] = (GLfloat) gcolours[i];
  set_cpanel_colour(widget, centry);

  if (gtk_gl_area_make_current(GTK_GL_AREA(centry->page->thumbnail)))
  {
    draw_thumbnail(centry->page);
    gtk_gl_area_swapbuffers(GTK_GL_AREA(centry->page->thumbnail));
  }
}

/*
 * Edits the selected colour 
 */
void
edit_colour(GtkWidget * cpanel, GdkEvent * event, ColourEntry * centry)
{
  static ColourEntry **centry_ptr = NULL;
  static GtkWidget *colour_sel_dialog = NULL;
  static GtkWidget *colour_selection;
  gdouble dcolours[4];
  int i;

  if (event->type != GDK_BUTTON_PRESS)
    return;

  /*
   * Change what colour entry we're editing 
   */
  if (centry_ptr == NULL)
    centry_ptr = malloc(sizeof(ColourEntry *));
  *centry_ptr = centry;

  /*
   * If not visible, make a dialog 
   */
  if (colour_sel_dialog == NULL)
  {
    GtkWidget *apply_button, *close_button;

    /*
     * The dialog 
     */
    colour_sel_dialog = gtk_dialog_new();
    gtk_window_set_title(GTK_WINDOW(colour_sel_dialog), "Pick new colour");
    gtk_signal_connect(GTK_OBJECT(colour_sel_dialog), "unrealize",
		       GTK_SIGNAL_FUNC(exit_dialog), &colour_sel_dialog);

    /*
     * The colour selection 
     */
    colour_selection = gtk_color_selection_new();
    gtk_color_selection_set_opacity(GTK_COLOR_SELECTION
				    (colour_selection), TRUE);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(colour_sel_dialog)->vbox),
		       colour_selection, TRUE, TRUE, 10);

    /*
     * The buttons 
     */
    close_button = gtk_button_new_with_label("Close");
    gtk_signal_connect_object(GTK_OBJECT(close_button),
			      "clicked",
			      GTK_SIGNAL_FUNC(gtk_widget_destroy),
			      GTK_OBJECT(colour_sel_dialog));
    gtk_box_pack_end(GTK_BOX
		     (GTK_DIALOG(colour_sel_dialog)->action_area),
		     close_button, FALSE, FALSE, 0);
    apply_button = gtk_button_new_with_label("Apply");
    gtk_signal_connect(GTK_OBJECT(apply_button),
		       "clicked", GTK_SIGNAL_FUNC(set_colour), centry_ptr);
    gtk_box_pack_end(GTK_BOX
		     (GTK_DIALOG(colour_sel_dialog)->action_area),
		     apply_button, FALSE, FALSE, 0);
  }

  /*
   * Remember the colour selection 
   */
  centry->colour_selection = colour_selection;

  /*
   * Set the colour of the colour selection 
   */
  for (i = 0; i < 4; i++)
    dcolours[i] = (gdouble) (*centry->colours)[i];
  gtk_color_selection_set_color(GTK_COLOR_SELECTION(colour_selection),
				dcolours);

  gtk_widget_show_all(colour_sel_dialog);
}

/*
 * Add an entry into the table 
 */
GtkWidget *add_colour_entry(GtkWidget * table, gchar * label_text,
			    gint loc)
{
  GtkWidget *label, *colour_panel;

  label = gtk_label_new(label_text);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, loc, loc + 1);

  colour_panel = gtk_drawing_area_new();
  gtk_widget_set_events(colour_panel, GDK_BUTTON_PRESS_MASK);
  gtk_table_attach_defaults(GTK_TABLE(table),
			    colour_panel, 1, 2, loc, loc + 1);

  return colour_panel;
}

/*
 * Add a colour entry into the table 
 */
ColourEntry *add_player_colour_entry(PlayerPage * page,
				     gchar * label_text,
				     GLfloat ** colours, gint loc)
{
  ColourEntry *centry = malloc(sizeof(ColourEntry));

  centry->page = page;
  centry->cpanel = add_colour_entry(page->table, label_text, loc);
  centry->colours = colours;

  gtk_signal_connect(GTK_OBJECT(centry->cpanel),
		     "realize", GTK_SIGNAL_FUNC(set_cpanel_colour),
		     centry);
  gtk_signal_connect(GTK_OBJECT(centry->cpanel), "event",
		     GTK_SIGNAL_FUNC(edit_colour), centry);

  return centry;
}

void change_player_name(GtkWidget * widget, Player * player)
{
  set_player_name(player, gtk_entry_get_text(GTK_ENTRY(widget)));
}

/*
 * Add into a notebook a page on the desired player 
 */
PlayerPage *add_player_page(PreferenceDialog * pdialog, Player * player,
			    char *page_name)
{
  GtkWidget *notebook = pdialog->notebook;
  GtkWidget *label;
  PlayerPage *page = malloc(sizeof(PlayerPage));

  page->pdialog = pdialog;
  page->player = player;

  page->table = gtk_table_new(7, 3, FALSE);

  /*
   * Add the entries 
   */
  page->name_entry = add_text_entry(page->table, "Player name:",
				    player->name, 0);
  gtk_signal_connect(GTK_OBJECT(page->name_entry),
		     "changed",
		     GTK_SIGNAL_FUNC(change_player_name), page->player);
  page->pcolour_entry = add_player_colour_entry(page, "Piece colour:",
						&page->player->
						piece_colour, 1);
  page->scolour_entry =
      add_player_colour_entry(page, "Selected piece colour:",
			      &page->player->selected_colour, 2);
  page->bcolour_entry =
      add_player_colour_entry(page, "Board colour:",
			      &page->player->board_colour, 3);

  page->tfile_entry = add_texture_file_entry(page, "Texture Directory:",
					     &page->player->texture_dir,
					     4);
  page->ptfile_entry =
      add_texture_file_entry(page, "Piece Texture:",
			     &page->player->piece_texture_file, 5);
  page->sptfile_entry =
      add_texture_file_entry(page, "Selected Piece Texture:",
			     &page->player->selected_piece_texture_file,
			     6);
  page->btfile_entry =
      add_texture_file_entry(page, "Board Texture:",
			     &page->player->board_texture_file, 7);

  /*
   * Add a model thumbnail 
   */
  add_piece_thumbnail(page, 8);

  /*
   * Add the table into the notebook 
   */
  label = gtk_label_new(page_name);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page->table, label);

  return page;
}

void set_cecp_mode(GtkWidget * entry, PreferenceDialog * pdialog)
{
  Game *game = pdialog->game_copy;

  if (!strcmp(gtk_entry_get_text(GTK_ENTRY(entry)), "HARD"))
    game->cecp.mode = CECP_MODE_HARD;
  else
    game->cecp.mode = CECP_MODE_EASY;
}

void set_cecp_searchd(GtkAdjustment * sd_adj, PreferenceDialog * pdialog)
{
  pdialog->game_copy->cecp.search_depth = (int) sd_adj->value;
}

void set_cecp_ai_player(GtkWidget * entry, PreferenceDialog * pdialog)
{
  Game *game = pdialog->game_copy;

  if (!strcmp(gtk_entry_get_text(GTK_ENTRY(entry)), "WHITE"))
    game->cecp.ai_player = WHITE PLAYER;
  else
    game->cecp.ai_player = BLACK PLAYER;
}

/*
 * Adds a notebook page for Engine/CECP options 
 */
void add_cecp_page(PreferenceDialog * pdialog)
{
  Game *game = pdialog->game_copy;
  GList *difficulty_list = NULL, *ai_player_list = NULL;
  GtkWidget *table, *label, *combo, *sd_spin;
  GtkAdjustment *sd_adj;

  table = gtk_table_new(3, 3, FALSE);

  label = gtk_label_new("Difficulty:");
  difficulty_list = g_list_append(difficulty_list, "EASY");
  difficulty_list = g_list_append(difficulty_list, "HARD");
  combo = gtk_combo_new();
  gtk_combo_set_popdown_strings(GTK_COMBO(combo), difficulty_list);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
  if (game->cecp.mode == CECP_MODE_HARD)
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), "HARD");
  else
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), "EASY");
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry),
		     "changed", GTK_SIGNAL_FUNC(set_cecp_mode), pdialog);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), combo, 1, 2, 0, 1);

  label = gtk_label_new("Search depth:");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
  sd_adj =
      (GtkAdjustment *) gtk_adjustment_new(game->cecp.search_depth, 0,
					   100, 1, 2, 0);
  sd_spin = gtk_spin_button_new(sd_adj, 0.1, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(sd_spin), TRUE);
  gtk_signal_connect(GTK_OBJECT(sd_adj),
		     "value_changed",
		     GTK_SIGNAL_FUNC(set_cecp_searchd), pdialog);
  gtk_table_attach_defaults(GTK_TABLE(table), sd_spin, 1, 2, 1, 2);

  label = gtk_label_new("AI Player:");
  ai_player_list = g_list_append(ai_player_list, "WHITE");
  ai_player_list = g_list_append(ai_player_list, "BLACK");
  combo = gtk_combo_new();
  gtk_combo_set_popdown_strings(GTK_COMBO(combo), ai_player_list);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
  if (game->cecp.ai_player == WHITE PLAYER)
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), "WHITE");
  else
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), "BLACK");
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry),
		     "changed",
		     GTK_SIGNAL_FUNC(set_cecp_ai_player), pdialog);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), combo, 1, 2, 2, 3);

  label = gtk_label_new("Engine/CECP");
  gtk_notebook_append_page(GTK_NOTEBOOK(pdialog->notebook), table, label);
}

/*
 * Apply the changes 
 */
void apply_preferences(GtkWidget * widget, PreferenceDialog * pdialog)
{
  glChessWidget *glcwidget = pdialog->glcwidget;
  Game *old_game = glcwidget->game, *game = pdialog->game_copy;
  Camera *tmp_camera;
  GtkWidget *menu_item;
  int i, j;

  /*
   * Update the main menu 
   */
  menu_item = gtk_item_factory_get_widget(glcwidget->item_factory,
					  "/View/Lighting");
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item),
				 game->is_light);
  menu_item =
      gtk_item_factory_get_widget(glcwidget->item_factory,
				  "/View/Shading");
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item),
				 game->is_smooth);
  menu_item =
      gtk_item_factory_get_widget(glcwidget->item_factory,
				  "/View/Reflections");
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item),
				 game->is_reflect);
  menu_item =
      gtk_item_factory_get_widget(glcwidget->item_factory,
				  "/View/Textures");
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item),
				 game->is_texture);
  menu_item =
      gtk_item_factory_get_widget(glcwidget->item_factory, "/View/Fog");
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item),
				 game->is_fog);
  menu_item =
      gtk_item_factory_get_widget(glcwidget->item_factory,
				  "/View/Show Coords");
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item),
				 game->is_coord);

  /*
   * Copy the game with the new settings and make it the current game 
   */
  glcwidget->game = copy_game(pdialog->game_copy);

  /*
   * But don't loose important (non-config) things -- like time 
   */
  game = glcwidget->game;

  /*
   * Player options 
   */
  game->white_player->pieces = old_game->white_player->pieces;
  game->white_player->score = old_game->white_player->score;
  game->white_player->time = old_game->white_player->time;
  game->black_player->pieces = old_game->black_player->pieces;
  game->black_player->score = old_game->black_player->score;
  game->black_player->time = old_game->black_player->time;
  if (old_game->current_player == old_game->white_player)
    game->current_player = game->white_player;
  else
    game->current_player = game->black_player;

  /*
   * The board 
   */
  for (i = 0; i < 8; i++)
    for (j = 0; j < 8; j++)
      game->board[i][j] = old_game->board[i][j];
  game->selected[0] = old_game->selected[0];
  game->selected[1] = old_game->selected[1];

  /*
   * Camera options 
   */
  tmp_camera = game->camera;
  game->camera = old_game->camera;
  old_game->camera = tmp_camera;

  /*
   * Time options 
   */
  game->is_paused = old_game->is_paused;
  game->turn = old_game->turn;
  game->elapsed_time = old_game->elapsed_time;

  free(old_game);

  /* If is now the ai players turn, make them go */
  if ((((game->cecp.ai_player == WHITE PLAYER)
	&& (game->current_player == game->white_player))
       || ((game->cecp.ai_player == BLACK PLAYER)
	   && (game->current_player == game->black_player)))
      && cecp_interface_up())
  {
    int engine_start[2], engine_target[2];

    write_to_engine("g");
    /*
     * Get the engines move 
     */
    parse_move_from_engine(game, read_from_engine(), engine_start,
			   engine_target);
    /*
     * Tell glChess 
     */
    game->selected[0] = engine_start[0];
    game->selected[1] = engine_start[1];
    move_piece(game, engine_target);
  }

  /*
   * Update the display 
   */
  display_new_preferences(widget, pdialog);
}

/*
 * Actually update the new preferences 
 */
void
display_new_preferences(GtkWidget * widget, PreferenceDialog * pdialog)
{
  glChessWidget *glcwidget = pdialog->glcwidget;
  Game *game = glcwidget->game;
  char *player_label_text;

  /*
   * If there is a name set, update it 
   */
  gtk_label_get(GTK_LABEL(glcwidget->player_label), &player_label_text);
  if (strcmp(player_label_text, ""))
    gtk_label_set_text(GTK_LABEL(glcwidget->player_label),
		       game->current_player->name);

  /*
   * Reload the textures 
   */
  if (gtk_gl_area_make_current(GTK_GL_AREA(glcwidget->glarea)))
    reload_textures(game);

  /*
   * Redisplay the GL view 
   */
  post_redisplay(game);
}

/*
 * Make the preferences dialog 
 */
void preferences(glChessWidget * glcwidget, guint num, GtkWidget * widget)
{
  static PreferenceDialog *pdialog = NULL;
  GtkWidget *label, *table;

  if (pdialog == NULL)
  {
    pdialog = malloc(sizeof(PreferenceDialog));
    pdialog->dialog = NULL;
  }

  if (pdialog->dialog != NULL)
    return;

  /*
   * Remember the existing game 
   */
  pdialog->glcwidget = glcwidget;

  /*
   * Make a copy of the current game (preferences) 
   */
  pdialog->game_copy = copy_game(glcwidget->game);

  /*
   * Make the dialog 
   */
  pdialog->dialog = gtk_dialog_new();
  gtk_signal_connect(GTK_OBJECT(pdialog->dialog),
		     "unrealize",
		     GTK_SIGNAL_FUNC(exit_dialog),
		     (gpointer) & pdialog->dialog);

  /*
   * Make a notebook 
   */
  pdialog->notebook = gtk_notebook_new();

  /*
   * Player options 
   */
  pdialog->white_page = add_player_page(pdialog,
					pdialog->game_copy->white_player,
					"White Player");
  pdialog->black_page = add_player_page(pdialog,
					pdialog->game_copy->black_player,
					"Black Player");

  /*
   * Graphic options 
   */
  table = gtk_table_new(6, 3, FALSE);
  label = gtk_label_new("Graphics");

  add_check_button(pdialog, table, "Lighting",
		   &pdialog->game_copy->is_light, 0);
  add_check_button(pdialog, table, "Shading",
		   &pdialog->game_copy->is_smooth, 1);
  add_check_button(pdialog, table, "Reflections",
		   &pdialog->game_copy->is_reflect, 2);
  add_check_button(pdialog, table, "Textures",
		   &pdialog->game_copy->is_texture, 3);
  add_check_button(pdialog, table, "Fog", &pdialog->game_copy->is_fog, 4);
  add_check_button(pdialog, table, "Show Coords",
		   &pdialog->game_copy->is_coord, 5);

  gtk_notebook_append_page(GTK_NOTEBOOK(pdialog->notebook), table, label);

  /* Engine/CECP options */
  add_cecp_page(pdialog);


  pdialog->ok_button = gtk_button_new_with_label("OK");
  /* Apply the changes */
  gtk_signal_connect(GTK_OBJECT(pdialog->ok_button),
		     "clicked",
		     GTK_SIGNAL_FUNC(apply_preferences), pdialog);
  gtk_signal_connect_object(GTK_OBJECT(pdialog->ok_button),
			    "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(pdialog->dialog));
  pdialog->apply_button = gtk_button_new_with_label("Apply");
  /*
   * Apply the changes 
   */
  gtk_signal_connect(GTK_OBJECT(pdialog->apply_button),
		     "clicked",
		     GTK_SIGNAL_FUNC(apply_preferences), pdialog);
  pdialog->cancel_button = gtk_button_new_with_label("Cancel");
  /*
   * Close the dialog 
   */
  gtk_signal_connect_object(GTK_OBJECT(pdialog->cancel_button),
			    "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(pdialog->dialog));

  /*
   * FIXME: temp warning label 
   */
  label = gtk_label_new("!!!WARNING!!!\n"
			"The are odd things going on in this dialog,"
			" so be warned that things may not act as you"
			" suspect. Things may have to be selected more"
			" than once to make them work.\n");
  gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(pdialog->dialog)->vbox),
		     label, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(pdialog->dialog)->vbox),
		     pdialog->notebook, TRUE, TRUE, 0);
  gtk_box_pack_end(GTK_BOX(GTK_DIALOG(pdialog->dialog)->action_area),
		   pdialog->cancel_button, TRUE, TRUE, 0);
  gtk_box_pack_end(GTK_BOX(GTK_DIALOG(pdialog->dialog)->action_area),
		   pdialog->apply_button, TRUE, TRUE, 0);
  gtk_box_pack_end(GTK_BOX(GTK_DIALOG(pdialog->dialog)->action_area),
		   pdialog->ok_button, TRUE, TRUE, 0);

  gtk_widget_show_all(pdialog->dialog);
}
