/*
 * glChess - A 3D chess interface
 *
 * Copyright (C) 2001  Robert  Ancell <bob27@users.sourceforge.net>
 *                     Michael Duelli <duelli@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* picksquare.c - contains the mouse functions, selection algorithm, and move
 * rules. */

#include <stdio.h>
#include <gdk/gdk.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include "global.h"
#include "picksquare.h"
#include "draw.h"

#include "engine.h"

extern int engine;
extern int debug;
extern int promotion;

/* Click the screen */
void mouse(Game * game, int button, int x, int y)
{
  int square[2], colour;
  Player *player;

  game->camera->oldx = x;
  game->camera->oldy = y;
  game->camera->button = button;

  if (pick_square(game, x, y, square) != -1)
  {
    /* remember selected square and move piece */
    if (game->selected[0] != -1)	/* second selection: target */
    {
      if (engine != HUMAN)
      {
	int engine_start[2];
	int engine_target[2];

	/* Null move check must take place before write_to_engine */
	if (game->selected[0] == square[0]
	    && game->selected[1] == square[1])
	{
	  game->selected[0] = -1;	/* unselect pieces */
	  return;
	}

	/* Now it's the engine's turn */
	write_to_engine(convert_notation(game->selected, square));

	/* FIXME: We got to check the code ourself to make each move
	   * happen when it is made */

	/* Now parse the engine's move, our move could be illegal */
	if (parse_move_from_engine
	    (game, read_from_engine(), engine_start, engine_target))
	{
	  /* Your move ... */
	  move_piece(game, square);

	  game->selected[0] = engine_start[0];
	  game->selected[1] = engine_start[1];

	  /* ... the engine's move */
	  move_piece(game, engine_target);
	} else
	{
	  if (debug)
	    debug_output("Your move was illegal, no move.\n");
	}

      } else
      {
	move_piece(game, square);
      }
    } else
    {				/* first selection: start */

      colour = game->board[square[0]][square[1]];
      if (colour > 10)
	player = game->black_player;
      else
	player = game->white_player;

      if ((game->board[square[0]][square[1]] != -1)
	  && (player == game->current_player))
      {
	game->selected[0] = square[0];
	game->selected[1] = square[1];
      }
    }
  }
}

/* The trackball */
void mouse_move(Game * game, int button, int x, int y)
{
  int dx, dy;

  if (game->camera->mode != MODE_TRACKBALL)
    return;

  dx = x - game->camera->oldx;
  dy = y - game->camera->oldy;

  if (button == 1)
  {
    game->camera->track_rot[0] += 1.0f * (GLfloat) dx;
    /* Why should we look at the bottom ? I want to see the board * not
       * the interior of the pieces ! */
    if (game->camera->track_rot[1] + (1.0f * (GLfloat) dy) >= 6.0 &&
	game->camera->track_rot[1] + (1.0f * (GLfloat) dy) <= 90.0)
      game->camera->track_rot[1] += 1.0f * (GLfloat) dy;
  } else if (button == 2)
    game->camera->track_rad += 1.0f * (GLfloat) dy;

  game->camera->oldx = x;
  game->camera->oldy = y;

  post_redisplay(game);
}

/* Draws the board squares for picking */
void draw_squares(void)
{
  GLuint x, y;

  /* draw the floor squares */

  for (x = 0; x < 8; x++)
  {
    glLoadName(x);

    for (y = 0; y < 8; y++)
    {
      glPushName(y);

      /* Draw square */
      glBegin(GL_QUADS);
      glVertex3f(x * 10.0 - 40.0, 0.0, 40.0 - y * 10.0);
      glVertex3f(x * 10.0 - 30.0, 0.0, 40.0 - y * 10.0);
      glVertex3f(x * 10.0 - 30.0, 0.0, 30.0 - y * 10.0);
      glVertex3f(x * 10.0 - 40.0, 0.0, 30.0 - y * 10.0);
      glEnd();

      glPopName();
    }
  }
}

/*
 * Select the piece under the pointer 
 */
int pick_square(Game * game, int x, int y, int sq[])
{
  GLuint selectBuf[5];
  GLint hits;
  GLint viewport[4];

  glGetIntegerv(GL_VIEWPORT, viewport);

  glSelectBuffer(5, selectBuf);
  (void) glRenderMode(GL_SELECT);

  glInitNames();
  glPushName(0);

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();

  /*
   * create 5x5 pixel picking region near cursor location 
   */
  gluPickMatrix((GLdouble) x, (GLdouble) (viewport[3] - y), 1.0, 1.0,
		viewport);

  redo_perspective(game, viewport[2], viewport[3]);

  draw_squares();
  glPopMatrix();
  glFlush();

  hits = glRenderMode(GL_RENDER);

  if (hits <= 0)
  {
    glMatrixMode(GL_MODELVIEW);
    return -1;
  }

  sq[0] = selectBuf[3];
  sq[1] = selectBuf[4];

  glMatrixMode(GL_MODELVIEW);

  /*
   * Update view 
   */
  post_redisplay(game);

  return FALSE;
}

/*
 * Moves the selected piece to target[] 
 */
int move_piece(Game * game, int *target)
{
  int piece, tpiece, colour, tcolour;
  int dx, dy, lx, ly;

  if (engine == HUMAN)
  {

    /* If target is selected piece */
    if ((target[0] == game->selected[0])
	&& (target[1] == game->selected[1]))
    {
      game->selected[0] = -1;	/* reset selection */
      engine_dialog("Your move was illegal. No move");
      return -1;		/* illegal move */
    }
  }

  piece = game->board[game->selected[0]][game->selected[1]];	/* selected piece */
  /* If black */
  if (piece > 10)
  {
    colour = 0;			/* black */
    piece -= 10;
  } else
    colour = 1;			/* white */

  tpiece = game->board[target[0]][target[1]];	/* target piece */
  /* If black */
  if (tpiece > 10)
  {
    tcolour = 0;		/* black */
    tpiece -= 10;
  } else if (tpiece != -1)
    tcolour = 1;		/* white */
  else
    tcolour = -1;

  /* If target is friendly - null move */
  if (tcolour == colour)
    return -1;

  if (engine != HUMAN)
  {

    /* We got to do this for moving the rook when castling, * move rules
     * are done by the engine, but a castling move is simply e1c1, e1g1,
     * e8c8 or e8g8 */
    if (piece == KING)
    {
      if (game->selected[0] == 4 && game->selected[1] == 0)
      {				/* e1 */
	/* either e1c1 or e1g1 */
	if ((target[0] == 2 || target[0] == 6) && target[1] == 0)
	{
	  if (target[0] == 2 && game->board[0][0] == WHITE ROOK)
	  {
	    game->board[0][0] = -1;	/*
					 * Take the rook and ... 
					 */
	    game->board[3][0] = WHITE ROOK;	/*
						 * ... drop it here 
						 */
	  } else if (target[0] == 6 && game->board[7][0] == WHITE ROOK)
	  {
	    game->board[7][0] = -1;	/*
					 * Take the rook and ... 
					 */
	    game->board[5][0] = WHITE ROOK;	/*
						 * ... drop it here 
						 */
	  }
	}
      } else if (game->selected[0] == 4 && game->selected[1] == 7)
      {				/* e8 */
	/* Either e8c8 or e8g8 */
	if ((target[0] == 2 || target[0] == 6) && target[1] == 7)
	{
	  if (target[0] == 2 && game->board[0][7] == BLACK ROOK)
	  {
	    game->board[0][7] = -1;	/* Take the rook and ... */
	    game->board[3][7] = BLACK ROOK;	/* ... drop it here */
	  } else if (target[0] == 6 && game->board[7][7] == BLACK ROOK)
	  {
	    game->board[7][7] = -1;	/* Take the rook and ... */
	    game->board[5][7] = BLACK ROOK;	/* ... drop it here */
	  }
	}
      }
    }

    /* Pawn promotion */
    /* FIXME: Add promotion choice dialog here */
    if (piece == PAWN && promotion != -1)
    {
      /* White: line 7 -> line 8 */
      if (game->selected[1] == 6 && target[1] == 7)
      {
	game->board[game->selected[0]][game->selected[1]] =
	    WHITE promotion;
	promotion = -1;
      }
      /* Black: line 2 -> line 1 */
      else if (game->selected[1] == 1 && target[1] == 0)
      {
	game->board[game->selected[0]][game->selected[1]] =
	    BLACK promotion;
	promotion = -1;
      }
    }

    if (piece == PAWN)
    {
      /* The target must be empty */
      if (game->board[target[0]][target[1]] == -1)
      {
	/* Only takes place when moving from row 4 to 3, resp. 6 to 7 */
	if (game->selected[1] == 3 && target[1] == 2)
	{
	  game->board[target[0]][target[1] + 1] = -1;
	} else if (game->selected[1] == 5 && target[1] == 6)
	{
	  game->board[target[0]][target[1] - 1] = -1;
	}
      }
    }
  }

  else
  {

    dx = lx = target[0] - game->selected[0];
    dy = ly = target[1] - game->selected[1];
    if (lx < 0)
      lx = -lx;
    if (ly < 0)
      ly = -ly;

    switch (piece)
    {
    case KING:
      if ((dx * dx + dy * dy) > 2)
      {
	/* Castling rule: */
	/* FIXME: doesn't check if in check */
	if (colour == 1)
	{			/* White */
	  /* if moving sideways 2 and on base line */
	  if ((lx == 2) && (dy == 0) && (game->selected[1] == 0)
	      /* and the white rook is still there */
	      && (game->board[target[0] + dx / lx][0] == 5)
	      /* and there is free space between them  */
	      && check_ortho(game, game->selected, target))
	  {
	    game->board[target[0] + dx / lx][0] = -1;	/* move the rook now */
	    game->board[target[0] - dx / lx][0] = 5;
	  } else
	    return -1;
	} else
	{			/* Black */
	  /* If moving sideways 2 and on base line */
	  if ((lx == 2) && (dy == 0) && (game->selected[1] == 7)
	      /* and the black rook is still there */
	      && (game->board[target[0] + dx / lx][7] == 15)
	      /*
	       * and there is free space between them 
	       */
	      && check_ortho(game, game->selected, target))
	  {
	    game->board[target[0] + dx / lx][7] = -1;	/*
							 * move the
							 * rook now 
							 */
	    game->board[target[0] - dx / lx][7] = 15;
	  } else
	    return -1;
	}
      }
      break;

    case QUEEN:
      if ((dx == 0) || (dy == 0))
      {
	if (!check_ortho(game, game->selected, target))
	  return -1;
      } else if (lx == ly)
      {
	if (!check_diag(game, game->selected, target))
	  return -1;
      } else
	return -1;

      break;

    case BISHOP:
      if (lx == ly)
      {
	if (!check_diag(game, game->selected, target))
	  return -1;
      } else
	return -1;
      break;

    case KNIGHT:
      if (!(((lx == 2) && (ly == 1)) || (((lx == 1) && (ly == 2)))))
	return -1;
      break;

    case ROOK:
      if ((dx == 0) || (dy == 0))
      {
	if (!check_ortho(game, game->selected, target))
	  return -1;
      } else
	return -1;
      break;

    case PAWN:
      /*
       * needs work to be cleaned up 
       */

      /*
       * if moving "fowards" 1 unit and laterally no more than 1 unit 
       */
      if ((dy == (2 * colour - 1)) && (lx <= 1))
      {
	/*
	 * if not moving diagonally into enemy piece 
	 */
	if ((lx == 1) && !(tcolour != -1))
	  return -1;
	/*
	 * if moving foward into enemy 
	 */
	if ((dx == 0) && (tpiece != -1))
	  return -1;
      } else
      {
	/*
	 * If not on baseline and moving foward two units 
	 */
	if (!((dy == (4 * colour - 2))
	      && (lx == 0)
	      && (game->selected[1] == (6 - 5 * colour))
	      && (tpiece == -1)))
	  return -1;
      }
      break;
    }
  }


  /* Move piece */
  game->board[target[0]][target[1]] =
      game->board[game->selected[0]][game->selected[1]];
  game->board[game->selected[0]][game->selected[1]] = -1;

  /* reset selection */
  game->selected[0] = -1;

  /*
   * Swap turns 
   */
  if (game->current_player == game->white_player)
    game->current_player = game->black_player;
  else
    game->current_player = game->white_player;
  game->turn++;

  return TRUE;
}

/* Check the (orthogonal) line from source[] to dest[] is free of pieces */
int check_ortho(Game * game, int *source, int *dest)
{
  int step, i;

  if (source[0] == dest[0])
  {				/*
				 * along x-axis 
				 */
    if (dest[1] > source[1])
      step = 1;
    else
      step = -1;
    for (i = source[1] + step; i != dest[1]; i += step)
    {
      if (game->board[source[0]][i] != -1)
	return FALSE;
    }
  } else
  {				/*
				 * along y-axis 
				 */

    if (dest[0] > source[0])
      step = 1;
    else
      step = -1;
    for (i = source[0] + step; i != dest[0]; i += step)
    {
      if (game->board[i][source[1]] != -1)
	return FALSE;
    }
  }
  return TRUE;
}

/* Check the diagonal from source[] to dest[] is free of pieces */
int check_diag(Game * game, int *source, int *dest)
{
  int step, i;

  if ((dest[0] - source[0]) == (dest[1] - source[1]))
  {				/* NE/SW */
    if (dest[0] > source[0])
      step = 1;			/* NE */
    else
      step = -1;		/* SW */
    for (i = step; i != dest[0] - source[0]; i += step)
    {
      if (game->board[source[0] + i][source[1] + i] != -1)
	return FALSE;
    }
  } else
  {				/* along NW/SE */

    if (dest[1] > source[1])
      step = 1;			/* NW */
    else
      step = -1;		/* SE */
    for (i = step; i != source[0] - dest[0]; i += step)
    {
      if (game->board[source[0] - i][source[1] + i] != -1)
	return FALSE;
    }
  }
  return TRUE;
}
