/*
 * glChess - A 3D chess interface
 *
 * Copyright (C) 2001  Robert Ancell <bob27@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <GL/gl.h>
#include <math.h>
#include <stdio.h>

#include "models.h"

/*
 * Revolutions start in the positive z-axis (towards camera) and go
 * anti-clockwise 
 */

/*
 * Make a revolved piece 
 */
int revolve_line(float *trace_r, float *trace_h, float max_iheight)
{
  float theta, norm_theta, sin_theta, cos_theta;
  float norm_ptheta = 0.0f, sin_ptheta = 0.0f, cos_ptheta = 1.0f;
  float radius, pradius;
  float max_height = max_iheight, height, pheight;
  float dx, dy, len;
  int npoints, p;

  /*
   * Get the number of points 
   */
  /*
   * The list is terminated by (0, 0) 
   */
  for (npoints = 0;
       (trace_r[npoints] != 0.0f) || (trace_h[npoints] != 0.0f);
       npoints++);

  /*
   * If less than two points exist cannot revolve 
   */
  if (npoints < 2)
    return -1;

  /*
   * If the max_height hasn't been defined, find it 
   */
  if (max_height == 0.0f)
    for (p = 0; p < npoints; p++)
      if (max_height < trace_h[p])
	max_height = trace_h[p];

  /*
   * Draw the revolution 
   */
  for (theta = DTHETA; theta <= (2.0f * M_PI); theta += DTHETA)
  {
    sin_theta = sin(theta);
    cos_theta = cos(theta);
    norm_theta = theta / (2.0f * M_PI);
    pradius = trace_r[0];
    pheight = trace_h[0];

    for (p = 1; p < npoints; p++)
    {
      radius = trace_r[p];
      height = trace_h[p];

      /*
       * Get the normalized lengths of the normal vector 
       */
      dx = radius - pradius;
      dy = height - pheight;
      len = sqrt(dx * dx + dy * dy);
      dx /= len;
      dy /= len;

      /*
       * If only triangles required 
       */
      if (radius == 0.0f)
      {
	glBegin(GL_TRIANGLES);

	glNormal3f(dy * sin_ptheta, -dx, dy * cos_ptheta);
	glTexCoord2f(norm_ptheta, pheight / max_height);
	glVertex3f(pradius * sin_ptheta, pheight, pradius * cos_ptheta);

	glNormal3f(dy * sin_theta, -dx, dy * cos_theta);
	glTexCoord2f(norm_theta, pheight / max_height);
	glVertex3f(pradius * sin_theta, pheight, pradius * cos_theta);

	glTexCoord2f(0.5f * (norm_theta + norm_ptheta),
		     height / max_height);
	glVertex3f(0.0f, height, 0.0f);

	glEnd();
      } else
      {				/*
				 * quads required 
				 */

	glBegin(GL_QUADS);

	glNormal3f(dy * sin_ptheta, -dx, dy * cos_ptheta);
	glTexCoord2f(norm_ptheta, pheight / max_height);
	glVertex3f(pradius * sin_ptheta, pheight, pradius * cos_ptheta);

	glNormal3f(dy * sin_theta, -dx, dy * cos_theta);
	glTexCoord2f(norm_theta, pheight / max_height);
	glVertex3f(pradius * sin_theta, pheight, pradius * cos_theta);

	glTexCoord2f(norm_theta, height / max_height);
	glVertex3f(radius * sin_theta, height, radius * cos_theta);

	glNormal3f(dy * sin_ptheta, -dx, dy * cos_ptheta);
	glTexCoord2f(norm_ptheta, height / max_height);
	glVertex3f(radius * sin_ptheta, height, radius * cos_ptheta);

	glEnd();
      }

      pradius = radius;
      pheight = height;
    }

    sin_ptheta = sin_theta;
    cos_ptheta = cos_theta;
    norm_ptheta = norm_theta;
  }

  return 1;
}

void gen_model_lists(void)
{
  glNewList(1, GL_COMPILE);
  draw_king();
  glEndList();

  glNewList(2, GL_COMPILE);
  draw_queen();
  glEndList();

  glNewList(3, GL_COMPILE);
  draw_bishop();
  glEndList();

  glNewList(4, GL_COMPILE);
  draw_knight();
  glEndList();

  glNewList(5, GL_COMPILE);
  draw_rook();
  glEndList();

  glNewList(6, GL_COMPILE);
  draw_pawn();
  glEndList();
}

void draw_pawn(void)
{
  float trace_r[] =
      { 3.5f, 3.5f, 2.5f, 2.5f, 1.5f, 1.0f, 1.8f, 1.0f, 2.0f, 1.0f, 0.0f,
    0.0f
  };
  float trace_h[] =
      { 0.0f, 2.0f, 3.0f, 4.0f, 6.0f, 8.8f, 8.8f, 9.2f, 11.6f, 13.4f,
    13.4f, 0.0f
  };

  revolve_line(trace_r, trace_h, 0.0f);
}

void draw_rook(void)
{
  float trace_r[] =
      { 3.8f, 3.8f, 2.6f, 2.0f, 2.8f, 2.8f, 2.2f, 2.2f, 0.0f, 0.0f };
  float trace_h[] =
      { 0.0f, 2.0f, 5.0f, 10.2f, 10.2f, 13.6f, 13.6f, 13.0f, 13.0f,
    0.0f
  };

  revolve_line(trace_r, trace_h, 0.0f);
}

void draw_knight(void)
{
  float trace_r[] = { 4.1f, 4.1f, 2.0f, 2.0f, 2.6f, 0.0f };
  float trace_h[] = { 0.0f, 2.0f, 3.6f, 4.8f, 5.8f, 0.0f };
  float norm_theta = 0.0f;

  /*
   * Revolved base 
   */
  revolve_line(trace_r, trace_h, 17.8f);

  /*
   * Non revolved pieces 
   */
  /*
   * Quads 
   */
  glBegin(GL_QUADS);
  /*
   * Square base 
   */
  glNormal3f(0.0, -1.0, 0.0);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(2.6, 5.8, 2.6);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(-2.6, 5.8, 2.6);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(-2.6, 5.8, -0.8);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(2.6, 5.8, -0.8);

  /*
   * Upper edge of nose 
   */
  glNormal3f(0.0, 0.707107, 0.707107);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(0.8, 16.2, 4.0);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, 3.4);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, 3.4);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(-0.8, 16.2, 4.0);

  /*
   * Above head 
   */
  glNormal3f(0.0, 1.0, 0.0);

  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, 3.4);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, 3.0);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, 3.0);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, 3.4);

  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, 3.0);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(0.5, 16.8, 1.6);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-0.5, 16.8, 1.6);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, 3.0);

  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(0.5, 16.8, 1.6);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, 0.2);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, 0.2);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-0.5, 16.8, 1.6);

  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, 0.2);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, -0.2);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, -0.2);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, 0.2);

  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, -0.2);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(0.4, 16.8, -1.1);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-0.4, 16.8, -1.1);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, -0.2);

  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(0.4, 16.8, -1.1);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, -2.0);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, -2.0);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-0.4, 16.8, -1.1);

  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, -2.0);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, -4.4);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, -4.4);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, -2.0);

  /*
   * Back of head 
   */
  glNormal3f(0.0, 0.0, -1.0);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, -4.4);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, -4.4);
  glTexCoord2f(norm_theta, 15.0f / 17.8f);
  glVertex3f(1.0, 15.0, -4.4);
  glTexCoord2f(norm_theta, 15.0f / 17.8f);
  glVertex3f(-1.0, 15.0, -4.4);

  /*
   * Under back 
   */
  glNormal3f(0.0, 0.0, -1.0);
  glTexCoord2f(norm_theta, 15.0f / 17.8f);
  glVertex3f(-1.0, 15.0, -4.4);
  glTexCoord2f(norm_theta, 15.0f / 17.8f);
  glVertex3f(1.0, 15.0, -4.4);
  glTexCoord2f(norm_theta, 14.8f / 17.8f);
  glVertex3f(0.55, 14.8, -2.8);
  glTexCoord2f(norm_theta, 14.8f / 17.8f);
  glVertex3f(-0.55, 14.8, -2.8);

  /*
   * Right side of face 
   */
  glNormal3f(-0.933878, 0.128964, -0.333528);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, 3.0);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-0.5, 16.8, 1.6);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(-1.0, 14.0, 1.3);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-1.2, 13.8, 2.4);

  glNormal3f(-0.966676, 0.150427, 0.207145);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-0.5, 16.8, 1.6);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-0.8, 16.8, 0.2);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-1.2, 13.8, 0.2);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(-1.0, 14.0, 1.3);

  /*
   * (above and below eye) 
   */
  glNormal3f(-0.934057, 0.124541, -0.334704);

  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-0.82666667, 16.6, 0.2);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-0.8, 16.8, 0.2);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, -0.2);
  glTexCoord2f(norm_theta, 16.6f / 17.8f);
  glVertex3f(-1.0, 16.6, -0.38);

  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-1.2, 13.8, 0.2);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(-0.88, 16.2, 0.2);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(-1.0, 16.2, -0.74);
  glTexCoord2f(norm_theta, 13.6f / 17.8f);
  glVertex3f(-1.2, 13.6, -0.2);

  glTexCoord2f(norm_theta, 13.6f / 17.8f);
  glVertex3f(-1.2, 13.6, -0.2);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(-1.0, 16.2, -0.74);
  glTexCoord2f(norm_theta, 15.8f / 17.8f);
  glVertex3f(-1.0, 15.8, -1.1);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(-0.6, 14.0, -1.4);

  glNormal3f(-0.970801, -0.191698, -0.144213);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, -2.0);
  glTexCoord2f(norm_theta, 14.8f / 17.8f);
  glVertex3f(-0.55, 14.8, -2.8);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(-0.6, 14.0, -1.4);
  glTexCoord2f(norm_theta, 15.8f / 17.8f);
  glVertex3f(-1.0, 15.8, -1.1);

  glNormal3f(-0.975610, 0.219512, 0.0);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, -2.0);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, -4.4);
  glTexCoord2f(norm_theta, 15.0f / 17.8f);
  glVertex3f(-1.0, 15.0, -4.4);
  glTexCoord2f(norm_theta, 14.8f / 17.8f);
  glVertex3f(-0.55, 14.8, -2.8);

  /*
   * Left side of face 
   */
  glNormal3f(0.933878, 0.128964, -0.333528);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, 3.0);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(1.2, 13.8, 2.4);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(1.0, 14.0, 1.3);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(0.5, 16.8, 1.6);

  glNormal3f(0.966676, 0.150427, 0.207145);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(0.5, 16.8, 1.6);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(1.0, 14.0, 1.3);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(1.2, 13.8, 0.2);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(0.8, 16.8, 0.2);

  /*
   * (above and below eye) 
   */
  glNormal3f(0.934057, 0.124541, -0.334704);

  glTexCoord2f(norm_theta, 16.6f / 17.8f);
  glVertex3f(0.82666667, 16.6, 0.2);
  glTexCoord2f(norm_theta, 16.6f / 17.8f);
  glVertex3f(1.0, 16.6, -0.38);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, -0.2);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(0.8, 16.8, 0.2);

  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(1.2, 13.8, 0.2);
  glTexCoord2f(norm_theta, 13.6f / 17.8f);
  glVertex3f(1.2, 13.6, -0.2);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(1.0, 16.2, -0.74);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(0.88, 16.2, 0.2);

  glTexCoord2f(norm_theta, 13.6f / 17.8f);
  glVertex3f(1.2, 13.6, -0.2);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(0.6, 14.0, -1.4);
  glTexCoord2f(norm_theta, 15.8f / 17.8f);
  glVertex3f(1.0, 15.8, -1.1);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(1.0, 16.2, -0.74);

  glNormal3f(0.970801, -0.191698, -0.144213);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, -2.0);
  glTexCoord2f(norm_theta, 15.8f / 17.8f);
  glVertex3f(1.0, 15.8, -1.1);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(0.6, 14.0, -1.4);
  glTexCoord2f(norm_theta, 14.8f / 17.8f);
  glVertex3f(0.55, 14.8, -2.8);

  glNormal3f(0.975610, -0.219512, 0.0);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, -2.0);
  glTexCoord2f(norm_theta, 14.8f / 17.8f);
  glVertex3f(0.55, 14.8, -2.8);
  glTexCoord2f(norm_theta, 15.0f / 17.8f);
  glVertex3f(1.0, 15.0, -4.4);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, -4.4);

  /*
   * Eyes 
   */
  glNormal3f(0.598246, 0.797665, 0.076372);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(0.88, 16.2, 0.2);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(1.0, 16.2, -0.74);
  glTexCoord2f(norm_theta, 16.4f / 17.8f);
  glVertex3f(0.8, 16.4, -0.56);
  glTexCoord2f(norm_theta, 16.4f / 17.8f);
  glVertex3f(0.61333334, 16.4, 0.2);

  glNormal3f(0.670088, -0.714758, 0.200256);
  glTexCoord2f(norm_theta, 16.4f / 17.8f);
  glVertex3f(0.61333334, 16.4, 0.2);
  glTexCoord2f(norm_theta, 16.4f / 17.8f);
  glVertex3f(0.8, 16.4, -0.56);
  glTexCoord2f(norm_theta, 16.6f / 17.8f);
  glVertex3f(1.0, 16.6, -0.38);
  glTexCoord2f(norm_theta, 16.6f / 17.8f);
  glVertex3f(0.82666667, 16.6, 0.2);

  glNormal3f(-0.598246, 0.797665, 0.076372);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(-0.88, 16.2, 0.2);
  glTexCoord2f(norm_theta, 16.4f / 17.8f);
  glVertex3f(-0.61333334, 16.4, 0.2);
  glTexCoord2f(norm_theta, 16.4f / 17.8f);
  glVertex3f(-0.8, 16.4, -0.56);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(-1.0, 16.2, -0.74);

  glNormal3f(-0.670088, -0.714758, 0.200256);
  glTexCoord2f(norm_theta, 16.4f / 17.8f);
  glVertex3f(-0.61333334, 16.4, 0.2);
  glTexCoord2f(norm_theta, 16.6f / 17.8f);
  glVertex3f(-0.82666667, 16.6, 0.2);
  glTexCoord2f(norm_theta, 16.6f / 17.8f);
  glVertex3f(-1.0, 16.6, -0.38);
  glTexCoord2f(norm_theta, 16.4f / 17.8f);
  glVertex3f(-0.8, 16.4, -0.56);

  /*
   * Hair 
   */
  glNormal3f(0.0, 1.0, 0.0);
  glTexCoord2f(norm_theta, 17.8f / 17.8f);
  glVertex3f(0.35, 17.8, -0.8);
  glTexCoord2f(norm_theta, 17.8f / 17.8f);
  glVertex3f(0.35, 17.8, -4.4);
  glTexCoord2f(norm_theta, 17.8f / 17.8f);
  glVertex3f(-0.35, 17.8, -4.4);
  glTexCoord2f(norm_theta, 17.8f / 17.8f);
  glVertex3f(-0.35, 17.8, -0.8);

  glNormal3f(1.0, 0.0, 0.0);
  glTexCoord2f(norm_theta, 17.8f / 17.8f);
  glVertex3f(0.35, 17.8, -0.8);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(0.35, 16.8, -0.8);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(0.35, 16.8, -4.4);
  glTexCoord2f(norm_theta, 17.8f / 17.8f);
  glVertex3f(0.35, 17.8, -4.4);

  glNormal3f(-1.0, 0.0, 0.0);
  glTexCoord2f(norm_theta, 17.8f / 17.8f);
  glVertex3f(-0.35, 17.8, -0.8);
  glTexCoord2f(norm_theta, 17.8f / 17.8f);
  glVertex3f(-0.35, 17.8, -4.4);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-0.35, 16.8, -4.4);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-0.35, 16.8, -0.8);

  glNormal3f(0.0, 0.0, 1.0);
  glTexCoord2f(norm_theta, 17.8f / 17.8f);
  glVertex3f(0.35, 17.8, -0.8);
  glTexCoord2f(norm_theta, 17.8f / 17.8f);
  glVertex3f(-0.35, 17.8, -0.8);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-0.35, 16.8, -0.8);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(0.35, 16.8, -0.8);

  glNormal3f(0.0, 0.0, -1.0);
  glTexCoord2f(norm_theta, 17.8f / 17.8f);
  glVertex3f(0.35, 17.8, -4.4);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(0.35, 16.8, -4.4);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-0.35, 16.8, -4.4);
  glTexCoord2f(norm_theta, 17.8f / 17.8f);
  glVertex3f(-0.35, 17.8, -4.4);

  /*
   * Under chin 
   */
  glNormal3f(0.0, -0.853282, 0.521450);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(-1.0, 14.0, 1.3);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-1.2, 13.8, 0.2);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(1.2, 13.8, 0.2);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(1.0, 14.0, 1.3);

  glNormal3f(0.0, -0.983870, -0.178885);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(-1.0, 14.0, 1.3);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(1.0, 14.0, 1.3);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(1.2, 13.8, 2.4);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-1.2, 13.8, 2.4);

  /*
   * Mane 
   */
  /*
   * Right 
   */
  glNormal3f(-0.788443, 0.043237, -0.613587);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(-2.6, 5.8, -0.8);
  glTexCoord2f(norm_theta, 14.8f / 17.8f);
  glVertex3f(-0.55, 14.8, -2.8);
  glTexCoord2f(norm_theta, 15.0f / 17.8f);
  glVertex3f(0.0, 15.0, -3.6);
  glTexCoord2f(norm_theta, 7.8f / 17.8f);
  glVertex3f(0.0, 7.8, -4.0);

  /*
   * Left 
   */
  glNormal3f(0.788443, 0.043237, -0.613587);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(2.6, 5.8, -0.8);
  glTexCoord2f(norm_theta, 7.8f / 17.8f);
  glVertex3f(0.0, 7.8, -4.0);
  glTexCoord2f(norm_theta, 15.0f / 17.8f);
  glVertex3f(0.0, 15.0, -3.6);
  glTexCoord2f(norm_theta, 14.8f / 17.8f);
  glVertex3f(0.55, 14.8, -2.8);

  /*
   * Chest 
   */
  /*
   * Front 
   */
  glNormal3f(0.0, 0.584305, 0.811534);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-0.5, 13.8, 0.4);
  glTexCoord2f(norm_theta, 8.8f / 17.8f);
  glVertex3f(-2.0, 8.8, 4.0);
  glTexCoord2f(norm_theta, 8.8f / 17.8f);
  glVertex3f(2.0, 8.8, 4.0);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(0.5, 13.8, 0.4);

  /*
   * Bottom 
   */
  glNormal3f(0.0, -0.422886, 0.906183);
  glTexCoord2f(norm_theta, 8.8f / 17.8f);
  glVertex3f(-2.0, 8.8, 4.0);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(-2.6, 5.8, 2.6);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(2.6, 5.8, 2.6);
  glTexCoord2f(norm_theta, 8.8f / 17.8f);
  glVertex3f(2.0, 8.8, 4.0);

  /*
   * Right 
   */
  glNormal3f(-0.969286, 0.231975, -0.081681);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-0.5, 13.8, 0.4);
  glTexCoord2f(norm_theta, 12.2f / 17.8f);
  glVertex3f(-1.4, 12.2, -0.4);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(-2.6, 5.8, 2.6);
  glTexCoord2f(norm_theta, 8.8f / 17.8f);
  glVertex3f(-2.0, 8.8, 4.0);

  glNormal3f(-0.982872, 0.184289, 0.0);
  glTexCoord2f(norm_theta, 12.2f / 17.8f);
  glVertex3f(-1.4, 12.2, -0.4);
  glTexCoord2f(norm_theta, 12.2f / 17.8f);
  glVertex3f(-1.1422222222, 12.2, -2.2222222222);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(-2.6, 5.8, -0.8);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(-2.6, 5.8, 2.6);

  glTexCoord2f(norm_theta, 14.8f / 17.8f);
  glVertex3f(-0.55, 14.8, -2.8);
  glTexCoord2f(norm_theta, 12.2f / 17.8f);
  glVertex3f(-1.1422222222, 12.2, -2.2222222222);
  glTexCoord2f(norm_theta, 12.2f / 17.8f);
  glVertex3f(-1.4, 12.2, -0.4);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(-0.6, 14.0, -1.4);

  /*
   * Left 
   */
  glNormal3f(0.969286, 0.231975, -0.081681);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(0.5, 13.8, 0.4);
  glTexCoord2f(norm_theta, 8.8f / 17.8f);
  glVertex3f(2.0, 8.8, 4.0);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(2.6, 5.8, 2.6);
  glTexCoord2f(norm_theta, 12.2f / 17.8f);
  glVertex3f(1.4, 12.2, -0.4);

  glNormal3f(0.982872, 0.184289, 0.0);
  glTexCoord2f(norm_theta, 12.2f / 17.8f);
  glVertex3f(1.4, 12.2, -0.4);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(2.6, 5.8, 2.6);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(2.6, 5.8, -0.8);
  glTexCoord2f(norm_theta, 12.2f / 17.8f);
  glVertex3f(1.1422222222, 12.2, -2.2222222222);

  glTexCoord2f(norm_theta, 14.8f / 17.8f);
  glVertex3f(0.55, 14.8, -2.8);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(0.6, 14.0, -1.4);
  glTexCoord2f(norm_theta, 12.2f / 17.8f);
  glVertex3f(1.4, 12.2, -0.4);
  glTexCoord2f(norm_theta, 12.2f / 17.8f);
  glVertex3f(1.1422222222, 12.2, -2.2222222222);
  glEnd();

  /*
   * Triangles 
   */
  glBegin(GL_TRIANGLES);
  /*
   * Base 
   */
  glNormal3f(0.0, -1.0, 0.0);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(1.44, 5.8, -2.6);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(-2.6, 5.8, -0.8);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(-1.44, 5.8, -2.6);

  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(1.44, 5.8, -2.6);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(2.6, 5.8, -0.8);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(-2.6, 5.8, -0.8);

  /*
   * Under mane 
   */
  glNormal3f(0.819890, -0.220459, -0.528373);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(2.6, 5.8, -0.8);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(1.44, 5.8, -2.6);
  glTexCoord2f(norm_theta, 7.8f / 17.8f);
  glVertex3f(0.0, 7.8, -4.0);

  glNormal3f(0.0, -0.573462, -0.819232);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(1.44, 5.8, -2.6);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(-1.44, 5.8, -2.6);
  glTexCoord2f(norm_theta, 7.8f / 17.8f);
  glVertex3f(0.0, 7.8, -4.0);

  glNormal3f(-0.819890, -0.220459, -0.528373);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(-2.6, 5.8, -0.8);
  glTexCoord2f(norm_theta, 7.8f / 17.8f);
  glVertex3f(0.0, 7.8, -4.0);
  glTexCoord2f(norm_theta, 5.8f / 17.8f);
  glVertex3f(-1.44, 5.8, -2.6);

  /*
   * Nose tip 
   */
  glNormal3f(0.0, 0.0, 1.0);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(0.0, 14.0, 4.0);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(0.8, 16.2, 4.0);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(-0.8, 16.2, 4.0);

  /*
   * Mouth left 
   */
  glNormal3f(-0.752714, -0.273714, 0.598750);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(0.0, 14.0, 4.0);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(-0.8, 16.2, 4.0);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-1.2, 13.8, 2.4);

  glNormal3f(-0.957338, 0.031911, 0.287202);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-1.2, 13.8, 2.4);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(-0.8, 16.2, 4.0);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, 3.4);

  glNormal3f(-0.997785, 0.066519, 0.0);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-1.2, 13.8, 2.4);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, 3.4);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, 3.0);

  /*
   * Mouth right 
   */
  glNormal3f(0.752714, -0.273714, 0.598750);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(0.0, 14.0, 4.0);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(1.2, 13.8, 2.4);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(0.8, 16.2, 4.0);

  glNormal3f(0.957338, 0.031911, 0.287202);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(1.2, 13.8, 2.4);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, 3.4);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(0.8, 16.2, 4.0);

  glNormal3f(0.997785, 0.066519, 0.0);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(1.2, 13.8, 2.4);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, 3.0);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, 3.4);

  /*
   * Under nose 
   */
  glNormal3f(0.0, -0.992278, 0.124035);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(0.0, 14.0, 4.0);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-1.2, 13.8, 2.4);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(1.2, 13.8, 2.4);

  /*
   * Neck indents 
   */
  glNormal3f(-0.854714, 0.484047, 0.187514);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(-0.6, 14.0, -1.4);
  glTexCoord2f(norm_theta, 12.2f / 17.8f);
  glVertex3f(-1.4, 12.2, -0.4);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-0.45, 13.8, -0.2);

  glNormal3f(-0.853747, 0.515805, -0.071146);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-0.45, 13.8, -0.2);
  glTexCoord2f(norm_theta, 12.2f / 17.8f);
  glVertex3f(-1.4, 12.2, -0.4);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-0.5, 13.8, 0.4);

  glNormal3f(0.854714, 0.484047, 0.187514);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(0.6, 14.0, -1.4);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(0.45, 13.8, -0.2);
  glTexCoord2f(norm_theta, 12.2f / 17.8f);
  glVertex3f(1.4, 12.2, -0.4);

  glNormal3f(0.853747, 0.515805, -0.071146);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(0.45, 13.8, -0.2);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(0.5, 13.8, 0.4);
  glTexCoord2f(norm_theta, 12.2f / 17.8f);
  glVertex3f(1.4, 12.2, -0.4);

  /*
   * Under chin 
   */
  glNormal3f(0.252982, -0.948683, -0.189737);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(0.6, 14.0, -1.4);
  glTexCoord2f(norm_theta, 13.6f / 17.8f);
  glVertex3f(1.2, 13.6, -0.2);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(0.45, 13.8, -0.2);

  glNormal3f(0.257603, -0.966012, 0.021467);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(0.5, 13.8, 0.4);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(0.45, 13.8, -0.2);
  glTexCoord2f(norm_theta, 13.6f / 17.8f);
  glVertex3f(1.2, 13.6, -0.2);

  glNormal3f(0.126745, -0.887214, 0.443607);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(0.5, 13.8, 0.4);
  glTexCoord2f(norm_theta, 13.6f / 17.8f);
  glVertex3f(1.2, 13.6, -0.2);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(1.2, 13.8, 0.2);

  glNormal3f(-0.252982, -0.948683, -0.189737);
  glTexCoord2f(norm_theta, 14.0f / 17.8f);
  glVertex3f(-0.6, 14.0, -1.4);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-0.45, 13.8, -0.2);
  glTexCoord2f(norm_theta, 13.6f / 17.8f);
  glVertex3f(-1.2, 13.6, -0.2);

  glNormal3f(-0.257603, -0.966012, 0.021467);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-0.5, 13.8, 0.4);
  glTexCoord2f(norm_theta, 13.6f / 17.8f);
  glVertex3f(-1.2, 13.6, -0.2);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-0.45, 13.8, -0.2);

  glNormal3f(-0.126745, -0.887214, 0.443607);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-0.5, 13.8, 0.4);
  glTexCoord2f(norm_theta, 13.8f / 17.8f);
  glVertex3f(-1.2, 13.8, 0.2);
  glTexCoord2f(norm_theta, 13.6f / 17.8f);
  glVertex3f(-1.2, 13.6, -0.2);

  /*
   * Eyes 
   */
  glNormal3f(0.0, 0.0, -1.0);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(0.88, 16.2, 0.2);
  glTexCoord2f(norm_theta, 16.4f / 17.8f);
  glVertex3f(0.61333334, 16.4, 0.2);
  glTexCoord2f(norm_theta, 16.6f / 17.8f);
  glVertex3f(0.82666667, 16.6, 0.2);

  glNormal3f(0.000003, -0.668965, 0.743294);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(1.0, 16.2, -0.74);
  glTexCoord2f(norm_theta, 16.6f / 17.8f);
  glVertex3f(1.0, 16.6, -0.38);
  glTexCoord2f(norm_theta, 16.4f / 17.8f);
  glVertex3f(0.8, 16.4, -0.56);

  glNormal3f(0.0, 0.0, -1.0);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(-0.88, 16.2, 0.2);
  glTexCoord2f(norm_theta, 16.6f / 17.8f);
  glVertex3f(-0.82666667, 16.6, 0.2);
  glTexCoord2f(norm_theta, 16.4f / 17.8f);
  glVertex3f(-0.61333334, 16.4, 0.2);

  glNormal3f(-0.000003, -0.668965, 0.743294);
  glTexCoord2f(norm_theta, 16.2f / 17.8f);
  glVertex3f(-1.0, 16.2, -0.74);
  glTexCoord2f(norm_theta, 16.4f / 17.8f);
  glVertex3f(-0.8, 16.4, -0.56);
  glTexCoord2f(norm_theta, 16.6f / 17.8f);
  glVertex3f(-1.0, 16.6, -0.38);

  /*
   * Behind eyes 
   */
  /*
   * Right 
   */
  glNormal3f(-0.997484, 0.070735, 0.004796);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-0.4, 16.8, -1.1);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, -2.0);
  glTexCoord2f(norm_theta, 15.8f / 17.8f);
  glVertex3f(-1.0, 15.8, -1.1);

  glNormal3f(-0.744437, 0.446663, -0.496292);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-1.0, 16.8, -0.2);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(-0.4, 16.8, -1.1);
  glTexCoord2f(norm_theta, 15.8f / 17.8f);
  glVertex3f(-1.0, 15.8, -1.1);

  /*
   * Left 
   */
  glNormal3f(0.997484, 0.070735, 0.004796);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(0.4, 16.8, -1.1);
  glTexCoord2f(norm_theta, 15.8f / 17.8f);
  glVertex3f(1.0, 15.8, -1.1);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, -2.0);

  glNormal3f(0.744437, 0.446663, -0.496292);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(1.0, 16.8, -0.2);
  glTexCoord2f(norm_theta, 15.8f / 17.8f);
  glVertex3f(1.0, 15.8, -1.1);
  glTexCoord2f(norm_theta, 16.8f / 17.8f);
  glVertex3f(0.4, 16.8, -1.1);

  glEnd();
}

void draw_bishop(void)
{
  float trace_r[] =
      { 4.0f, 4.0f, 2.5f, 2.5f, 1.5f, 1.2f, 2.5f, 1.7f, 1.7f, 2.2f, 2.2f,
    1.0f, 0.8f, 1.2f, 0.8f, 0.0f, 0.0f
  };
  float trace_h[] =
      { 0.0f, 2.0f, 3.0f, 4.0f, 7.0f, 9.4f, 9.4f, 11.0f, 12.2f, 13.2f,
    14.8f, 16.0f, 17.0f, 17.7f, 18.4f, 18.4f, 0.0f
  };

  revolve_line(trace_r, trace_h, 0.0f);
}

void draw_queen(void)
{
  float trace_r[] =
      { 4.8f, 4.8f, 3.4f, 3.4f, 1.8f, 1.4f, 2.9f, 1.8f, 1.8f, 2.0f, 2.7f,
    2.4f, 1.7f, 0.95f, 0.7f, 0.9f, 0.7f, 0.0f, 0.0f
  };
  float trace_h[] =
      { 0.0f, 2.2f, 4.0f, 5.0f, 8.0f, 11.8f, 11.8f, 13.6f, 15.2f, 17.8f,
    19.2f, 20.0f, 20.0f, 20.8f, 20.8f, 21.4f, 22.0f, 22.0f, 0.0f
  };

  revolve_line(trace_r, trace_h, 0.0f);
}

void draw_king(void)
{
  float trace_r[] =
      { 5.0f, 5.0f, 3.5f, 3.5f, 2.0f, 1.4f, 3.0f, 2.0f, 2.0f, 2.8f, 1.6f,
    1.6f, 0.0f, 0.0f
  };
  float trace_h[] =
      { 0.0f, 2.0f, 3.0f, 4.6f, 7.6f, 12.6f, 12.6f, 14.6f, 15.6f, 19.1f,
    19.7f, 20.1f, 20.1f, 0.0f
  };

  revolve_line(trace_r, trace_h, 0.0f);

  glBegin(GL_QUADS);

  /*
   * Cross front 
   */
  glNormal3f(0.0, 0.0, 1.0);

  glVertex3f(-0.3, 20.1, 0.351);
  glVertex3f(0.3, 20.1, 0.35);
  glVertex3f(0.3, 23.1, 0.35);
  glVertex3f(-0.3, 23.1, 0.35);

  glVertex3f(-0.9, 21.1, 0.35);
  glVertex3f(-0.3, 21.1, 0.35);
  glVertex3f(-0.3, 22.1, 0.35);
  glVertex3f(-0.9, 22.1, 0.35);

  glVertex3f(0.9, 21.1, 0.35);
  glVertex3f(0.9, 22.1, 0.35);
  glVertex3f(0.3, 22.1, 0.35);
  glVertex3f(0.3, 21.1, 0.35);

  /*
   * Cross back 
   */
  glNormal3f(0.0, 0.0, -1.0);

  glVertex3f(0.3, 20.1, -0.35);
  glVertex3f(-0.3, 20.1, -0.35);
  glVertex3f(-0.3, 23.1, -0.35);
  glVertex3f(0.3, 23.1, -0.35);

  glVertex3f(-0.3, 21.1, -0.35);
  glVertex3f(-0.9, 21.1, -0.35);
  glVertex3f(-0.9, 22.1, -0.35);
  glVertex3f(-0.3, 22.1, -0.35);

  glVertex3f(0.3, 21.1, -0.35);
  glVertex3f(0.3, 22.1, -0.35);
  glVertex3f(0.9, 22.1, -0.35);
  glVertex3f(0.9, 21.1, -0.35);

  /*
   * Cross left 
   */
  glNormal3f(-1.0, 0.0, 0.0);

  glVertex3f(-0.9, 21.1, 0.35);
  glVertex3f(-0.9, 22.1, 0.35);
  glVertex3f(-0.9, 22.1, -0.35);
  glVertex3f(-0.9, 21.1, -0.35);

  glVertex3f(-0.3, 20.1, 0.35);
  glVertex3f(-0.3, 21.1, 0.35);
  glVertex3f(-0.3, 21.1, -0.35);
  glVertex3f(-0.3, 20.1, -0.35);

  glVertex3f(-0.3, 22.1, 0.3);
  glVertex3f(-0.3, 23.1, 0.3);
  glVertex3f(-0.3, 23.1, -0.3);
  glVertex3f(-0.3, 22.1, -0.3);

  /*
   * Cross right 
   */
  glNormal3f(1.0, 0.0, 0.0);

  glVertex3f(0.9, 21.1, -0.35);
  glVertex3f(0.9, 22.1, -0.35);
  glVertex3f(0.9, 22.1, 0.35);
  glVertex3f(0.9, 21.1, 0.35);

  glVertex3f(0.3, 20.1, -0.35);
  glVertex3f(0.3, 21.1, -0.35);
  glVertex3f(0.3, 21.1, 0.35);
  glVertex3f(0.3, 20.1, 0.35);

  glVertex3f(0.3, 22.1, -0.35);
  glVertex3f(0.3, 23.1, -0.35);
  glVertex3f(0.3, 23.1, 0.35);
  glVertex3f(0.3, 22.1, 0.35);

  /*
   * Cross bottom 
   */
  glNormal3f(0.0, -1.0, 0.0);

  glVertex3f(-0.9, 21.1, 0.35);
  glVertex3f(-0.9, 21.1, -0.35);
  glVertex3f(-0.3, 21.1, -0.35);
  glVertex3f(-0.3, 21.1, 0.35);

  glVertex3f(0.3, 21.1, -0.35);
  glVertex3f(0.9, 21.1, -0.35);
  glVertex3f(0.9, 21.1, 0.35);
  glVertex3f(0.3, 21.1, 0.35);

  /*
   * Cross top 
   */
  glNormal3f(0.0, 1.0, 0.0);

  glVertex3f(-0.9, 22.1, -0.35);
  glVertex3f(-0.9, 22.1, 0.35);
  glVertex3f(-0.3, 22.1, 0.35);
  glVertex3f(-0.3, 22.1, -0.35);

  glVertex3f(0.3, 22.1, -0.35);
  glVertex3f(0.3, 22.1, 0.35);
  glVertex3f(0.9, 22.1, 0.35);
  glVertex3f(0.9, 22.1, -0.35);

  glVertex3f(-0.3, 23.1, -0.35);
  glVertex3f(-0.3, 23.1, 0.35);
  glVertex3f(0.3, 23.1, 0.35);
  glVertex3f(0.3, 23.1, -0.35);

  glEnd();
}
