/*
 * glChess - A 3D chess interface
 *
 * Copyright (C) 2001  Robert  Ancell <bob27@users.sourceforge.net>
 *                     Michael Duelli <duelli@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <GL/gl.h>

#include "global.h"
#include "menu.h"
#include "preferences.h"
#include "draw.h"
#include "splash.h"
#include "player.h"
#include "interface.h"
#include "config.h"

#include "engine.h"

extern int engine;
extern int debug;

GtkWidget *generate_menus(glChessWidget * glcwidget,
			  GtkWidget ** view_radio, GtkWidget ** fps_radio)
{
  Game *game = glcwidget->game;

  /*
   * The menus: 
   */
  static GtkItemFactoryEntry menu_items[] = {
    {"/_Game", NULL, NULL, 0, "<Branch>"},
    {"/_Game/tear", NULL, NULL, 0, "<Tearoff>"},
    {"/_Game/_Play a game against...", NULL, NULL, 0, "<Branch>"},
    {"/_Game/_Play a game against.../tear", NULL, NULL, 0, "<Tearoff>"},
    {"/_Game/_Play a game against.../_Crafty", "C", start_game, 1, NULL},

    {"/_Game/_Play a game against.../_GNU Chess 4", "G", start_game, 2,
     NULL},
    {"/_Game/_Play a game against.../_Yourself (no engine)", "Y",
     start_game, 3, NULL},
    {"/_Game/_Resign", "<control>R", resign_game, 1, NULL},
    {"/_Game/sep1", NULL, NULL, 0, "<Separator>"},
    {"/_Game/_Quit", "<control>Q", quit_program, 1, NULL},
    {"/_Settings", NULL, NULL, 0, "<Branch>"},
    {"/_Settings/tear", NULL, NULL, 0, "<Tearoff>"},
    {"/_Settings/_Preferences...", "<control>F", preferences, 1, NULL},
    {"/_Settings/_Save Settings", NULL, save_settings, 1, NULL},
    {"/_View", NULL, NULL, 0, "<Branch>"},
    {"/_View/tear", NULL, NULL, 0, "<Tearoff>"},
    {"/_View/_Paused", "<control>P", toggle_pause, 1, "<CheckItem>"},
    {"/_View/_Mode", NULL, NULL, 0, "<Branch>"},
    {"/_View/_Mode/tear", NULL, NULL, 0, "<Tearoff>"},
    {"/_View/_Mode/_Free", "1", set_view_mode, MODE_FREE,
     "<RadioItem>"},
    {"/_View/_Mode/_Rotating", "2", set_view_mode, MODE_ROTATING,
     "/View/Mode/Free"},
    {"/_View/_Mode/_Ortho", "3", set_view_mode, MODE_ORTHO,
     "/View/Mode/Free"},
    {"/_View/_Mode/_Trackball", "4", set_view_mode, MODE_TRACKBALL,
     "/View/Mode/Free"},
    {"/_View/Max _FPS", NULL, NULL, 0, "<Branch>"},
    {"/_View/Max _FPS/tear", NULL, NULL, 0, "<Tearoff>"},
    {"/_View/Max _FPS/_1 fps", NULL, set_max_fps, 1, "<RadioItem>"},
    {"/_View/Max _FPS/_2 fps", NULL, set_max_fps, 2,
     "/View/Max FPS/1 fps"},
    {"/_View/Max _FPS/_4 fps", NULL, set_max_fps, 4,
     "/View/Max FPS/1 fps"},
    {"/_View/Max _FPS/_6 fps", NULL, set_max_fps, 6,
     "/View/Max FPS/1 fps"},
    {"/_View/Max _FPS/_8 fps", NULL, set_max_fps, 8,
     "/View/Max FPS/1 fps"},
    {"/_View/Max _FPS/_10 fps", NULL, set_max_fps, 10,
     "/View/Max FPS/1 fps"},
    {"/_View/Max _FPS/_15 fps", NULL, set_max_fps, 15,
     "/View/Max FPS/1 fps"},
    {"/_View/Max _FPS/_20 fps", NULL, set_max_fps, 20,
     "/View/Max FPS/1 fps"},
    {"/_View/Max _FPS/_30 fps", NULL, set_max_fps, 30,
     "/View/Max FPS/1 fps"},
    {"/_View/Max _FPS/_Unlimited", NULL, set_max_fps, 666,
     "/View/Max FPS/1 fps"},
    {"/_View/sep1", NULL, NULL, 0, "<Separator>"},
    {"/_View/_Lighting", "L", toggle_lighting, 1, "<CheckItem>"},
    {"/_View/_Shading", "H", toggle_shading, 1, "<CheckItem>"},
    {"/_View/_Reflections", "J", toggle_reflect, 1, "<CheckItem>"},
    {"/_View/_Textures", "K", toggle_texture, 1, "<CheckItem>"},
    {"/_View/_Fog", "M", toggle_fog, 1, "<CheckItem>"},
    {"/_View/Show _Coords", "N", toggle_coords, 1, "<CheckItem>"},
    {"/_Engine", NULL, NULL, 0, "<Branch>"},
    {"/_Engine/tear", NULL, NULL, 0, "<Tearoff>"},
    {"/_Engine/Beep on illegal move", NULL, toggle_beep, 0, "<CheckItem>"},
    
	{"/_Engine/Debug engine on console", NULL, toggle_debug, 0,
     "<CheckItem>"},
    {"/_Help", NULL, NULL, 0, "<LastBranch>"},
    {"/_Help/tear", NULL, NULL, 0, "<Tearoff>"},
    {"/_Help/_About...", NULL, help_about, 0, NULL},
  };

  GtkAccelGroup *accel_group;
  gint nmenu_items = sizeof(menu_items) / sizeof(menu_items[0]);

  GtkWidget *check_item = NULL;

  /*
   * Make the item factory 
   */
  accel_group = gtk_accel_group_new();
  glcwidget->item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR,
						 "<main>", accel_group);

  /*
   * Convert the itemfactory to menu items 
   */
  gtk_item_factory_create_items(glcwidget->item_factory,
				nmenu_items, menu_items, glcwidget);

  /*
   * Set the defaults on the view options 
   */
  check_item =
      gtk_item_factory_get_widget(glcwidget->item_factory,
				  "/View/Lighting");
  if (game->is_light)
    GTK_CHECK_MENU_ITEM(check_item)->active = TRUE;
  check_item =
      gtk_item_factory_get_widget(glcwidget->item_factory,
				  "/View/Shading");
  if (game->is_smooth)
    GTK_CHECK_MENU_ITEM(check_item)->active = TRUE;
  check_item =
      gtk_item_factory_get_widget(glcwidget->item_factory,
				  "/View/Reflections");
  if (game->is_reflect)
    GTK_CHECK_MENU_ITEM(check_item)->active = TRUE;
  check_item =
      gtk_item_factory_get_widget(glcwidget->item_factory,
				  "/View/Textures");
  if (game->is_texture)
    GTK_CHECK_MENU_ITEM(check_item)->active = TRUE;
  check_item =
      gtk_item_factory_get_widget(glcwidget->item_factory, "/View/Fog");
  if (game->is_fog)
    GTK_CHECK_MENU_ITEM(check_item)->active = TRUE;
  check_item =
      gtk_item_factory_get_widget(glcwidget->item_factory,
				  "/View/Show Coords");
  if (game->is_coord)
    GTK_CHECK_MENU_ITEM(check_item)->active = TRUE;
  switch (game->camera->mode)
  {
  case MODE_FREE:
    check_item =
	gtk_item_factory_get_widget(glcwidget->item_factory,
				    "/View/Mode/Free");
    break;
  case MODE_ROTATING:
    check_item =
	gtk_item_factory_get_widget(glcwidget->item_factory,
				    "/View/Mode/Rotating");
    break;
  case MODE_ORTHO:
    check_item =
	gtk_item_factory_get_widget(glcwidget->item_factory,
				    "/View/Mode/Ortho");
    break;
  case MODE_TRACKBALL:
    check_item =
	gtk_item_factory_get_widget(glcwidget->item_factory,
				    "/View/Mode/Trackball");
    break;
  default:
    break;
  }

  /*
   * The following doesn't work if you're using NVIDIA's shit drivers --
   * there is no logic to this of course 
   */
#ifndef NVIDIAS_DRIVERS_ARE_CRAP
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(check_item), TRUE);
#endif

  check_item = gtk_item_factory_get_widget(glcwidget->item_factory,
					   "/Engine/Beep on illegal move");
  if (game->cecp.beep_signal)
    GTK_CHECK_MENU_ITEM(check_item)->active = TRUE;

  check_item = gtk_item_factory_get_widget(glcwidget->item_factory,
					   "/Engine/Debug engine on console");
  if (debug)
    GTK_CHECK_MENU_ITEM(check_item)->active = TRUE;


  check_item = gtk_item_factory_get_widget(glcwidget->item_factory,
					   "/View/Max FPS/Unlimited");
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(check_item), TRUE);

  /*
   * Add the shortcuts 
   */
  gtk_window_add_accel_group(GTK_WINDOW(glcwidget->window), accel_group);

  /*
   * Return the root menu 
   */
  return gtk_item_factory_get_widget(glcwidget->item_factory, "<main>");
}

/* Starts the game */
void start_game(glChessWidget * glcwidget, guint num, GtkWidget * widget)
{
  end_splash(widget, glcwidget);

  engine = num;

  /* Start a game */
  init_game(glcwidget);
}

/*
 * Finish a game 
 */
void resign_game(glChessWidget * glcwidget, guint num, GtkWidget * widget)
{
  end_game(glcwidget);
}

/* Quits the program */
void quit_program(glChessWidget * glcwidget, guint num, GtkWidget * widget)
{
  /* FIXME: add a quit warning here */

  close_cecp_interface();

  gtk_main_quit();
}

void toggle_pause(glChessWidget * glcwidget, guint num, GtkWidget * widget)
{
  Game *game = glcwidget->game;

  game->is_paused = !game->is_paused;
}

/*
 * Saves the current settings to ~/.glchessrc 
 */
void
save_settings(glChessWidget * glcwidget, guint num, GtkWidget * widget)
{
  Game *game = glcwidget->game;
  char fname[50];

  snprintf(fname, 50, "%s/.glchessrc", getenv("HOME"));
  save_rc(game, fname);
}

/*
 * Toggles use of lighting 
 */
void
toggle_lighting(glChessWidget * glcwidget, guint num, GtkWidget * widget)
{
  Game *game = glcwidget->game;

  game->is_light = !game->is_light;
  if (game->is_light)
    glEnable(GL_LIGHTING);
  else
    glDisable(GL_LIGHTING);
  post_redisplay(game);
}

/*
 * Toggles shaded polygons 
 */
void
toggle_shading(glChessWidget * glcwidget, guint num, GtkWidget * widget)
{
  Game *game = glcwidget->game;

  game->is_smooth = !game->is_smooth;
  if (game->is_smooth)
    glShadeModel(GL_SMOOTH);
  else
    glShadeModel(GL_FLAT);
  post_redisplay(game);
}

/*
 * Toggles reflective board 
 */
void
toggle_reflect(glChessWidget * glcwidget, guint num, GtkWidget * widget)
{
  Game *game = glcwidget->game;

  game->is_reflect = !game->is_reflect;
  post_redisplay(game);
}

/*
 * Toggles use of textures 
 */
void
toggle_texture(glChessWidget * glcwidget, guint num, GtkWidget * widget)
{
  Game *game = glcwidget->game;

  game->is_texture = !game->is_texture;
  if (game->is_texture)
    glEnable(GL_TEXTURE_2D);
  else
    glDisable(GL_TEXTURE_2D);
  post_redisplay(game);
}

/*
 * Toggles the use of fog  
 */
void toggle_fog(glChessWidget * glcwidget, guint num, GtkWidget * widget)
{
  Game *game = glcwidget->game;

  game->is_fog = !game->is_fog;
  if (game->is_fog)
    glEnable(GL_FOG);
  else
    glDisable(GL_FOG);
  post_redisplay(game);
}

/*
 * Toggles the use of co-ordinates 
 */
void
toggle_coords(glChessWidget * glcwidget, guint num, GtkWidget * widget)
{
  Game *game = glcwidget->game;

  game->is_coord = !game->is_coord;
  post_redisplay(game);
}

/* Toggles beep on illegal move */
void toggle_beep(glChessWidget * glcwidget, guint num, GtkWidget * widget)
{
  Game *game = glcwidget->game;

  game->cecp.beep_signal = !game->cecp.beep_signal;
}

/* Toggles debuging for engines */
void toggle_debug(glChessWidget * glcwidget, guint num, GtkWidget * widget)
{
  debug = !debug;
}

/* Sets the view mode */
void
set_view_mode(glChessWidget * glcwidget, guint mode, GtkWidget * widget)
{
  Game *game = glcwidget->game;

  game->camera->mode = mode;
  reset_perspective(game);

  post_redisplay(game);
}

/*
 * Sets what the minimum dt is, so not all availiable cpu is used 
 */
void set_max_fps(glChessWidget * glcwidget, guint fps, GtkWidget * widget)
{
  if (fps == 666)
    glcwidget->game->min_dt = 0;
  else
    glcwidget->game->min_dt = 1.0 / (float) fps;
}

/* The help about screen */
void help_about(glChessWidget * glcwidget, guint num, GtkWidget * widget)
{
  static GtkWidget *dialog = NULL;
  GtkWidget *hbox, *label, *ok_button;

  /* Quit if already open */
  if (dialog != NULL)
    return;

  dialog = gtk_dialog_new();
  gtk_signal_connect(GTK_OBJECT(dialog),
		     "unrealize", GTK_SIGNAL_FUNC(exit_dialog), &dialog);
  hbox = gtk_hbox_new(FALSE, 10);

  label =
      gtk_label_new("glChess " GLCHESS_VERSION_STRING "\n" "\n"
		    "A 3D chess interface\n" "\n" "Authors:\n"
		    "Robert  Ancell <bob27@users.sourceforge.net>\n"
		    "Michael Duelli <duelli@users.sourceforge.net>\n"
		    "\n" "Homepage:\n" "glchess.sourceforge.net");
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, TRUE, TRUE,
		     10);
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 10);

  ok_button = gtk_button_new_with_label("OK");
  gtk_signal_connect_object(GTK_OBJECT(ok_button),
			    "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(dialog));
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		     ok_button, TRUE, TRUE, 0);

  gtk_widget_show_all(dialog);
}

/* Set a dialog to NULL when quit */
void exit_dialog(GtkWidget * widget, GtkWidget ** dialog)
{
  *dialog = NULL;
}
