/*
 * glChess - A 3D chess interface
 *
 * Copyright (C) 2001  Robert Ancell <bob27@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * This file contains (almost) all the GTK calls 
 */

#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkgl/gtkglarea.h>

#include "global.h"
#include "interface.h"
#include "menu.h"
#include "draw.h"
#include "picksquare.h"
#include "animation.h"
#include "splash.h"
#include "game.h"

#include "engine.h"

/*
 * Initialize the gtk interface 
 */
glChessWidget *init_interface(int argc, char **argv, Game * game)
{
  gtk_init(&argc, &argv);

  /*
   * Check if OpenGL is supported. 
   */
  if (gdk_gl_query() == FALSE)
  {
    g_print("OpenGL not supported\n");
    return NULL;
  }

  /*
   * Create the main window 
   */
  return create_window(game);
}

/*
 * FIXME: i'm assuming that 0 is not a valid tag 
 */
static guint idle_tag = 0;
static gint button_signal = -1, motion_signal = -1, key_signal = -1;
static GTimer *timer = NULL;

/*
 * Set up the view/widgets etc for a game 
 */
void init_game(glChessWidget * glcwidget)
{
  Game *game = glcwidget->game;
  int engine_start[2], engine_target[2];

  /*
   * Reset the game 
   */
  reset_board(game);
  game->current_player = game->white_player;
  game->selected[0] = -1;

  timer = g_timer_new();

  gtk_gl_area_make_current(GTK_GL_AREA(glcwidget->glarea));
  /*
   * Add the animation function 
   */
  idle_tag = gtk_idle_add(interface_idle, (gpointer) glcwidget);
  /*
   * If was paused before, unpause 
   */
  glcwidget->game->is_paused = FALSE;

  /*
   * FPS meter not used in main game 
   */
  gtk_label_set_text(GTK_LABEL(glcwidget->fps_label), "");

/* FIXME: Add engine selection here */
  init_cecp_interface(game);
  /*
   * If the ai is white, let them go now 
   */
  if (game->cecp.ai_player == WHITE PLAYER)
  {
    write_to_engine("go");
    /*
     * Get the engines move 
     */
    parse_move_from_engine(game, read_from_engine(), engine_start,
			   engine_target);
    /*
     * Tell glChess 
     */
    game->selected[0] = engine_start[0];
    game->selected[1] = engine_start[1];
    move_piece(game, engine_target);
  }

  /*
   * Display the current player 
   */
  gtk_label_set_text(GTK_LABEL(glcwidget->player_label),
		     glcwidget->game->current_player->name);

  /*
   * Reconnect some input signals 
   */
  motion_signal = gtk_signal_connect(GTK_OBJECT(glcwidget->glarea),
				     "motion_notify_event",
				     GTK_SIGNAL_FUNC
				     (interface_motion_view), glcwidget);
  button_signal =
      gtk_signal_connect(GTK_OBJECT(glcwidget->glarea),
			 "button_press_event",
			 GTK_SIGNAL_FUNC(interface_click_view), glcwidget);
  key_signal =
      gtk_signal_connect(GTK_OBJECT(glcwidget->window),
			 "key_press_event",
			 GTK_SIGNAL_FUNC(interface_key_press_view),
			 glcwidget);

  g_timer_start(timer);

  post_redisplay(glcwidget->game);
}

/*
 * Clean up after a game 
 */
void end_game(glChessWidget * glcwidget)
{
  /*
   * Quit if game hasn't been started 
   */
  if (motion_signal == -1)
    return;

  /*
   * Stop the clock 
   */
  g_timer_destroy(timer);
  /*
   * Remove the animation function 
   */
  gtk_idle_remove(idle_tag);

  /*
   * Disconnect some input signals 
   */
  gtk_signal_disconnect(GTK_OBJECT(glcwidget->glarea), motion_signal);
  motion_signal = -1;
  gtk_signal_disconnect(GTK_OBJECT(glcwidget->glarea), button_signal);
  gtk_signal_disconnect(GTK_OBJECT(glcwidget->window), key_signal);

  /* Tell Engine/CECP */
  close_cecp_interface();

  /*
   * Nothing to display so back to the splash screen 
   */
  init_splash(glcwidget);
}

/*
 * Called when nothing else's happening 
 */
gint interface_idle(gpointer in_glcwidget)
{
  glChessWidget *glcwidget = in_glcwidget;
  Game *game = glcwidget->game;

  gdouble dt;
  static int nframes = 0;
  static char time_string[10];
  static double update_dt = 0.0;
  float time;
  int nhours, nmins;

  dt = g_timer_elapsed(timer, NULL);
  g_timer_reset(timer);

  /*
   * Calculate animation 
   */
  if (!game->is_paused)
    animate(game, (double) dt);

  /*
   * Update the statusbar 
   */
  update_dt += dt;
  if (update_dt > 0.1)
  {
    time = game->current_player->time;
    nhours = (int) (time / 3600.0);
    nmins = (int) (time / 60.0);
    snprintf(time_string, 10, "%i:%i:%.1f",
	     nhours, nmins, time - 60.0 * nmins - 3600.0 * nhours);
    gtk_label_set_text(GTK_LABEL(glcwidget->time_label), time_string);

    update_dt = 0.0;
  }

  /*
   * openGL functions can only be called if make_current) returns TRUE 
   */
  if (game->redisplay
      && gtk_gl_area_make_current(GTK_GL_AREA(glcwidget->glarea)))
  {
    /*
     * Draw the scene 
     */
    display(game);
    nframes++;

    /*
     * Swap the buffers 
     */
    gtk_gl_area_swapbuffers(GTK_GL_AREA(glcwidget->glarea));
    /*
     * Don't redisplay again 
     */
    game->redisplay = FALSE;
  }

  return TRUE;
}

/*
 * Links to the functions that do the following 
 */
gint
interface_expose_view(GtkWidget * widget,
		      GdkEventExpose * event, glChessWidget * glcwidget)
{
  /*
   * Draw only last expose 
   */
  if (event->count < 0)
    return TRUE;

  /*
   * Set the scene to be redrawn 
   */
  post_redisplay(glcwidget->game);

  return TRUE;
}

gint
interface_reshape_view(GtkWidget * widget,
		       GdkEventConfigure * event,
		       glChessWidget * glcwidget)
{
  return interface_init_view(widget, glcwidget);
}

gint interface_init_view(GtkWidget * widget, glChessWidget * glcwidget)
{
  /*
   * openGL functions can only be called if make_current) returns TRUE 
   */
  if (gtk_gl_area_make_current(GTK_GL_AREA(widget)))
  {
    reshape(glcwidget->game,
	    widget->allocation.width, widget->allocation.height);
    return TRUE;
  } else
    return FALSE;
}

gint
interface_key_press_view(GtkWidget * widget,
			 GdkEventKey * event, glChessWidget * glcwidget)
{
  Game *game = glcwidget->game;

  if (game->camera->mode != MODE_FREE)
    return FALSE;

  switch (event->keyval)
  {
  case 'w':
  case 'W':
    game->camera->free_pos[2] += -5.0;
    break;
  case 's':
  case 'S':
    game->camera->free_pos[2] += 5.0;
    break;
  case 'a':
  case 'A':
    game->camera->free_pos[0] += -5.0;
    break;
  case 'd':
  case 'D':
    game->camera->free_pos[0] += 5.0;
    break;
  case 't':
  case 'T':
    game->camera->free_pos[1] += 5.0;
    break;
  case 'g':
  case 'G':
    game->camera->free_pos[1] += -5.0;
    break;
  case 'q':
  case 'Q':
    if (game->camera->free_rot[0] >= 360)
      game->camera->free_rot[0] = 5.0;
    else
      game->camera->free_rot[0] += 5.0;
    break;
  case 'e':
  case 'E':
    if (game->camera->free_rot[0] <= 0)
      game->camera->free_rot[0] = 355.0;
    else
      game->camera->free_rot[0] += -5.0;
    break;
  case 'r':
  case 'R':
    if (game->camera->free_rot[1] >= 360)
      game->camera->free_rot[1] = 5.0;
    else
      game->camera->free_rot[1] += 5.0;
    break;
  case 'f':
  case 'F':
    if (game->camera->free_rot[1] <= 0)
      game->camera->free_rot[1] = 355.0;
    else
      game->camera->free_rot[1] += -5.0;
    break;
  case 'z':
  case 'Z':
    if (game->camera->free_rot[2] <= 0)
      game->camera->free_rot[2] = 355.0;
    else
      game->camera->free_rot[2] += -5.0;
    break;
  case 'x':
  case 'X':
    if (game->camera->free_rot[2] >= 360)
      game->camera->free_rot[2] = 5.0;
    else
      game->camera->free_rot[2] += 5.0;
    break;
  }

  /*
   * Update the view 
   */
  post_redisplay(game);

  return TRUE;
}

gint
interface_motion_view(GtkWidget * widget,
		      GdkEventMotion * event, glChessWidget * glcwidget)
{
  int button;

  /*
   * FIXME: there must be a better way 
   */
  if (event->state & 256)
    button = 1;
  else if (event->state & 1024)
    button = 2;
  else
    button = 0;

  if (button)
    mouse_move(glcwidget->game, button, event->x, event->y);

  return TRUE;
}

gint
interface_click_view(GtkWidget * widget,
		     GdkEventButton * event, glChessWidget * glcwidget)
{
  mouse(glcwidget->game, event->button, event->x, event->y);

  /*
   * Update the name on the stausbar 
   */
  gtk_label_set_text(GTK_LABEL(glcwidget->player_label),
		     glcwidget->game->current_player->name);

  return TRUE;
}

glChessWidget *create_window(Game * game)
{
  int i;
  glChessWidget *glcwidget = malloc(sizeof(glChessWidget));

  /*
   * For the glarea 
   */
  int attrlist[] = {
    GDK_GL_RGBA,
    GDK_GL_RED_SIZE, 1,
    GDK_GL_GREEN_SIZE, 1,
    GDK_GL_BLUE_SIZE, 1,
    GDK_GL_DOUBLEBUFFER,
    GDK_GL_DEPTH_SIZE, 1,
    GDK_GL_STENCIL_SIZE, 1,
    GDK_GL_NONE
  };

  glcwidget->game = game;
  glcwidget->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(glcwidget->window),
		       "glChess " GLCHESS_VERSION_STRING);
  gtk_signal_connect(GTK_OBJECT(glcwidget->window),
		     "delete_event", GTK_SIGNAL_FUNC(quit_program), NULL);

  /*
   * Make sure GLX context is destroyed on quit 
   */
  gtk_quit_add_destroy(1, GTK_OBJECT(glcwidget->window));

  glcwidget->vbox = gtk_vbox_new(FALSE, 0);

  /*
   * Make the menubar/menus 
   */
  glcwidget->radio_list = NULL;
  for (i = 0; i < 4; i++)
    glcwidget->view_radio[i] = gtk_radio_button_new(glcwidget->radio_list);
  glcwidget->menu_handle_box = gtk_handle_box_new();
  glcwidget->menubar = generate_menus(glcwidget,
				      glcwidget->view_radio,
				      glcwidget->fps_radio);

  /*
   * The openGL area 
   */
  glcwidget->glarea = gtk_gl_area_new(attrlist);
  gtk_widget_set_usize(GTK_WIDGET(glcwidget->glarea), 400, 400);
  /*
   * Signals 
   */
  gtk_widget_set_events(glcwidget->glarea,
			GDK_EXPOSURE_MASK |
			GDK_BUTTON_PRESS_MASK | GDK_POINTER_MOTION_MASK);
  gtk_signal_connect(GTK_OBJECT(glcwidget->glarea),
		     "expose_event",
		     GTK_SIGNAL_FUNC(interface_expose_view), glcwidget);
  gtk_signal_connect(GTK_OBJECT(glcwidget->glarea),
		     "configure_event",
		     GTK_SIGNAL_FUNC(interface_reshape_view), glcwidget);
  gtk_signal_connect(GTK_OBJECT(glcwidget->glarea),
		     "realize",
		     GTK_SIGNAL_FUNC(interface_init_view), glcwidget);

  /*
   * The status bar 
   */
  glcwidget->status_handle_box = gtk_handle_box_new();
  glcwidget->statusbar = gtk_hbox_new(FALSE, 2);

  glcwidget->player_label = gtk_label_new("");
  if (0)
  {				/*
				 * for(i = 0; i < 16; i++) 
				 */
    char fname[50];
    GdkBitmap *mask;
    GtkStyle *style;

    if (i < 6)
      snprintf(fname, 50, "icons/piece%i.xpm", i);
    else
      snprintf(fname, 50, "icons/piece6.xpm");
    style = gtk_widget_get_style(glcwidget->statusbar);
    glcwidget->piece_gdk_pixmaps[i] =
	gdk_pixmap_create_from_xpm(glcwidget->window->window, &mask,
				   &style->bg[GTK_STATE_NORMAL], fname);
    if (glcwidget->piece_gdk_pixmaps[i] == NULL)
      fprintf(stderr, "Unable to load file %s\n", fname);
    glcwidget->piece_pixmaps[i] =
	gtk_pixmap_new(glcwidget->piece_gdk_pixmaps[i], mask);
  }
  glcwidget->time_label = gtk_label_new("");
  glcwidget->fps_label = gtk_label_new("0 fps");

  /*
   * Pack it all together 
   */
  gtk_container_add(GTK_CONTAINER(glcwidget->window), glcwidget->vbox);
  gtk_box_pack_start(GTK_BOX(glcwidget->vbox),
		     glcwidget->menu_handle_box, FALSE, FALSE, 0);
  gtk_container_add(GTK_CONTAINER(glcwidget->menu_handle_box),
		    glcwidget->menubar);
  gtk_box_pack_start(GTK_BOX(glcwidget->vbox), glcwidget->glarea, TRUE,
		     TRUE, 0);
  gtk_box_pack_start(GTK_BOX(glcwidget->vbox),
		     glcwidget->status_handle_box, FALSE, FALSE, 0);
  gtk_container_add(GTK_CONTAINER(glcwidget->status_handle_box),
		    glcwidget->statusbar);
  gtk_box_pack_start(GTK_BOX(glcwidget->statusbar),
		     glcwidget->player_label, FALSE, FALSE, 2);
  gtk_box_pack_end(GTK_BOX(glcwidget->statusbar), glcwidget->fps_label,
		   FALSE, FALSE, 2);
  gtk_box_pack_end(GTK_BOX(glcwidget->statusbar), glcwidget->time_label,
		   FALSE, FALSE, 2);

  gtk_widget_show_all(glcwidget->window);

  return glcwidget;
}
