/*
 * glChess - A 3D chess interface
 *
 * Copyright (C) 2001  Robert Ancell <bob27@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * This file contains all the game creation and modification routines 
 */

#include <stdio.h>
#include <stdlib.h>

#include "global.h"
#include "game.h"
#include "player.h"

/*
 * Make a new game 
 */
Game *new_game(void)
{
  Game *game = malloc(sizeof(Game));
  int i;

  if (game == NULL)
  {
    fprintf(stderr, "Unable to allocate space for new game\n");
    return NULL;
  }

  /*
   * The players 
   */
  game->white_player = new_player("White Player");
  game->black_player = new_player("Black Player");
  game->current_player = game->white_player;

  /*
   * The board 
   */
  reset_board(game);
  game->selected[0] = -1;
  game->selected[1] = -1;

  /*
   * The camera 
   */
  game->camera = new_camera();

  /*
   * The view 
   */
  game->redisplay = FALSE;
  game->min_dt = 0.0;

  /*
   * Graphics options 
   */
  game->is_light = TRUE;
  game->is_smooth = TRUE;
  game->is_reflect = FALSE;
  game->is_texture = TRUE;
  game->is_fog = FALSE;
  game->is_coord = TRUE;

  /*
   * Time values 
   */
  game->is_paused = FALSE;
  game->elapsed_time = 0.0;
  game->turn = FALSE;

  game->texture_dir = NULL;
  game->number_colour = malloc(sizeof(GLfloat) * 4);
  game->letter_colour = malloc(sizeof(GLfloat) * 4);
  for (i = 0; i < 4; i++)
  {
    game->number_colour[i] = 1.0;
    game->letter_colour[i] = 1.0;
  }

  /* Engine/CECP options */
  game->cecp.mode = CECP_MODE_EASY;
  game->cecp.search_depth = 6;
  game->cecp.ai_player = BLACK PLAYER;

  return game;
}

/*
 * Resets the board 
 */
void reset_board(Game * game)
{
  int x, y;

  for (x = 0; x < 8; x++)
    for (y = 0; y < 8; y++)
      game->board[x][y] = -1;

  /*
   * White pieces 
   */
  game->board[0][0] = WHITE ROOK;
  game->board[1][0] = WHITE KNIGHT;
  game->board[2][0] = WHITE BISHOP;
  game->board[3][0] = WHITE QUEEN;
  game->board[4][0] = WHITE KING;
  game->board[5][0] = WHITE BISHOP;
  game->board[6][0] = WHITE KNIGHT;
  game->board[7][0] = WHITE ROOK;
  game->board[0][1] = WHITE PAWN;
  game->board[1][1] = WHITE PAWN;
  game->board[2][1] = WHITE PAWN;
  game->board[3][1] = WHITE PAWN;
  game->board[4][1] = WHITE PAWN;
  game->board[5][1] = WHITE PAWN;
  game->board[6][1] = WHITE PAWN;
  game->board[7][1] = WHITE PAWN;

  /*
   * black pieces 
   */
  game->board[0][7] = BLACK ROOK;
  game->board[1][7] = BLACK KNIGHT;
  game->board[2][7] = BLACK BISHOP;
  game->board[3][7] = BLACK QUEEN;
  game->board[4][7] = BLACK KING;
  game->board[5][7] = BLACK BISHOP;
  game->board[6][7] = BLACK KNIGHT;
  game->board[7][7] = BLACK ROOK;
  game->board[0][6] = BLACK PAWN;
  game->board[1][6] = BLACK PAWN;
  game->board[2][6] = BLACK PAWN;
  game->board[3][6] = BLACK PAWN;
  game->board[4][6] = BLACK PAWN;
  game->board[5][6] = BLACK PAWN;
  game->board[6][6] = BLACK PAWN;
  game->board[7][6] = BLACK PAWN;
}

/*
 * Makes a new camera 
 */
Camera *new_camera(void)
{
  Camera *camera = malloc(sizeof(Camera));

  if (camera == NULL)
  {
    fprintf(stderr, "Unable to allocate space for new camera\n");
    return NULL;
  }

  /*
   * Default mode 
   */
  camera->mode = MODE_FREE;

  /*
   * For MODE_FREE 
   */
  camera->free_pos[0] = 0.0;
  camera->free_pos[1] = 60.0;
  camera->free_pos[2] = 85.0;
  camera->free_rot[0] = 0.0;
  camera->free_rot[1] = -40.0;
  camera->free_rot[2] = 0.0;

  /*
   * For MODE_ROTATING 
   */
  camera->rotating_rad = 35.0;
  camera->rotating_height = 90.0;
  camera->rotating_rot = 0.0;
  camera->rotating_pitch = 70.0;
  camera->rotating_t = 0.0;

  /*
   * For MODE_TRACKBALL 
   */
  camera->track_rot[0] = 0.0;
  camera->track_rot[1] = 40.0;
  camera->track_rot[2] = 0.0;
  camera->track_rad = 104.04326f;

  /*
   * For working out dragging on the screen 
   */
  /*
   * FIXME: these probably shouldn't be here 
   */
  camera->oldx = 0;
  camera->oldy = 0;
  camera->button = 0;

  return camera;
}

/*
 * Copy a game 
 */
Game *copy_game(Game * game)
{
  Game *game_copy = malloc(sizeof(Game));
  int i, j;

  if (game_copy == NULL)
  {
    fprintf(stderr, "Unable to allocate space for game copy\n");
    return NULL;
  }

  /*
   * Copy the players 
   */
  game_copy->white_player = copy_player(game->white_player);
  game_copy->black_player = copy_player(game->black_player);
  if (game->current_player == game->white_player)
    game_copy->current_player = game_copy->white_player;
  else
    game_copy->current_player = game_copy->black_player;

  /*
   * Copy the board 
   */
  for (i = 0; i < 8; i++)
    for (j = 0; j < 8; j++)
      game_copy->board[i][j] = game->board[i][j];
  game_copy->selected[0] = game->selected[0];
  game_copy->selected[1] = game->selected[1];

  /*
   * Copy the camera 
   */
  game_copy->camera = copy_camera(game->camera);

  /*
   * Copy the view 
   */
  game_copy->redisplay = game->redisplay;
  game_copy->min_dt = game->min_dt;

  /*
   * Copy the graphics options 
   */
  game_copy->is_light = game->is_light;
  game_copy->is_smooth = game->is_smooth;
  game_copy->is_reflect = game->is_reflect;
  game_copy->is_texture = game->is_texture;
  game_copy->is_fog = game->is_fog;
  game_copy->is_coord = game->is_coord;

  /*
   * Copy the time values 
   */
  game_copy->is_paused = game->is_paused;
  game_copy->elapsed_time = game->elapsed_time;
  game_copy->turn = game->turn;

  /*
   * Texture info 
   */
  game_copy->texture_dir = g_strdup(game->texture_dir);
  game_copy->number_colour = malloc(sizeof(GLfloat) * 4);
  game_copy->letter_colour = malloc(sizeof(GLfloat) * 4);
  for (i = 0; i < 4; i++)
  {
    game_copy->number_colour[i] = game->number_colour[i];
    game_copy->letter_colour[i] = game->letter_colour[i];
  }

  /* Engine/CECP options */
  game_copy->cecp.mode = game->cecp.mode;
  game_copy->cecp.search_depth = game->cecp.search_depth;
  game_copy->cecp.ai_player = game->cecp.ai_player;

  return game_copy;
}

/*
 * Copies an existing camera 
 */
Camera *copy_camera(Camera * camera)
{
  Camera *camera_copy = malloc(sizeof(Camera));
  int i;

  if (camera_copy == NULL)
  {
    fprintf(stderr, "Unable to allocate space for camera copy\n");
    return NULL;
  }

  /*
   * Copy default mode 
   */
  camera_copy->mode = camera->mode;

  /*
   * Copy MODE_FREE 
   */
  for (i = 0; i < 3; i++)
  {
    camera_copy->free_pos[i] = camera->free_pos[i];
    camera_copy->free_rot[i] = camera->free_rot[i];
  }

  /*
   * Copy MODE_ROTATING 
   */
  camera_copy->rotating_rad = camera->rotating_rad;
  camera_copy->rotating_height = camera->rotating_height;
  camera_copy->rotating_rot = camera->rotating_rot;
  camera_copy->rotating_pitch = camera->rotating_pitch;
  camera_copy->rotating_t = camera->rotating_t;

  /*
   * Copy MODE_TRACKBALL 
   */
  for (i = 0; i < 3; i++)
    camera_copy->track_rot[i] = camera->track_rot[i];
  camera_copy->track_rad = camera->track_rad;

  /*
   * Copy screen stats 
   */
  camera_copy->oldx = camera->oldx;
  camera_copy->oldy = camera->oldy;
  camera_copy->button = camera->button;

  return camera_copy;
}

/*
 * Remove a camera 
 */
void free_camera(Camera * camera)
{
  free(camera);
}

/*
 * Remove a game 
 */
void free_game(Game * game)
{
  free(game->white_player);
  free(game->black_player);
  free(game->camera);
}
