/*
 * glChess - A 3D chess interface
 *
 * Copyright (C) 2001  Robert  Ancell <bob27@users.sourceforge.net>
 *                     Michael Duelli <duelli@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <sys/poll.h>
#include <sys/types.h>
#include <signal.h>

#include "global.h"
#include "san.h"
#include "engine.h"

extern int engine;
extern int debug;
extern int promotion;

#define MAXBUF 256
#define TIME_OUT 150		/* time out in milliseconds */

int pid;
int input_pipe[2];
int output_pipe[2] = { -1, -1 };	/* -1 means disabled */

/* Simple short abbreviations everybody can remember */
#define INPUT   input_pipe[0]
#define OUTPUT output_pipe[1]

/* Prepares the pipes, starts the engine */
void init_communication(char *cmd)
{
  /* Create the pipes */
  if (pipe(input_pipe) || pipe(output_pipe))
  {
    fprintf(stderr, "An error occured while creating the pipes !\n");
    exit(-1);
  }

  signal(SIGPIPE, SIG_IGN);

  pid = fork();
  if (!pid)
  {
    dup2(input_pipe[1], STDOUT_FILENO);
    dup2(output_pipe[0], STDIN_FILENO);

    close(input_pipe[0]);
    close(output_pipe[1]);

    setpgid(getpid(), 0);
    execlp(cmd, cmd, NULL);
    /* This won't be reached until the engine won't start for some reasons */
    fprintf(stderr, "An error occured while starting the engine !\n");
    exit(-1);
  } else
  {
    if (engine == CRAFTY)
      clear_startup_msg();
    if (engine == GNUCHESS)
      clear_engine_msg();
  }
}

void clear_startup_msg(void)
{
  char msg[MAXBUF];

  while (strcmp(msg, "White(1)") != 0)
  {
    char *s = msg;

    char c;

    while (read_char(&c) > 0)
    {
      *s = c;
      if (*s == '\n' || *s == ':')
	break;
      ++s;
    }
    *s = '\0';

    if (debug)
      debug_output("READ_STARTUP_MSG: %s\n", msg);
  }
}

/* Read a char without reading for ever */
int read_char(char *ch)
{
  int ret;
  struct pollfd pfd;

  pfd.fd = INPUT;
  pfd.events = POLLIN | POLLERR | POLLHUP;

  /* If time runs out and there's still no char, we guess * there won't be
   * any more */
  if (poll(&pfd, 1, TIME_OUT) < 0)
  {
    return (-1);
  }

  if ((pfd.revents & (POLLERR | POLLHUP)) != 0)
    return (-1);

  if ((pfd.revents & POLLIN) != 0)
    ret = read(INPUT, ch, 1);
  else
    ret = 0;

  return ret;
}

/* Inits the CEC Protocol interface */
void init_cecp_interface(Game * game)
{
  char *cmd;
  char sd[10];

  if (cecp_interface_up() == TRUE)
  {
    engine_dialog("You have to quit the running engine\n"
		  "before starting a new one");
    return;
  }

  switch (engine)
  {
  case CRAFTY:
    cmd = strdup("crafty");
    break;
  case GNUCHESS:
    cmd = strdup("gnuchessx");
    break;
  default:			/* Otherwise play local Human vs. local Human */
    return;
    break;
  }


  init_communication(cmd);

  if (engine == CRAFTY)
  {

    /* We need this for CECP init ! */
    write_to_engine("xboard");
    clear_engine_msg();
  }

  if (engine == CRAFTY)
  {
    /* Set the mode */
    if (game->cecp.mode == CECP_MODE_EASY)
      write_to_engine("easy");
    else
      /* FIXME: This is not implemented at all until now */
      write_to_engine("hard");
    clear_engine_msg();
  }

  if (engine == CRAFTY)
  {
    /* Set the search depth */
    snprintf(sd, 10, "sd %d", game->cecp.search_depth);
    write_to_engine(sd);
  } else if (engine == GNUCHESS)
  {
    snprintf(sd, 10, "%d", game->cecp.search_depth);
    write_to_engine("depth");
    write_to_engine(sd);
  }
  clear_engine_msg();
}

/* Tells if the CECP interface is up */
int cecp_interface_up(void)
{
  if (OUTPUT == -1)
    return FALSE;
  else
    return TRUE;
}

/* Closes the CEC Protocol interface */
void close_cecp_interface(void)
{
  /* If already closed */
  if (OUTPUT == -1)
    return;

  /* Shut the engine down */
  write_to_engine("quit");
  /* Clear quit messages */
  clear_engine_msg();

  if (pid)
    kill(-pid, SIGKILL);
  pid = 0;

  /* Set the interface to be closed */
  OUTPUT = -1;
}

/* Engine might print some useless message, get rid of these */
void clear_engine_msg(void)
{
  char *s = malloc(sizeof(char) * MAXBUF);
  char *buf = s;
  char c;

  while (read_char(&c) > 0)
    *buf++ = c;

  if ((buf = strchr(s, '\n')) != NULL)
    *buf = '\0';

  if (debug)
    debug_output("CLEAR_ENGINE_MSG: \"%s\"\n", s);

  free(s);
}

/* Writes text to the engine */
void write_to_engine(char *str)
{
  char *buf = malloc(sizeof(char) * MAXBUF);

  sprintf(buf, "%s\n", str);
  write(OUTPUT, buf, strlen(buf));

  if (debug)
    debug_output("WRITE_TO_ENGINE: %s\n", buf);

  free(buf);
}

/* Reads text from the engine */
char *read_from_engine(void)
{
  char msg[MAXBUF];
  char *s = msg;
  char c;

  while (((engine == CRAFTY && strstr(msg, "move") == NULL)
	  || (engine == GNUCHESS && strstr(msg, "move is:") == NULL))
	 && strstr(msg, "Illegal") == NULL)
  {
    while (read_char(&c) > 0)
    {
      *s++ = c;

      if (c == '\n')
	break;
    }
  }
  *s = '\0';

  if (debug)
    debug_output("READ_FROM_ENGINE: %s\n", msg);

  return strdup(msg);
}

/* Convert glChess' internal to coordinated notation */
char *convert_notation(int *start, int *target)
{
  char *move = malloc(sizeof(char) * 6);	/* e.g. e2e4 or e7e8Q */

  snprintf(move, 6, "%c%d%c%d",
	   'a' + start[0], 1 + start[1], 'a' + target[0], 1 + target[1]);

  if (debug)
    debug_output("CONVERT_NOTATION: %s\n", move);

  return move;
}

/* Checks whether move is in coordinated or SAN notation */
int check_notation_format(char *move)
{
  if (isalpha(move[0]) && isdigit(move[1]) &&
      isalpha(move[2]) && isdigit(move[3]))
    return 1;			/* Coordinated */

  return 0;			/* SAN */
}

/* FIXME: Reorganize this stuff */
/* Parses the engine's move */
int parse_move_from_engine(Game * game, char *in, int *start, int *target)
{
  char *buf = malloc(sizeof(char) * MAXBUF);
  char *s = in;
  int format;

  if (strstr(in, "Illegal"))
  {
    engine_dialog("Your move was illegal. No move.");

    if (game->cecp.beep_signal)
      beep();

    if (debug)
      debug_output("PARSE_MOVE_FROM_ENGINE: %s", in);
    return 0;
  } else if (strstr(in, "learn"))	/* Ignore learning messages (Crafty only) */
  {
    if ((s = strstr(in, "move")) == NULL)	/* Did we already read the move */
    {
      /* The real move follows upon the learning message */
/* FIXME: Replace this by read_from_engine */
      read(INPUT, buf, MAXBUF);
      s = &buf[0];
    }
    if (debug)
      debug_output("PARSE_MOVE_FROM_ENGINE: %s", s);
  } else if (strstr(in, "mate") || strstr(in, "draw"))
  {
    engine_dialog("The game is over");

    /* The game is already over */
    if (debug)
      debug_output("RESULT: %s", in);
  }

  /* Otherwise, let's parse the move */

  if (engine == CRAFTY)
  {
    /* Get rid of the 'move ' prefix */
    s += 5;			/* e.g. move e2e4 */
  } else if (engine == GNUCHESS)
  {
    /* Get rid of the whole shebang GNU Chess creates, e.g. */
    s = strchr(s, '\n');	/* 1. e4 */
    s = strchr(s, '\n');	/* 1. ... g7g6 */
    s = strchr(s, ':') + 2;	/* My move is: g7g6 */
  }

  format = check_notation_format(s);

  if (debug)
    debug_output("CHECK_NOTATION_FORMAT: %s\n",
		 (format ? "Coordinated" : "SAN"));

  if (format == 0)		/* SAN */
    s = san_to_coordinated(s, game->board);
  else
    promotion = get_promotion_piece(s);

  start[0] = *s++ - 97;
  start[1] = *s++ - 49;
  target[0] = *s++ - 97;
  target[1] = *s - 49;

  if (debug)
    debug_output("PARSE_MOVE_FROM_ENGINE: %d,%d,%d,%d\n",
		 start[0], start[1], target[0], target[1]);

  return 1;			/* Everything alright */
}

/* Makes the speaker beep on illegal move */
void beep(void)
{
  putc(7, stderr);
}

/* Just for simplifying debug output */
void debug_output(char *format, ...)
{
  va_list ap;

  va_start(ap, format);
  vfprintf(stderr, format, ap);
  va_end(ap);
}

/* This shows a dialog with a messages */
void engine_dialog(char *msg)
{
  GtkWidget *dialog;
  GtkWidget *hbox, *label, *ok_button;

  dialog = gtk_dialog_new();

  hbox = gtk_hbox_new(FALSE, 10);

  /* Create the label */
  label = gtk_label_new(msg);

  /* Create the button */
  ok_button = gtk_button_new_with_label("OK");
  gtk_signal_connect_object(GTK_OBJECT(ok_button), "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(dialog));

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, TRUE, TRUE,
		     10);
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		     ok_button, TRUE, TRUE, 0);

  gtk_widget_show_all(dialog);
}

/* Extracts the promotion piece */
int get_promotion_piece(char *str)
{
  char *s;

  /* Promotion piece is before newline char */
  if ((s = strchr(str, '\n')) != NULL)
  {
    switch (tolower(*--s))	/* The promotion piece, e.g. e7e8q, e8=Q */
    {
    case 'q':
      return QUEEN;
      break;
    case 'r':
      return ROOK;
      break;
    case 'n':
      return KNIGHT;
      break;
    case 'b':
      return BISHOP;
      break;
    default:
      break;
    }
  }

  return -1;			/* No promotion */
}
